<?php
/**
 * A/B Testing Page Template
 *
 * @package FlxWoo\Admin
 * @since 2.5.0
 *
 * @var array $tests List of A/B tests
 */

if (!defined('ABSPATH')) {
    exit;
}

// Prepare nonce for AJAX requests
$nonce = wp_create_nonce('flx_ab_test_nonce');
?>

<div class="wrap flx-ab-testing-page">
    <h1><?php esc_html_e('A/B Testing', 'flx-woo'); ?></h1>

    <!-- Page Header -->
    <div class="flx-dashboard-header">
        <p class="description">
            <?php esc_html_e('Test different variations of your checkout to optimize conversion rates. Create tests, monitor results, and deploy winners with statistical confidence.', 'flx-woo'); ?>
        </p>
        <button class="button button-primary flx-create-test-btn" id="flx-create-test-btn">
            <?php esc_html_e('Create New Test', 'flx-woo'); ?>
        </button>
    </div>

    <!-- Test List -->
    <div class="flx-dashboard-section">
        <h2><?php esc_html_e('Active & Recent Tests', 'flx-woo'); ?></h2>

        <?php if (empty($tests)): ?>
            <div class="flx-empty-state">
                <p><?php esc_html_e('No tests created yet. Create your first A/B test to start optimizing!', 'flx-woo'); ?></p>
            </div>
        <?php else: ?>
            <table class="wp-list-table widefat fixed striped flx-test-table">
                <thead>
                    <tr>
                        <th><?php esc_html_e('Test Name', 'flx-woo'); ?></th>
                        <th><?php esc_html_e('Status', 'flx-woo'); ?></th>
                        <th><?php esc_html_e('Variants', 'flx-woo'); ?></th>
                        <th><?php esc_html_e('Created', 'flx-woo'); ?></th>
                        <th><?php esc_html_e('Actions', 'flx-woo'); ?></th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($tests as $test): ?>
                        <tr data-test-id="<?php echo esc_attr($test['id']); ?>">
                            <td>
                                <strong><?php echo esc_html($test['test_name']); ?></strong>
                                <div class="test-key"><?php echo esc_html($test['test_key']); ?></div>
                            </td>
                            <td>
                                <span class="flx-status-badge flx-status-<?php echo esc_attr($test['status']); ?>">
                                    <?php echo esc_html(ucfirst($test['status'])); ?>
                                </span>
                            </td>
                            <td><?php echo esc_html(count($test['variants'])); ?> variants</td>
                            <td><?php echo esc_html(date_i18n(get_option('date_format'), $test['created_at'])); ?></td>
                            <td class="test-actions">
                                <?php if ($test['status'] === 'draft'): ?>
                                    <button class="button button-small flx-start-test" data-test-id="<?php echo esc_attr($test['id']); ?>">
                                        <?php esc_html_e('Start', 'flx-woo'); ?>
                                    </button>
                                <?php elseif ($test['status'] === 'active'): ?>
                                    <button class="button button-small flx-view-results" data-test-id="<?php echo esc_attr($test['id']); ?>">
                                        <?php esc_html_e('View Results', 'flx-woo'); ?>
                                    </button>
                                    <button class="button button-small flx-stop-test" data-test-id="<?php echo esc_attr($test['id']); ?>">
                                        <?php esc_html_e('Stop', 'flx-woo'); ?>
                                    </button>
                                <?php else: ?>
                                    <button class="button button-small flx-view-results" data-test-id="<?php echo esc_attr($test['id']); ?>">
                                        <?php esc_html_e('View Results', 'flx-woo'); ?>
                                    </button>
                                <?php endif; ?>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        <?php endif; ?>
    </div>
</div>

<!-- Create Test Modal -->
<div id="flx-create-test-modal" class="flx-modal" style="display: none;">
    <div class="flx-modal-content">
        <div class="flx-modal-header">
            <h2><?php esc_html_e('Create New A/B Test', 'flx-woo'); ?></h2>
            <span class="flx-modal-close">&times;</span>
        </div>
        <div class="flx-modal-body">
            <form id="flx-create-test-form">
                <div class="flx-form-field">
                    <label for="test_name"><?php esc_html_e('Test Name', 'flx-woo'); ?></label>
                    <input type="text" id="test_name" name="test_name" required
                           placeholder="<?php esc_attr_e('e.g., Green vs Blue Checkout Button', 'flx-woo'); ?>">
                </div>

                <div class="flx-form-field">
                    <label for="test_key"><?php esc_html_e('Test Key (slug)', 'flx-woo'); ?></label>
                    <input type="text" id="test_key" name="test_key" required
                           placeholder="<?php esc_attr_e('e.g., checkout-button-color', 'flx-woo'); ?>">
                    <p class="description"><?php esc_html_e('Unique identifier for this test (lowercase, no spaces)', 'flx-woo'); ?></p>
                </div>

                <div class="flx-form-field">
                    <label><?php esc_html_e('Variants', 'flx-woo'); ?></label>
                    <div id="flx-variants-container">
                        <!-- Variant 1 (Control) -->
                        <div class="flx-variant-row" data-variant-index="0">
                            <input type="text" class="variant-id" value="control" readonly>
                            <input type="text" class="variant-name" placeholder="<?php esc_attr_e('Control Name', 'flx-woo'); ?>" value="Control" required>
                            <input type="number" class="variant-traffic" min="0" max="100" value="50" required>
                            <span>%</span>
                        </div>
                        <!-- Variant 2 -->
                        <div class="flx-variant-row" data-variant-index="1">
                            <input type="text" class="variant-id" placeholder="<?php esc_attr_e('variant_a', 'flx-woo'); ?>" value="variant_a" required>
                            <input type="text" class="variant-name" placeholder="<?php esc_attr_e('Variant Name', 'flx-woo'); ?>" value="Variant A" required>
                            <input type="number" class="variant-traffic" min="0" max="100" value="50" required>
                            <span>%</span>
                            <button type="button" class="button flx-remove-variant">×</button>
                        </div>
                    </div>
                    <button type="button" class="button" id="flx-add-variant">
                        <?php esc_html_e('Add Variant', 'flx-woo'); ?>
                    </button>
                    <p class="description"><?php esc_html_e('Traffic split must sum to 100%', 'flx-woo'); ?></p>
                </div>

                <div class="flx-modal-footer">
                    <button type="submit" class="button button-primary">
                        <?php esc_html_e('Create Test', 'flx-woo'); ?>
                    </button>
                    <button type="button" class="button flx-modal-cancel">
                        <?php esc_html_e('Cancel', 'flx-woo'); ?>
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Results Modal -->
<div id="flx-results-modal" class="flx-modal" style="display: none;">
    <div class="flx-modal-content flx-modal-large">
        <div class="flx-modal-header">
            <h2 id="results-test-name"><?php esc_html_e('Test Results', 'flx-woo'); ?></h2>
            <span class="flx-modal-close">&times;</span>
        </div>
        <div class="flx-modal-body">
            <div id="flx-results-container">
                <p class="flx-loading"><?php esc_html_e('Loading results...', 'flx-woo'); ?></p>
            </div>
        </div>
    </div>
</div>

<script>
// Pass data to JavaScript
var flxABTestingData = {
    nonce: '<?php echo esc_js($nonce); ?>',
    ajaxUrl: '<?php echo esc_js(admin_url('admin-ajax.php')); ?>',
    strings: {
        confirmStart: '<?php echo esc_js(__('Start this test? Traffic will be split according to your configuration.', 'flx-woo')); ?>',
        confirmStop: '<?php echo esc_js(__('Stop this test? You can view final results afterwards.', 'flx-woo')); ?>',
        createSuccess: '<?php echo esc_js(__('Test created successfully!', 'flx-woo')); ?>',
        createError: '<?php echo esc_js(__('Failed to create test. Please try again.', 'flx-woo')); ?>',
        trafficSplitError: '<?php echo esc_js(__('Traffic split must sum to exactly 100%', 'flx-woo')); ?>',
    }
};
</script>
