/**
 * FlxWoo Settings Dashboard JavaScript
 *
 * Handles settings form submission and reset functionality.
 *
 * @package FlxWoo\Admin
 * @since 2.3.0 (Simplified from v2.2.0)
 */

(function($) {
    'use strict';

    /**
     * Initialize dashboard functionality
     */
    $(document).ready(function() {
        // Get nonces from localized data
        const nonces = flxDashboard?.nonces || {};

        /**
         * Handle settings form submission
         */
        $('#flx-settings-form').on('submit', function(e) {
            e.preventDefault();

            const $form = $(this);
            const $button = $('#flx-save-settings');
            const $message = $('#flx-settings-message');
            const originalText = $button.text();

            // Disable button and show loading state
            $button.prop('disabled', true).text('Saving...');
            $message.hide();

            // Get form data
            const formData = {
                action: 'flx_save_settings',
                nonce: nonces.settings || '',
                fallback_enabled: $('#fallback_enabled').is(':checked') ? '1' : '',
                active_pages: $('input[name="active_pages[]"]:checked').map(function() {
                    return $(this).val();
                }).get(),
                dev_mode: $('#dev_mode').is(':checked') ? '1' : ''
            };

            // Submit via AJAX
            $.post(flxDashboard.ajaxurl, formData)
                .done(function(response) {
                    if (response.success) {
                        $message
                            .removeClass('notice-error')
                            .addClass('notice-success')
                            .find('p').text(response.data.message);
                    } else {
                        $message
                            .removeClass('notice-success')
                            .addClass('notice-error')
                            .find('p').text(response.data.message || 'Failed to save settings.');
                    }
                    $message.show();
                })
                .fail(function() {
                    $message
                        .removeClass('notice-success')
                        .addClass('notice-error')
                        .find('p').text('AJAX request failed. Please try again.');
                    $message.show();
                })
                .always(function() {
                    $button.prop('disabled', false).text(originalText);
                });
        });

        /**
         * Handle reset to defaults button
         */
        $('#flx-reset-settings').on('click', function(e) {
            e.preventDefault();

            if (!confirm('Are you sure you want to reset all settings to defaults? This cannot be undone.')) {
                return;
            }

            const $button = $(this);
            const $message = $('#flx-settings-message');
            const originalText = $button.text();

            // Disable button and show loading state
            $button.prop('disabled', true).text('Resetting...');
            $message.hide();

            // Submit via AJAX
            $.post(flxDashboard.ajaxurl, {
                action: 'flx_reset_settings',
                nonce: nonces.dashboard || ''
            })
                .done(function(response) {
                    if (response.success) {
                        $message
                            .removeClass('notice-error')
                            .addClass('notice-success')
                            .find('p').text(response.data.message);
                        $message.show();

                        // Reload page after 1 second to show new defaults
                        setTimeout(function() {
                            location.reload();
                        }, 1000);
                    } else {
                        $message
                            .removeClass('notice-success')
                            .addClass('notice-error')
                            .find('p').text(response.data.message || 'Failed to reset settings.');
                        $message.show();
                        $button.prop('disabled', false).text(originalText);
                    }
                })
                .fail(function() {
                    $message
                        .removeClass('notice-success')
                        .addClass('notice-error')
                        .find('p').text('AJAX request failed. Please try again.');
                    $message.show();
                    $button.prop('disabled', false).text(originalText);
                });
        });
    });

})(jQuery);
