/**
 * FlxWoo Feature Flags JavaScript
 *
 * Enhances feature flags page with interactive functionality.
 *
 * @package FlxWoo\Admin
 * @since 2.4.0
 */

(function($) {
	'use strict';

	/**
	 * Initialize feature flags enhancements
	 */
	function initFeatureFlags() {
		initKillSwitchConfirmation();
		initRolloutSliders();
		initFeatureCardInteractions();
		initFormValidation();
		initCopyStoreId();
		initManualCleanup();
	}

	/**
	 * Kill switch confirmation dialog
	 */
	function initKillSwitchConfirmation() {
		$('#kill_switch').on('change', function() {
			const $checkbox = $(this);
			const isChecked = $checkbox.is(':checked');

			if (isChecked) {
				const confirmed = confirm(
					'Are you sure you want to activate the global kill switch?\n\n' +
					'This will immediately disable ALL v2.3.0 features including:\n' +
					'• Analytics\n' +
					'• Benchmarking\n' +
					'• A/B Testing\n' +
					'• Compatibility\n\n' +
					'This action should only be used in emergency situations.'
				);

				if (!confirmed) {
					$checkbox.prop('checked', false);
				}
			}
		});
	}

	/**
	 * Rollout slider real-time updates
	 */
	function initRolloutSliders() {
		$('.flx-rollout-slider input[type="range"]').each(function() {
			const $slider = $(this);
			const $valueDisplay = $slider.siblings('.flx-rollout-slider-header').find('.flx-rollout-value');

			// Update display on slider change
			$slider.on('input', function() {
				const value = $(this).val();
				$valueDisplay.text(value + '%');

				// Update progress bar color based on value
				updateSliderColor($slider, value);
			});

			// Initialize color
			const initialValue = $slider.val();
			updateSliderColor($slider, initialValue);
		});
	}

	/**
	 * Update slider track color based on value
	 *
	 * @param {jQuery} $slider Slider element
	 * @param {number} value Current value
	 */
	function updateSliderColor($slider, value) {
		const percentage = value;
		const gradient = `linear-gradient(90deg, #2271b1 0%, #2271b1 ${percentage}%, #dcdcde ${percentage}%, #dcdcde 100%)`;
		$slider.css('background', gradient);
	}

	/**
	 * Feature card interactions
	 */
	function initFeatureCardInteractions() {
		// Toggle rollout section visibility based on global enable checkbox
		$('.flx-feature-card').each(function() {
			const $card = $(this);
			const $enableCheckbox = $card.find('input[type="checkbox"][name*="_enabled"]');
			const $rolloutSection = $card.find('.flx-rollout-slider');

			$enableCheckbox.on('change', function() {
				const isEnabled = $(this).is(':checked');

				// If globally enabled, hide rollout controls
				if (isEnabled) {
					$rolloutSection.slideUp(200);
				} else {
					$rolloutSection.slideDown(200);
				}
			});

			// Initialize visibility
			if ($enableCheckbox.is(':checked')) {
				$rolloutSection.hide();
			}
		});
	}

	/**
	 * Form submission validation
	 */
	function initFormValidation() {
		$('form[method="post"]').on('submit', function(e) {
			const $form = $(this);

			// Only validate if this is a feature flags form (not kill switch)
			if ($form.find('.flx-feature-cards').length === 0) {
				return true;
			}

			const enabledFeatures = [];

			// Collect all enabled features
			$form.find('input[type="checkbox"][name*="_enabled"]:checked').each(function() {
				const name = $(this).attr('name').replace('flag_', '').replace('_enabled', '');
				enabledFeatures.push(name);
			});

			// Validate dependencies (if dependency data is available)
			if (typeof flxFeatureDependencies !== 'undefined') {
				let validationErrors = [];

				enabledFeatures.forEach(function(feature) {
					const dependencies = flxFeatureDependencies[feature] || [];
					dependencies.forEach(function(dep) {
						if (!enabledFeatures.includes(dep)) {
							validationErrors.push(
								`Cannot enable ${feature}: Missing dependency ${dep}`
							);
						}
					});
				});

				if (validationErrors.length > 0) {
					e.preventDefault();
					alert('Validation Errors:\n\n' + validationErrors.join('\n'));
					return false;
				}
			}

			return true;
		});
	}

	/**
	 * Store ID copy-to-clipboard functionality
	 */
	function initCopyStoreId() {
		$('.flx-copy-store-id').on('click', function(e) {
			e.preventDefault();
			const $button = $(this);
			const $codeDisplay = $button.siblings('.flx-store-id-display');
			const fullStoreId = $codeDisplay.data('full-id');

			// Copy to clipboard
			if (navigator.clipboard && navigator.clipboard.writeText) {
				// Modern API
				navigator.clipboard.writeText(fullStoreId).then(function() {
					showCopySuccess($button);
				}).catch(function() {
					// Fallback to textarea method
					fallbackCopyToClipboard(fullStoreId, $button);
				});
			} else {
				// Fallback for older browsers
				fallbackCopyToClipboard(fullStoreId, $button);
			}
		});
	}

	/**
	 * Show copy success feedback
	 *
	 * @param {jQuery} $button Button element
	 */
	function showCopySuccess($button) {
		// Add success class and change text
		$button.addClass('copied').text('Copied!');

		// Reset after 2 seconds
		setTimeout(function() {
			$button.removeClass('copied').text('Copy');
		}, 2000);
	}

	/**
	 * Fallback copy method for older browsers
	 *
	 * @param {string} text Text to copy
	 * @param {jQuery} $button Button element
	 */
	function fallbackCopyToClipboard(text, $button) {
		const $textarea = $('<textarea>')
			.val(text)
			.css({
				position: 'fixed',
				top: 0,
				left: 0,
				width: '2em',
				height: '2em',
				padding: 0,
				border: 'none',
				outline: 'none',
				boxShadow: 'none',
				background: 'transparent'
			})
			.appendTo('body');

		$textarea[0].select();

		try {
			const successful = document.execCommand('copy');
			if (successful) {
				showCopySuccess($button);
			} else {
				alert('Failed to copy Store ID. Please copy manually: ' + text);
			}
		} catch (err) {
			alert('Failed to copy Store ID. Please copy manually: ' + text);
		}

		$textarea.remove();
	}

	/**
	 * Manual cleanup confirmation and AJAX handler
	 */
	function initManualCleanup() {
		const $button = $('#flx-manual-cleanup');

		// Check if button exists
		if ($button.length === 0) {
			return;
		}

		$button.on('click', function(e) {
			e.preventDefault();
			const $btn = $(this);

			// Confirmation dialog
			const confirmed = confirm(
				'Are you sure you want to run manual cleanup now?\n\n' +
				'This will permanently delete old activity log records based on your retention period.\n\n' +
				'This action cannot be undone.'
			);

			if (!confirmed) {
				return;
			}

			// Disable button and show loading state
			$btn.prop('disabled', true).text('Running cleanup...');

			// Get AJAX URL and nonce from WordPress
			const ajaxUrl = typeof flxAjax !== 'undefined' ? flxAjax.ajaxurl : (typeof ajaxurl !== 'undefined' ? ajaxurl : '/wp-admin/admin-ajax.php');
			const nonce = typeof flxAjax !== 'undefined' ? flxAjax.nonce : '';

			// Make AJAX request
			$.ajax({
				url: ajaxUrl,
				type: 'POST',
				data: {
					action: 'flx_manual_cleanup',
					nonce: nonce
				},
				success: function(response) {
					if (response.success) {
						const data = response.data;
						let message = data.message;

						// Add details if available
						if (data.deleted_count > 0) {
							message += '\n\n' +
								'Deleted: ' + data.deleted_count + ' records\n' +
								'Retention period: ' + data.retention_days + ' days';
						}

						alert(message);

						// Reload page to show updated statistics
						location.reload();
					} else {
						alert('Error: ' + (response.data.message || 'Unknown error occurred'));
						$btn.prop('disabled', false).text('Run Manual Cleanup Now');
					}
				},
				error: function(xhr, status, error) {
					console.error('FlxWoo Manual Cleanup Error:', {status: status, error: error});
					alert('Failed to run cleanup. Please try again.');
					$btn.prop('disabled', false).text('Run Manual Cleanup Now');
				}
			});
		});
	}

	/**
	 * Initialize on document ready
	 */
	$(document).ready(function() {
		initFeatureFlags();
	});

})(jQuery);
