/**
 * Benchmarking Dashboard JavaScript
 *
 * Handles:
 * - Period selector tabs
 * - AJAX data loading
 * - Chart.js visualization
 * - Error handling and loading states
 *
 * @package FlxWoo\Admin\Assets
 * @since 2.4.0
 */

(function($) {
    'use strict';

    let benchmarkChart = null;

    /**
     * Initialize benchmarking dashboard
     */
    function initBenchmarkingDashboard() {
        // Initialize chart
        const comparisonData = getComparisonData();
        if (comparisonData && comparisonData.has_data) {
            renderBenchmarkChart(comparisonData);
        }

        // Bind period selector
        $('.flx-period-tab').on('click', function() {
            const period = $(this).data('period');
            loadBenchmarkData(period);
        });
    }

    /**
     * Get comparison data from page
     */
    function getComparisonData() {
        const dataElement = $('#flx-comparison-data');
        if (dataElement.length === 0) {
            return null;
        }

        try {
            return JSON.parse(dataElement.text());
        } catch (e) {
            console.error('Failed to parse comparison data:', e);
            return null;
        }
    }

    /**
     * Render benchmark comparison chart
     */
    function renderBenchmarkChart(data) {
        const canvas = document.getElementById('flx-benchmark-chart');
        if (!canvas) {
            return;
        }

        // Destroy existing chart
        if (benchmarkChart !== null) {
            benchmarkChart.destroy();
        }

        const ctx = canvas.getContext('2d');

        // Prepare chart data
        const labels = ['Conversion Rate (%)', 'Avg Cart Value ($)', 'Page Load (seconds)'];
        const storeData = [
            data.conversion_rate.store,
            data.cart_value.store,
            data.page_load.store
        ];
        const industryData = [
            data.conversion_rate.industry,
            data.cart_value.industry,
            data.page_load.industry
        ];

        // Create chart
        benchmarkChart = new Chart(ctx, {
            type: 'bar',
            data: {
                labels: labels,
                datasets: [
                    {
                        label: 'Your Store',
                        data: storeData,
                        backgroundColor: 'rgba(34, 113, 177, 0.8)',
                        borderColor: 'rgba(34, 113, 177, 1)',
                        borderWidth: 2
                    },
                    {
                        label: 'Industry Average',
                        data: industryData,
                        backgroundColor: 'rgba(100, 105, 112, 0.5)',
                        borderColor: 'rgba(100, 105, 112, 1)',
                        borderWidth: 2
                    }
                ]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        position: 'top',
                        labels: {
                            font: {
                                size: 14,
                                weight: '600'
                            },
                            padding: 15
                        }
                    },
                    tooltip: {
                        backgroundColor: 'rgba(29, 35, 39, 0.95)',
                        titleFont: {
                            size: 14,
                            weight: '600'
                        },
                        bodyFont: {
                            size: 13
                        },
                        padding: 12,
                        cornerRadius: 4,
                        callbacks: {
                            label: function(context) {
                                let label = context.dataset.label || '';
                                if (label) {
                                    label += ': ';
                                }
                                const value = context.parsed.y;
                                const metric = context.label;

                                // Format based on metric
                                if (metric.includes('Cart Value')) {
                                    label += '$' + value.toFixed(2);
                                } else if (metric.includes('Page Load')) {
                                    label += value.toFixed(2) + 's';
                                } else {
                                    label += value.toFixed(1) + '%';
                                }

                                return label;
                            }
                        }
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true,
                        ticks: {
                            font: {
                                size: 12
                            }
                        },
                        grid: {
                            color: 'rgba(0, 0, 0, 0.05)'
                        }
                    },
                    x: {
                        ticks: {
                            font: {
                                size: 13,
                                weight: '600'
                            }
                        },
                        grid: {
                            display: false
                        }
                    }
                }
            }
        });
    }

    /**
     * Load benchmark data via AJAX
     */
    function loadBenchmarkData(period) {
        // Show loading overlay
        $('#flx-benchmark-loading').fadeIn(200);

        // Update active tab
        $('.flx-period-tab').removeClass('active');
        $('.flx-period-tab[data-period="' + period + '"]').addClass('active');

        // AJAX request
        $.ajax({
            url: window.flxBenchmarkData.ajaxurl,
            type: 'POST',
            data: {
                action: 'flx_get_benchmark_data',
                nonce: window.flxBenchmarkData.nonce,
                period: period
            },
            success: function(response) {
                if (response.success) {
                    updateDashboard(response.data);
                } else {
                    showError(response.data.message || 'Failed to load benchmark data');
                }
            },
            error: function(xhr, status, error) {
                console.error('AJAX error:', status, error);
                showError('Failed to connect to server. Please try again.');
            },
            complete: function() {
                $('#flx-benchmark-loading').fadeOut(200);
            }
        });
    }

    /**
     * Update dashboard with new data
     */
    function updateDashboard(data) {
        // Update overview stats
        if (data.overview) {
            updateOverviewStats(data.overview);
        }

        // Update chart
        if (data.comparison && data.comparison.has_data) {
            renderBenchmarkChart(data.comparison);
        }

        // Update insights
        if (data.insights) {
            updateInsights(data.insights);
        }
    }

    /**
     * Update overview statistics
     */
    function updateOverviewStats(overview) {
        // Update conversion rate
        $('.flx-info-item:eq(0) span:first').text(overview.conversion_rate);

        // Update cart value
        $('.flx-info-item:eq(1) span:first').text(overview.cart_value);

        // Update page load
        $('.flx-info-item:eq(2) span:first').text(overview.page_load);

        // Update revenue
        $('.flx-info-item:eq(3) span:first').text(overview.total_revenue);
    }

    /**
     * Update insights panel
     */
    function updateInsights(insights) {
        const container = $('.flx-insights-list');
        container.empty();

        if (insights.length === 0) {
            container.append('<p style="color: #646970;">No insights available for this period.</p>');
            return;
        }

        insights.forEach(function(insight) {
            const item = $('<div class="flx-insight-item"></div>');
            item.append('<span class="dashicons dashicons-arrow-right-alt" style="color: #2271b1;"></span>');
            item.append('<p>' + escapeHtml(insight) + '</p>');
            container.append(item);
        });
    }

    /**
     * Show error message
     */
    function showError(message) {
        const notice = $('<div class="notice notice-error is-dismissible"><p></p></div>');
        notice.find('p').text(message);

        $('.flx-benchmarking-dashboard h1').after(notice);

        // Auto-dismiss after 5 seconds
        setTimeout(function() {
            notice.slideUp(300, function() {
                $(this).remove();
            });
        }, 5000);
    }

    /**
     * Escape HTML to prevent XSS
     */
    function escapeHtml(text) {
        const map = {
            '&': '&amp;',
            '<': '&lt;',
            '>': '&gt;',
            '"': '&quot;',
            "'": '&#039;'
        };
        return text.replace(/[&<>"']/g, function(m) { return map[m]; });
    }

    // Initialize on document ready
    $(document).ready(function() {
        // Only initialize if on benchmarking page
        if ($('.flx-benchmarking-dashboard').length > 0) {
            initBenchmarkingDashboard();
        }
    });

})(jQuery);
