/**
 * Activity Analytics Filters
 *
 * Handles advanced filtering, CSV export, table population, and pagination
 * for the Activity Analytics dashboard.
 *
 * @package FlxWoo
 * @since 2.4.0
 */

(function($) {
	'use strict';

	const ActivityFilters = {
		currentPage: 1,
		perPage: 25,
		totalPages: 0,
		totalCount: 0,
		filtersActive: false,

		/**
		 * Initialize filters
		 */
		init: function() {
			this.bindEvents();
			this.loadStoredFilters();
		},

		/**
		 * Bind event handlers
		 */
		bindEvents: function() {
			// Filter form submission
			$('#flx-analytics-filters').on('submit', (e) => {
				e.preventDefault();
				this.currentPage = 1;
				this.applyFilters();
			});

			// Clear filters button
			$('#flx-clear-filters').on('click', () => {
				this.clearFilters();
			});

			// Export CSV button
			$('#flx-export-csv').on('click', () => {
				this.exportCSV();
			});

			// Per-page selector
			$('#flx-per-page').on('change', () => {
				this.perPage = parseInt($('#flx-per-page').val());
				this.currentPage = 1;
				this.loadTableData();
			});

			// Pagination clicks (delegated)
			$(document).on('click', '#flx-activity-pagination a', (e) => {
				e.preventDefault();
				const page = $(e.target).data('page');
				if (page && page !== this.currentPage) {
					this.currentPage = page;
					this.loadTableData();
				}
			});

			// Date range change (update charts)
			$('#flx-date-range').on('change', () => {
				this.updateCharts();
			});
		},

		/**
		 * Apply filters and load table data
		 */
		applyFilters: function() {
			this.filtersActive = true;
			this.storeFilters();
			this.loadTableData();
			this.updateCharts();

			// Show table section
			$('#flx-activity-table-section').show();
		},

		/**
		 * Clear all filters
		 */
		clearFilters: function() {
			// Reset form
			$('#flx-analytics-filters')[0].reset();
			$('#flx-date-range').val('30'); // Default to 30 days

			// Clear localStorage
			localStorage.removeItem('flxwoo_analytics_filters');

			this.filtersActive = false;
			this.currentPage = 1;

			// Hide table section
			$('#flx-activity-table-section').hide();

			// Reload charts with default date range
			this.updateCharts();
		},

		/**
		 * Load table data via AJAX
		 */
		loadTableData: function() {
			const filters = this.getFilterValues();

			// Show loading
			$('#flx-activity-table-loading').show();
			$('#flx-activity-table-container').hide();
			$('#flx-activity-table-empty').hide();

			$.ajax({
				url: ajaxurl,
				type: 'POST',
				data: {
					action: 'flx_get_filtered_data',
					nonce: flxAnalytics.nonce,
					...filters,
					page: this.currentPage,
					per_page: this.perPage
				},
				success: (response) => {
					if (response.success) {
						this.totalCount = response.data.total_count;
						this.totalPages = response.data.total_pages;
						this.renderTable(response.data.entries);
						this.renderPagination();

						// Show/hide appropriate sections
						if (this.totalCount > 0) {
							$('#flx-activity-table-container').show();
							$('#flx-activity-table-empty').hide();
						} else {
							$('#flx-activity-table-container').hide();
							$('#flx-activity-table-empty').show();
						}
					} else {
						this.showError(response.data.message || 'Failed to load data');
					}
				},
				error: (xhr, status, error) => {
					this.showError('AJAX error: ' + error);
				},
				complete: () => {
					$('#flx-activity-table-loading').hide();
				}
			});
		},

		/**
		 * Render table rows
		 */
		renderTable: function(entries) {
			const tbody = $('#flx-activity-table-body');
			tbody.empty();

			if (!entries || entries.length === 0) {
				return;
			}

			entries.forEach((entry) => {
				const row = $('<tr></tr>');

				row.append($('<td></td>').html(
					'<span title="' + this.escapeHtml(entry.timestamp) + '">' +
					this.escapeHtml(entry.timestamp_human) +
					'</span>'
				));
				row.append($('<td></td>').text(entry.flag_name));
				row.append($('<td></td>').text(entry.action));
				row.append($('<td></td>').html(this.formatValue(entry.old_value)));
				row.append($('<td></td>').html(this.formatValue(entry.new_value)));
				row.append($('<td></td>').text(entry.user_login));
				row.append($('<td></td>').text(entry.ip_address));

				tbody.append(row);
			});

			// Update info text
			const start = ((this.currentPage - 1) * this.perPage) + 1;
			const end = Math.min(this.currentPage * this.perPage, this.totalCount);
			$('#flx-table-info').text(
				'Showing ' + start + ' - ' + end + ' of ' + this.totalCount + ' entries'
			);
		},

		/**
		 * Render pagination links
		 */
		renderPagination: function() {
			const pagination = $('#flx-activity-pagination');
			pagination.empty();

			if (this.totalPages <= 1) {
				return;
			}

			const links = $('<div class="pagination-links"></div>');

			// Previous button
			if (this.currentPage > 1) {
				links.append(
					$('<a href="#" data-page="' + (this.currentPage - 1) + '">‹ Prev</a>')
				);
			} else {
				links.append($('<span class="disabled">‹ Prev</span>'));
			}

			// Page numbers (show max 5 pages)
			const startPage = Math.max(1, this.currentPage - 2);
			const endPage = Math.min(this.totalPages, this.currentPage + 2);

			if (startPage > 1) {
				links.append($('<a href="#" data-page="1">1</a>'));
				if (startPage > 2) {
					links.append($('<span>...</span>'));
				}
			}

			for (let i = startPage; i <= endPage; i++) {
				if (i === this.currentPage) {
					links.append($('<span class="current">' + i + '</span>'));
				} else {
					links.append($('<a href="#" data-page="' + i + '">' + i + '</a>'));
				}
			}

			if (endPage < this.totalPages) {
				if (endPage < this.totalPages - 1) {
					links.append($('<span>...</span>'));
				}
				links.append($('<a href="#" data-page="' + this.totalPages + '">' + this.totalPages + '</a>'));
			}

			// Next button
			if (this.currentPage < this.totalPages) {
				links.append(
					$('<a href="#" data-page="' + (this.currentPage + 1) + '">Next ›</a>')
				);
			} else {
				links.append($('<span class="disabled">Next ›</span>'));
			}

			pagination.append(links);
		},

		/**
		 * Export data to CSV
		 */
		exportCSV: function() {
			const filters = this.getFilterValues();

			// Create a form and submit it
			const form = $('<form>', {
				method: 'POST',
				action: ajaxurl,
				target: '_blank'
			});

			form.append($('<input>', {
				type: 'hidden',
				name: 'action',
				value: 'flx_export_csv'
			}));

			form.append($('<input>', {
				type: 'hidden',
				name: 'nonce',
				value: flxAnalytics.nonce
			}));

			// Add all filter values
			for (const [key, value] of Object.entries(filters)) {
				if (Array.isArray(value)) {
					value.forEach((v) => {
						form.append($('<input>', {
							type: 'hidden',
							name: key + '[]',
							value: v
						}));
					});
				} else if (value !== '' && value !== null) {
					form.append($('<input>', {
						type: 'hidden',
						name: key,
						value: value
					}));
				}
			}

			$('body').append(form);
			form.submit();
			form.remove();
		},

		/**
		 * Get current filter values from form
		 */
		getFilterValues: function() {
			const filters = {};

			// Date range
			const dateRange = $('#flx-date-range').val();
			if (dateRange) {
				filters.date_range = dateRange;
			}

			// Feature flags (multiple select)
			const flagNames = $('#flx-flag-names').val();
			if (flagNames && flagNames.length > 0) {
				filters.flag_names = flagNames;
			}

			// Actions (multiple select)
			const actions = $('#flx-actions').val();
			if (actions && actions.length > 0) {
				filters.actions = actions;
			}

			// User
			const userId = $('#flx-users').val();
			if (userId) {
				filters.user_id = userId;
			}

			return filters;
		},

		/**
		 * Store filters in localStorage
		 */
		storeFilters: function() {
			const filters = this.getFilterValues();
			localStorage.setItem('flxwoo_analytics_filters', JSON.stringify(filters));
		},

		/**
		 * Load stored filters from localStorage
		 */
		loadStoredFilters: function() {
			const stored = localStorage.getItem('flxwoo_analytics_filters');
			if (!stored) return;

			try {
				const filters = JSON.parse(stored);

				// Apply filter values to form
				if (filters.date_range) {
					$('#flx-date-range').val(filters.date_range);
				}

				if (filters.flag_names) {
					$('#flx-flag-names').val(filters.flag_names);
				}

				if (filters.actions) {
					$('#flx-actions').val(filters.actions);
				}

				if (filters.user_id) {
					$('#flx-users').val(filters.user_id);
				}

				// Auto-apply filters if any were stored
				if (Object.keys(filters).length > 0) {
					this.filtersActive = true;
					this.loadTableData();
					$('#flx-activity-table-section').show();
				}
			} catch (e) {
				console.error('Failed to load stored filters:', e);
			}
		},

		/**
		 * Update charts based on current date range
		 */
		updateCharts: function() {
			const dateRange = $('#flx-date-range').val() || 30;

			// Trigger timeline chart update (assuming global function exists)
			if (typeof window.flxLoadTimelineChart === 'function') {
				window.flxLoadTimelineChart(dateRange);
			}
		},

		/**
		 * Format value for display
		 */
		formatValue: function(value) {
			if (value === '—' || value === '' || value === null) {
				return '<span style="color: #8c8f94;">—</span>';
			}

			// If value looks like JSON, pretty print it
			if (typeof value === 'string' && (value.startsWith('{') || value.startsWith('['))) {
				return '<code style="display: block; max-width: 200px; overflow: hidden; text-overflow: ellipsis;">' +
					this.escapeHtml(value) +
					'</code>';
			}

			return this.escapeHtml(value);
		},

		/**
		 * Escape HTML
		 */
		escapeHtml: function(text) {
			if (typeof text !== 'string') {
				return text;
			}
			const div = document.createElement('div');
			div.textContent = text;
			return div.innerHTML;
		},

		/**
		 * Show error message
		 */
		showError: function(message) {
			const errorDiv = $('<div class="notice notice-error is-dismissible"><p>' + message + '</p></div>');
			$('.flx-performance-dashboard').prepend(errorDiv);
			setTimeout(() => {
				errorDiv.fadeOut(() => errorDiv.remove());
			}, 5000);
		}
	};

	// Initialize on document ready
	$(document).ready(function() {
		ActivityFilters.init();
	});

})(jQuery);
