/**
 * Activity Analytics Dashboard JavaScript
 *
 * Initializes Chart.js charts and handles AJAX data loading for activity analytics.
 *
 * @package FlxWoo
 * @since 2.4.0
 */

(function($) {
	'use strict';

	let charts = {};
	let currentDateRange = 30;

	/**
	 * Initialize all analytics charts
	 */
	function initActivityAnalytics() {
		initTimelineChart();
		initFeatureChart();
		initUserChart();
		initActionChart();
		initDateRangeFilter();
	}

	/**
	 * Initialize timeline line chart
	 */
	function initTimelineChart() {
		const $canvas = $('#flx-timeline-chart');
		const $loading = $('#flx-timeline-loading');

		if (!$canvas.length) return;

		const ctx = $canvas[0].getContext('2d');

		$.post(flxAnalytics.ajaxurl, {
			action: 'flx_get_timeline_data',
			nonce: flxAnalytics.nonce,
			date_range: currentDateRange
		})
		.done(function(response) {
			if (response.success) {
				$loading.hide();
				$canvas.show();

				charts.timeline = new Chart(ctx, {
					type: 'line',
					data: response.data,
					options: {
						responsive: true,
						maintainAspectRatio: false,
						plugins: {
							legend: {
								display: false
							},
							tooltip: {
								backgroundColor: 'rgba(0, 0, 0, 0.8)',
								padding: 12,
								titleFont: {
									size: 14
								},
								bodyFont: {
									size: 13
								}
							}
						},
						scales: {
							y: {
								beginAtZero: true,
								ticks: {
									precision: 0
								},
								grid: {
									color: 'rgba(0, 0, 0, 0.05)'
								}
							},
							x: {
								grid: {
									display: false
								}
							}
						}
					}
				});
			}
		})
		.fail(function() {
			$loading.html('<p style="color: #d63638;">' +
				'Failed to load timeline data. Please refresh the page.' +
			'</p>');
		});
	}

	/**
	 * Initialize feature breakdown horizontal bar chart
	 */
	function initFeatureChart() {
		const $canvas = $('#flx-feature-chart');
		const $loading = $('#flx-feature-loading');

		if (!$canvas.length) return;

		const ctx = $canvas[0].getContext('2d');

		$.post(flxAnalytics.ajaxurl, {
			action: 'flx_get_feature_breakdown',
			nonce: flxAnalytics.nonce
		})
		.done(function(response) {
			if (response.success) {
				$loading.hide();
				$canvas.show();

				charts.feature = new Chart(ctx, {
					type: 'bar',
					data: response.data,
					options: {
						responsive: true,
						maintainAspectRatio: false,
						indexAxis: 'y',  // Horizontal bars
						plugins: {
							legend: {
								display: false
							},
							tooltip: {
								backgroundColor: 'rgba(0, 0, 0, 0.8)',
								padding: 12
							}
						},
						scales: {
							x: {
								beginAtZero: true,
								ticks: {
									precision: 0
								},
								grid: {
									color: 'rgba(0, 0, 0, 0.05)'
								}
							},
							y: {
								grid: {
									display: false
								}
							}
						}
					}
				});
			}
		})
		.fail(function() {
			$loading.html('<p style="color: #d63638;">' +
				'Failed to load feature data. Please refresh the page.' +
			'</p>');
		});
	}

	/**
	 * Initialize user activity pie chart
	 */
	function initUserChart() {
		const $canvas = $('#flx-user-chart');
		const $loading = $('#flx-user-loading');

		if (!$canvas.length) return;

		const ctx = $canvas[0].getContext('2d');

		$.post(flxAnalytics.ajaxurl, {
			action: 'flx_get_user_activity',
			nonce: flxAnalytics.nonce
		})
		.done(function(response) {
			if (response.success) {
				$loading.hide();
				$canvas.show();

				charts.user = new Chart(ctx, {
					type: 'pie',
					data: response.data,
					options: {
						responsive: true,
						maintainAspectRatio: false,
						plugins: {
							legend: {
								position: 'right',
								labels: {
									padding: 15,
									font: {
										size: 12
									}
								}
							},
							tooltip: {
								backgroundColor: 'rgba(0, 0, 0, 0.8)',
								padding: 12,
								callbacks: {
									label: function(context) {
										const label = context.label || '';
										const value = context.parsed || 0;
										const total = context.dataset.data.reduce((a, b) => a + b, 0);
										const percentage = ((value / total) * 100).toFixed(1);
										return label + ': ' + value + ' (' + percentage + '%)';
									}
								}
							}
						}
					}
				});
			}
		})
		.fail(function() {
			$loading.html('<p style="color: #d63638;">' +
				'Failed to load user data. Please refresh the page.' +
			'</p>');
		});
	}

	/**
	 * Initialize action distribution doughnut chart
	 */
	function initActionChart() {
		const $canvas = $('#flx-action-chart');
		const $loading = $('#flx-action-loading');

		if (!$canvas.length) return;

		const ctx = $canvas[0].getContext('2d');

		$.post(flxAnalytics.ajaxurl, {
			action: 'flx_get_action_distribution',
			nonce: flxAnalytics.nonce
		})
		.done(function(response) {
			if (response.success) {
				$loading.hide();
				$canvas.show();

				charts.action = new Chart(ctx, {
					type: 'doughnut',
					data: response.data,
					options: {
						responsive: true,
						maintainAspectRatio: false,
						plugins: {
							legend: {
								position: 'right',
								labels: {
									padding: 15,
									font: {
										size: 12
									}
								}
							},
							tooltip: {
								backgroundColor: 'rgba(0, 0, 0, 0.8)',
								padding: 12,
								callbacks: {
									label: function(context) {
										const label = context.label || '';
										const value = context.parsed || 0;
										const total = context.dataset.data.reduce((a, b) => a + b, 0);
										const percentage = ((value / total) * 100).toFixed(1);
										return label + ': ' + value + ' (' + percentage + '%)';
									}
								}
							}
						}
					}
				});
			}
		})
		.fail(function() {
			$loading.html('<p style="color: #d63638;">' +
				'Failed to load action data. Please refresh the page.' +
			'</p>');
		});
	}

	/**
	 * Initialize date range filter
	 */
	function initDateRangeFilter() {
		$('#flx-analytics-filters').on('submit', function(e) {
			e.preventDefault();

			const dateRange = parseInt($(this).find('[name="date_range"]').val(), 10);

			if (dateRange === currentDateRange) {
				return; // No change
			}

			currentDateRange = dateRange;
			updateCharts();
		});
	}

	/**
	 * Update all charts with new date range
	 */
	function updateCharts() {
		// Destroy existing charts
		if (charts.timeline) {
			charts.timeline.destroy();
			$('#flx-timeline-chart').hide();
			$('#flx-timeline-loading').show().text('Loading chart...');
		}

		// Reinitialize timeline with new date range
		// Other charts don't depend on date range
		initTimelineChart();
	}

	/**
	 * Initialize on document ready
	 */
	$(document).ready(function() {
		if ($('#flx-timeline-chart').length) {
			initActivityAnalytics();
		}
	});

})(jQuery);
