/**
 * A/B Testing Page JavaScript
 *
 * @package FlxWoo
 * @version 2.5.0
 */

(function($) {
    'use strict';

    // Modal Management
    const Modal = {
        open: function(modalId) {
            $('#' + modalId).addClass('active').fadeIn(200);
        },
        close: function(modalId) {
            $('#' + modalId).removeClass('active').fadeOut(200);
        }
    };

    // Test Creation
    const TestCreation = {
        variantIndex: 2, // Start at 2 (control and variant_a already exist)

        init: function() {
            $('#flx-create-test-btn').on('click', () => Modal.open('flx-create-test-modal'));
            $('#flx-add-variant').on('click', () => this.addVariant());
            $(document).on('click', '.flx-remove-variant', (e) => this.removeVariant(e));
            $('#flx-create-test-form').on('submit', (e) => this.submitForm(e));
        },

        addVariant: function() {
            if ($('.flx-variant-row').length >= 5) {
                alert('Maximum 5 variants allowed');
                return;
            }

            const html = `
                <div class="flx-variant-row" data-variant-index="${this.variantIndex}">
                    <input type="text" class="variant-id" placeholder="variant_${String.fromCharCode(97 + this.variantIndex - 1)}" required>
                    <input type="text" class="variant-name" placeholder="Variant Name" required>
                    <input type="number" class="variant-traffic" min="0" max="100" value="0" required>
                    <span>%</span>
                    <button type="button" class="button flx-remove-variant">×</button>
                </div>
            `;
            $('#flx-variants-container').append(html);
            this.variantIndex++;
            this.redistributeTraffic();
        },

        removeVariant: function(e) {
            const $row = $(e.target).closest('.flx-variant-row');
            if ($('.flx-variant-row').length <= 2) {
                alert('Minimum 2 variants required');
                return;
            }
            $row.remove();
            this.redistributeTraffic();
        },

        redistributeTraffic: function() {
            const $rows = $('.flx-variant-row');
            const count = $rows.length;
            const perVariant = Math.floor(100 / count);
            const remainder = 100 - (perVariant * count);

            $rows.each(function(i) {
                const value = i === 0 ? perVariant + remainder : perVariant;
                $(this).find('.variant-traffic').val(value);
            });
        },

        validateTrafficSplit: function() {
            let total = 0;
            $('.variant-traffic').each(function() {
                total += parseInt($(this).val()) || 0;
            });
            return Math.abs(total - 100) < 0.01;
        },

        submitForm: function(e) {
            e.preventDefault();

            if (!this.validateTrafficSplit()) {
                alert(flxABTestingData.strings.trafficSplitError);
                return;
            }

            const variants = [];
            const trafficSplit = {};

            $('.flx-variant-row').each(function() {
                const id = $(this).find('.variant-id').val();
                const name = $(this).find('.variant-name').val();
                const traffic = parseInt($(this).find('.variant-traffic').val());

                variants.push({
                    id: id,
                    name: name,
                    config: {}
                });
                trafficSplit[id] = traffic;
            });

            const data = {
                action: 'flx_create_ab_test',
                nonce: flxABTestingData.nonce,
                test_name: $('#test_name').val(),
                test_key: $('#test_key').val(),
                variants: JSON.stringify(variants),
                traffic_split: JSON.stringify(trafficSplit)
            };

            $.post(flxABTestingData.ajaxUrl, data, function(response) {
                if (response.success) {
                    alert(flxABTestingData.strings.createSuccess);
                    location.reload();
                } else {
                    alert(response.data.message || flxABTestingData.strings.createError);
                }
            }).fail(function() {
                alert(flxABTestingData.strings.createError);
            });
        }
    };

    // Test Actions
    const TestActions = {
        init: function() {
            $(document).on('click', '.flx-start-test', (e) => this.startTest(e));
            $(document).on('click', '.flx-stop-test', (e) => this.stopTest(e));
            $(document).on('click', '.flx-view-results', (e) => this.viewResults(e));
        },

        startTest: function(e) {
            const testId = $(e.target).data('test-id');
            if (!confirm(flxABTestingData.strings.confirmStart)) {
                return;
            }

            $.post(flxABTestingData.ajaxUrl, {
                action: 'flx_start_ab_test',
                nonce: flxABTestingData.nonce,
                test_id: testId
            }, function(response) {
                if (response.success) {
                    location.reload();
                } else {
                    alert(response.data.message || 'Failed to start test');
                }
            });
        },

        stopTest: function(e) {
            const testId = $(e.target).data('test-id');
            if (!confirm(flxABTestingData.strings.confirmStop)) {
                return;
            }

            $.post(flxABTestingData.ajaxUrl, {
                action: 'flx_stop_ab_test',
                nonce: flxABTestingData.nonce,
                test_id: testId
            }, function(response) {
                if (response.success) {
                    location.reload();
                } else {
                    alert(response.data.message || 'Failed to stop test');
                }
            });
        },

        viewResults: function(e) {
            const testId = $(e.target).data('test-id');
            const testName = $(e.target).closest('tr').find('strong').text();

            Modal.open('flx-results-modal');
            $('#results-test-name').text(testName);
            $('#flx-results-container').html('<p class="flx-loading">Loading results...</p>');

            $.post(flxABTestingData.ajaxUrl, {
                action: 'flx_get_test_results',
                nonce: flxABTestingData.nonce,
                test_id: testId
            }, function(response) {
                if (response.success) {
                    ResultsDisplay.render(response.data);
                } else {
                    $('#flx-results-container').html('<p class="error">Failed to load results</p>');
                }
            });
        }
    };

    // Results Display
    const ResultsDisplay = {
        render: function(data) {
            let html = '';

            // Recommendation
            if (data.recommendation) {
                const recommendationClass = data.statistical_significance >= 0.95 ? 'success' : '';
                html += `
                    <div class="flx-recommendation ${recommendationClass}">
                        <p><strong>Recommendation:</strong> ${data.recommendation}</p>
                    </div>
                `;
            }

            // Variant Results Cards
            html += '<div class="flx-results-grid">';
            data.variants.forEach(variant => {
                const isWinner = variant.variant_id === data.winner;
                const winnerClass = isWinner ? 'winner' : '';

                html += `
                    <div class="flx-result-card ${winnerClass}">
                        <h3>${variant.variant_name}${isWinner ? ' 🏆' : ''}</h3>
                        <div class="flx-result-stat">
                            <span>Impressions:</span>
                            <strong>${variant.impressions.toLocaleString()}</strong>
                        </div>
                        <div class="flx-result-stat">
                            <span>Conversions:</span>
                            <strong>${variant.conversions.toLocaleString()}</strong>
                        </div>
                        <div class="flx-result-stat">
                            <span>Conversion Rate:</span>
                            <strong>${variant.conversion_rate_percent.toFixed(2)}%</strong>
                        </div>
                        <div class="flx-result-stat">
                            <span>Confidence:</span>
                            <span>${variant.confidence_interval.lower.toFixed(2)}% - ${variant.confidence_interval.upper.toFixed(2)}%</span>
                        </div>
                        <div class="flx-result-stat">
                            <span>Total Revenue:</span>
                            <strong>$${(variant.total_revenue_cents / 100).toFixed(2)}</strong>
                        </div>
                    </div>
                `;
            });
            html += '</div>';

            // Statistical Significance
            html += `
                <div class="flx-result-stat" style="padding: 15px; background: #f6f7f7; border-radius: 4px;">
                    <span>Statistical Confidence:</span>
                    <strong>${(data.statistical_significance * 100).toFixed(1)}%</strong>
                </div>
            `;

            $('#flx-results-container').html(html);
        }
    };

    // Modal Close Handlers
    $('.flx-modal-close, .flx-modal-cancel').on('click', function() {
        $(this).closest('.flx-modal').removeClass('active').fadeOut(200);
    });

    // Close modal on outside click
    $('.flx-modal').on('click', function(e) {
        if ($(e.target).hasClass('flx-modal')) {
            $(this).removeClass('active').fadeOut(200);
        }
    });

    // Initialize
    $(document).ready(function() {
        TestCreation.init();
        TestActions.init();
    });

})(jQuery);
