<?php
/**
 * Settings Manager
 *
 * Handles all FlxWoo settings database operations, validation, and fallback logic.
 *
 * Fallback chain: Database Settings > wp-config.php Constants > Default Values
 *
 * @package FlxWoo
 * @since 2.1.0
 */

namespace FlxWoo\Admin;

if (!defined('ABSPATH')) {
    exit;
}

class SettingsManager {
    /**
     * WordPress option name for storing settings
     */
    const OPTION_NAME = 'flx_woo_settings';

    /**
     * Settings version for migration tracking
     */
    const SETTINGS_VERSION = '1.0.0';

    /**
     * Flag to prevent infinite recursion during update_option hooks
     */
    private static $updating = false;

    /**
     * Get all settings as an array
     *
     * @return array Associative array of all settings
     */
    public function get_all_settings() {
        $settings = get_option(self::OPTION_NAME, []);
        $defaults = $this->get_default_settings();

        // Ensure all keys exist with defaults
        $merged = array_merge($defaults, $settings);

        // Fix null values from database - use defaults instead
        foreach ($merged as $key => $value) {
            if ($value === null && isset($defaults[$key])) {
                $merged[$key] = $defaults[$key];
            }
        }

        return $merged;
    }

    /**
     * Get a single setting value
     *
     * @param string $key Setting key
     * @param mixed $default Default value if not found
     * @return mixed Setting value
     */
    public function get_setting($key, $default = null) {
        $settings = $this->get_all_settings();

        if (isset($settings[$key])) {
            return $settings[$key];
        }

        // Fallback to wp-config.php constants if available
        return $this->get_constant_fallback($key, $default);
    }

    /**
     * Update a single setting
     *
     * @param string $key Setting key
     * @param mixed $value Setting value
     * @return bool True on success, false on failure
     */
    public function update_setting($key, $value) {
        $settings = $this->get_all_settings();

        // Validate the value (null indicates validation failure)
        $validated_value = $this->validate_setting($key, $value);

        if ($validated_value === null) {
            return false;
        }

        $settings[$key] = $validated_value;

        return update_option(self::OPTION_NAME, $settings);
    }

    /**
     * Update all settings at once
     *
     * @param array $new_settings Associative array of settings to update
     * @return bool|array True on success, array of errors on validation failure
     */
    public function update_all_settings($new_settings) {
        \FlxWoo\Utils\Logger::debug('Starting settings update', ['new_settings' => $new_settings]);

        $errors = [];

        // Get current settings directly from database without any method calls
        // to avoid triggering any potential recursion
        global $wpdb;
        $current_raw = $wpdb->get_var($wpdb->prepare(
            "SELECT option_value FROM {$wpdb->options} WHERE option_name = %s",
            self::OPTION_NAME
        ));
        $current_settings = $current_raw ? \maybe_unserialize($current_raw) : [];
        if (!is_array($current_settings)) {
            $current_settings = [];
        }

        \FlxWoo\Utils\Logger::debug('Current settings from DB', ['current_settings' => $current_settings]);

        // Start with defaults to ensure all keys exist
        $validated_settings = $this->get_default_settings();

        // Merge in current settings (overrides defaults)
        $validated_settings = array_merge($validated_settings, $current_settings);

        // Validate and merge new settings
        foreach ($new_settings as $key => $value) {
            $validated_value = $this->validate_setting($key, $value);

            // Use null to indicate validation failure (false is a valid boolean value)
            if ($validated_value === null) {
                $errors[$key] = $this->get_validation_error($key);
            } else {
                $validated_settings[$key] = $validated_value;
            }
        }

        if (!empty($errors)) {
            return $errors;
        }

        \FlxWoo\Utils\Logger::debug('About to save settings', [
            'validated_settings' => $validated_settings,
        ]);

        // Write directly to database to completely bypass WordPress hooks
        $serialized_value = \maybe_serialize($validated_settings);
        $result = $wpdb->query($wpdb->prepare(
            "INSERT INTO {$wpdb->options} (option_name, option_value, autoload) VALUES (%s, %s, %s)
             ON DUPLICATE KEY UPDATE option_value = VALUES(option_value)",
            self::OPTION_NAME,
            $serialized_value,
            'no'
        ));

        \FlxWoo\Utils\Logger::debug('Direct database write result', [
            'query_result' => $result,
            'wpdb_last_error' => $wpdb->last_error,
        ]);

        // Verify it was saved
        $verify_raw = $wpdb->get_var($wpdb->prepare(
            "SELECT option_value FROM {$wpdb->options} WHERE option_name = %s",
            self::OPTION_NAME
        ));
        $verify_settings = $verify_raw ? \maybe_unserialize($verify_raw) : [];

        \FlxWoo\Utils\Logger::debug('Verified settings in DB', [
            'verified_settings' => $verify_settings,
        ]);

        // Return true if the write succeeded (affected rows > 0)
        return $result !== false && $result > 0;
    }

    /**
     * Reset all settings to defaults
     *
     * @return bool True on success
     */
    public function reset_to_defaults() {
        $result = $this->update_all_settings($this->get_default_settings());

        // update_all_settings returns true on success, or array of errors on validation failure
        // If settings are already at defaults, it may return false (0 rows affected)
        // We should treat "already at defaults" as success
        if ($result === false) {
            // Verify settings match defaults
            $current = $this->get_all_settings();
            $defaults = $this->get_default_settings();

            // If settings match defaults, consider it a success
            if ($current === $defaults) {
                return true;
            }
        }

        return $result === true;
    }

    /**
     * Get default settings
     *
     * NOTE: Renderer configuration (URL, timeout, version) is NOT stored in database
     * for SaaS model. These are hardcoded in Constants.php and managed by SaaS provider.
     *
     * NOTE: Caching is NOT applicable to FlxWoo. We render Cart, Checkout, and Thank You
     * pages which are all user-specific and dynamic. Caching would break e-commerce functionality.
     *
     * @return array Default settings array
     */
    public function get_default_settings() {
        return [
            // Renderer settings removed - hardcoded in Constants.php
            // 'renderer_url' => managed by SaaS provider
            // 'renderer_timeout' => managed by SaaS provider
            // 'renderer_version' => managed by SaaS provider

            // Cache settings removed - not applicable to dynamic e-commerce pages
            // 'cache_enabled' => NOT applicable (user-specific cart/checkout data)
            // 'cache_ttl' => NOT applicable

            // User-configurable settings
            'fallback_enabled' => true,  // Enable fallback to native WooCommerce by default
            'active_pages' => ['cart', 'checkout', 'thank-you'],  // All pages enabled by default
            'dev_mode' => false,  // Development mode disabled by default (requires HTTPS)

            'settings_version' => self::SETTINGS_VERSION,
        ];
    }

    /**
     * Validate a setting value
     *
     * NOTE: Renderer configuration validation removed for SaaS model
     * NOTE: Cache settings validation removed - not applicable to dynamic e-commerce pages
     * NOTE: Returns null to indicate validation failure (false is a valid boolean value)
     *
     * @param string $key Setting key
     * @param mixed $value Setting value
     * @return mixed|null Validated value or null on validation failure
     */
    public function validate_setting($key, $value) {
        switch ($key) {
            // Renderer settings removed - hardcoded in Constants.php
            // case 'renderer_url': managed by SaaS provider
            // case 'renderer_timeout': managed by SaaS provider
            // case 'renderer_version': managed by SaaS provider

            // Cache settings removed - not applicable to dynamic e-commerce pages
            // case 'cache_enabled': NOT applicable
            // case 'cache_ttl': NOT applicable

            // User-configurable settings
            case 'fallback_enabled':
            case 'dev_mode':
                return $this->validate_boolean($value);

            case 'active_pages':
                return $this->validate_active_pages($value);

            case 'settings_version':
                // Version is read-only, always return current version
                return self::SETTINGS_VERSION;

            default:
                // Unknown setting key - return null to indicate validation failure
                return null;
        }
    }

    /**
     * Validate renderer URL
     *
     * @param string $url URL to validate
     * @return string|false Sanitized URL or false on validation failure
     */
    private function validate_renderer_url($url) {
        // Sanitize URL
        $url = esc_url_raw($url, ['http', 'https']);

        if (empty($url)) {
            return false;
        }

        // Parse URL
        $parsed = wp_parse_url($url);

        if (!$parsed || !isset($parsed['scheme']) || !isset($parsed['host'])) {
            return false;
        }

        // Check if localhost
        $is_localhost = in_array(
            $parsed['host'],
            ['localhost', '127.0.0.1', '::1'],
            true
        );

        // Get development mode setting
        $settings = get_option(self::OPTION_NAME, []);
        $development_mode = isset($settings['development_mode']) ? (bool) $settings['development_mode'] : false;

        // Enforce HTTPS except for localhost in development mode
        if ($parsed['scheme'] !== 'https' && !($is_localhost && $development_mode)) {
            return false;
        }

        // Disallow path, query, or fragment in URL
        if (
            isset($parsed['path']) && $parsed['path'] !== '/' ||
            isset($parsed['query']) ||
            isset($parsed['fragment'])
        ) {
            return false;
        }

        // Return URL without trailing slash
        return rtrim($url, '/');
    }

    /**
     * Validate timeout value
     *
     * @param mixed $timeout Timeout value
     * @return int|false Validated timeout or false on validation failure
     */
    private function validate_timeout($timeout) {
        $timeout = absint($timeout);

        // Must be between 1 and 60 seconds
        if ($timeout < 1 || $timeout > 60) {
            return false;
        }

        return $timeout;
    }

    /**
     * Validate version string
     *
     * @param string $version Version string
     * @return string|false Validated version or false on validation failure
     */
    private function validate_version($version) {
        $version = sanitize_text_field($version);

        // Only allow v1 for now (future: v2, v3, etc.)
        $allowed_versions = ['v1'];

        if (!in_array($version, $allowed_versions, true)) {
            return false;
        }

        return $version;
    }

    /**
     * Validate boolean value
     *
     * @param mixed $value Value to validate
     * @return bool Boolean value
     */
    private function validate_boolean($value) {
        return (bool) $value;
    }

    /**
     * Validate active pages array
     *
     * @param mixed $value Value to validate
     * @return array Validated array of page types
     */
    private function validate_active_pages($value) {
        if (!is_array($value)) {
            return [];
        }

        $allowed_pages = ['cart', 'checkout', 'thank-you'];
        $validated = [];

        foreach ($value as $page) {
            $page = sanitize_text_field($page);
            if (in_array($page, $allowed_pages, true)) {
                $validated[] = $page;
            }
        }

        return $validated;
    }

    /**
     * Get fallback value from wp-config.php constants
     *
     * NOTE: Renderer configuration removed - now hardcoded in Constants.php
     *
     * @param string $key Setting key
     * @param mixed $default Default value if constant not defined
     * @return mixed Constant value or default
     */
    private function get_constant_fallback($key, $default) {
        // Renderer configuration is no longer stored in database for SaaS model
        // All settings now stored in database only

        return $default !== null ? $default : $this->get_default_settings()[$key] ?? null;
    }

    /**
     * Get validation error message for a setting key
     *
     * NOTE: Renderer configuration errors removed for SaaS model
     * NOTE: Cache settings errors removed - not applicable to dynamic e-commerce pages
     *
     * @param string $key Setting key
     * @return string Error message
     */
    private function get_validation_error($key) {
        $errors = [
            // Renderer errors removed - hardcoded in Constants.php
            // Cache errors removed - not applicable to dynamic e-commerce pages
            'fallback_enabled' => 'Fallback mode must be a boolean value.',
            'active_pages' => 'Active pages must be an array of valid page types (cart, checkout, thank-you).',
            'dev_mode' => 'Development mode must be a boolean value.',
            'settings_version' => 'Settings version is read-only.',
        ];

        return $errors[$key] ?? 'Invalid value for ' . esc_html($key) . '.';
    }

    /**
     * Check if settings exist in database
     *
     * @return bool True if settings exist
     */
    public function settings_exist() {
        $settings = get_option(self::OPTION_NAME, false);
        return $settings !== false;
    }

    /**
     * Get the renderer URL for use in code
     *
     * NOTE: For SaaS model, this now reads directly from Constants.php
     * (hardcoded by SaaS provider, not user-configurable)
     *
     * @return string Renderer URL
     */
    public function get_renderer_url() {
        // Read from Constants.php (hardcoded for SaaS model)
        require_once __DIR__ . '/../Constants/Constants.php';
        return defined('FLX_WOO_RENDERER_URL') ? FLX_WOO_RENDERER_URL : 'https://flx01.flxwoo.com';
    }

    /**
     * Get the renderer timeout for use in code
     *
     * NOTE: For SaaS model, this now reads directly from Constants.php
     * (hardcoded by SaaS provider, not user-configurable)
     *
     * @return int Timeout in seconds
     */
    public function get_renderer_timeout() {
        // Read from Constants.php (hardcoded for SaaS model)
        require_once __DIR__ . '/../Constants/Constants.php';
        return defined('FLX_WOO_RENDERER_TIMEOUT') ? FLX_WOO_RENDERER_TIMEOUT : 5;
    }
}
