<?php
/**
 * Settings Dashboard Page
 *
 * WordPress admin page for configuring FlxWoo settings.
 *
 * @package FlxWoo\Admin
 * @since 2.3.0 (Simplified from v2.2.0 Performance Dashboard)
 */

namespace FlxWoo\Admin;

if (!defined('ABSPATH')) {
    exit;
}

use FlxWoo\Utils\Logger;
use FlxWoo\Admin\SettingsManager;

class PerformanceDashboard {

    /**
     * Settings manager instance
     * @var SettingsManager
     */
    private $settings_manager;

    /**
     * Constructor
     */
    public function __construct() {
        $this->settings_manager = new SettingsManager();

        // Register AJAX handlers
        add_action('wp_ajax_flx_save_settings', [$this, 'ajax_save_settings']);
        add_action('wp_ajax_flx_reset_settings', [$this, 'ajax_reset_settings']);

        // Register upgrade handler
        add_action('admin_init', [$this, 'handle_version_upgrade']);
    }

    /**
     * Render dashboard page
     *
     * @return void
     */
    public function render(): void {
        // Check user permissions
        if (!current_user_can('manage_woocommerce')) {
            wp_die(esc_html__('You do not have sufficient permissions to access this page.', 'flx-woo'));
        }

        // Include view template
        include __DIR__ . '/views/performance.php';
    }

    /**
     * Get the plugin version
     *
     * @return string Plugin version
     */
    public function get_plugin_version(): string {
        if (!function_exists('get_plugin_data')) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }

        $plugin_file = dirname(dirname(__DIR__)) . '/flx-woo.php';
        $plugin_data = get_plugin_data($plugin_file);

        return $plugin_data['Version'] ?? '1.0.0';
    }

    /**
     * Get renderer version from health endpoint
     * Cached for 1 hour to avoid repeated API calls
     *
     * @return string Renderer version or 'Unknown' if unavailable
     */
    public function get_renderer_version(): string {
        // Check if renderer URL is configured
        if (!defined('FLX_WOO_RENDERER_URL')) {
            return 'Unknown';
        }

        // Try to get cached version first
        $cached_version = get_transient('flx_woo_renderer_version');
        if ($cached_version !== false) {
            return $cached_version;
        }

        // Fetch version from health endpoint
        $health_url = FLX_WOO_RENDERER_URL . '/api/health';
        $response = wp_remote_get($health_url, [
            'timeout' => 3,
            'sslverify' => true,
        ]);

        // Handle errors
        if (is_wp_error($response)) {
            Logger::debug('Failed to fetch renderer version', [
                'error' => $response->get_error_message(),
            ]);
            return 'Unknown';
        }

        $status_code = wp_remote_retrieve_response_code($response);
        if ($status_code === 200 || $status_code === 503) {
            $body = wp_remote_retrieve_body($response);
            $data = json_decode($body, true);

            if (isset($data['version'])) {
                $version = sanitize_text_field($data['version']);
                // Cache for 1 hour
                set_transient('flx_woo_renderer_version', $version, HOUR_IN_SECONDS);
                return $version;
            }
        }

        return 'Unknown';
    }

    /**
     * Get system status for dashboard
     *
     * @return array System status information
     */
    public function get_system_status(): array {
        return [
            'plugin_active' => true,
            'plugin_version' => $this->get_plugin_version(),
            'renderer_version' => $this->get_renderer_version(),
            'renderer_url' => defined('FLX_WOO_RENDERER_URL') ? FLX_WOO_RENDERER_URL : '',
            'timeout' => defined('FLX_WOO_RENDERER_TIMEOUT') ? FLX_WOO_RENDERER_TIMEOUT : 5,
        ];
    }

    /**
     * Get analytics tracking status (v2.3.0)
     *
     * @return array Analytics status information
     */
    public function get_analytics_status(): array {
        // Check if analytics is enabled
        $analytics_enabled = get_option('flxwoo_analytics_enabled', false);

        // Get last aggregation status (if any)
        $aggregation_status = get_transient('flxwoo_aggregation_status');

        // Get events tracked today from analytics API
        $events_today = $this->get_events_today();

        return [
            'enabled' => (bool) $analytics_enabled,
            'events_today' => $events_today,
            'last_aggregation' => $aggregation_status,
        ];
    }

    /**
     * Get events tracked today from Next.js analytics API
     *
     * @return int Number of events tracked today
     */
    private function get_events_today(): int {
        // If analytics is not enabled, return 0
        if (!get_option('flxwoo_analytics_enabled', false)) {
            return 0;
        }

        // Get renderer URL
        if (!defined('FLX_WOO_RENDERER_URL')) {
            return 0;
        }

        $analytics_url = FLX_WOO_RENDERER_URL . '/api/v1/analytics/stats?days=1';

        // Make API request with short timeout (non-blocking)
        $response = wp_remote_get($analytics_url, [
            'timeout' => 2,
            'headers' => [
                'Accept' => 'application/json',
            ],
        ]);

        // Check for errors
        if (is_wp_error($response)) {
            return 0;
        }

        // Parse response
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);

        if (!$data || !isset($data['success']) || !$data['success']) {
            return 0;
        }

        return (int) ($data['data']['events_today'] ?? 0);
    }

    /**
     * AJAX handler: Save settings
     *
     * @return void Sends JSON response
     */
    public function ajax_save_settings(): void {
        // Log that handler was called
        Logger::debug('ajax_save_settings handler called', ['post_data' => $_POST]);

        // Security check
        check_ajax_referer('flx_save_settings', 'nonce');

        // Permission check
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error([
                'message' => __('You do not have permission to save settings.', 'flx-woo'),
            ]);
            return;
        }

        // Parse the form data
        $new_settings = [];

        // Fallback mode (checkbox: checked = '1', unchecked = not present)
        $new_settings['fallback_enabled'] = !empty($_POST['fallback_enabled']);

        // Active pages (array of enabled pages)
        $new_settings['active_pages'] = isset($_POST['active_pages']) && is_array($_POST['active_pages'])
            ? array_map('sanitize_text_field', $_POST['active_pages'])
            : [];

        // Development mode (checkbox: checked = '1', unchecked = not present)
        $new_settings['dev_mode'] = !empty($_POST['dev_mode']);

        Logger::debug('Saving dashboard settings', [
            'new_settings' => $new_settings,
            'post_data' => $_POST,
        ]);

        // Update settings
        $result = $this->settings_manager->update_all_settings($new_settings);

        Logger::debug('Settings save result', [
            'result' => $result,
            'result_type' => gettype($result),
            'new_settings' => $new_settings,
            'saved_settings' => $this->settings_manager->get_all_settings(),
        ]);

        // Check if validation errors occurred (returns array of errors)
        if (is_array($result)) {
            wp_send_json_error([
                'message' => __('Validation failed: ', 'flx-woo') . implode(', ', $result),
            ]);
            return;
        }

        // If we get here, validation passed and update_option was called
        if ($result === true) {
            // Settings were updated (changed)
            wp_send_json_success([
                'message' => __('Settings saved successfully.', 'flx-woo'),
            ]);
        } elseif ($result === false) {
            // Settings unchanged (no change needed) - this is success
            wp_send_json_success([
                'message' => __('Settings saved successfully.', 'flx-woo'),
            ]);
        } else {
            // Unexpected return value (should never happen)
            Logger::error('Unexpected result from update_all_settings', [
                'result' => $result,
                'result_type' => gettype($result),
            ]);
            wp_send_json_error([
                'message' => __('Unexpected error saving settings. Please try again.', 'flx-woo'),
            ]);
        }
    }

    /**
     * AJAX handler: Reset settings to defaults
     *
     * @return void Sends JSON response
     */
    public function ajax_reset_settings(): void {
        // Security check
        check_ajax_referer('flx_dashboard_nonce', 'nonce');

        // Permission check
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error([
                'message' => __('You do not have permission to reset settings.', 'flx-woo'),
            ]);
            return;
        }

        Logger::debug('Resetting settings to defaults');

        // Reset to defaults
        $result = $this->settings_manager->reset_to_defaults();

        if ($result) {
            Logger::info('Settings reset to defaults successfully');
            wp_send_json_success([
                'message' => __('Settings reset to defaults successfully.', 'flx-woo'),
            ]);
        } else {
            Logger::error('Failed to reset settings to defaults');
            wp_send_json_error([
                'message' => __('Failed to reset settings. Please try again.', 'flx-woo'),
            ]);
        }
    }

    /**
     * Handle plugin version upgrade
     * Called on admin_init to check for version changes
     *
     * @return void
     */
    public function handle_version_upgrade(): void {
        $current_db_version = get_option('flx_woo_version', '0.0.0');
        $current_plugin_version = $this->get_plugin_version();

        // Update stored version if changed
        if (version_compare($current_db_version, $current_plugin_version, '<')) {
            // Run migration for v2.3.0+ (dashboard simplification)
            if (version_compare($current_db_version, '2.3.0', '<')) {
                $this->migrate_to_2_3_0();
            }

            update_option('flx_woo_version', $current_plugin_version);

            Logger::info('Plugin version updated', [
                'old_version' => $current_db_version,
                'new_version' => $current_plugin_version,
            ]);
        }
    }

    /**
     * Migration for v2.3.0: Remove obsolete performance monitoring options
     *
     * @return void
     */
    private function migrate_to_2_3_0(): void {
        Logger::info('Running migration to v2.3.0 - Cleaning up obsolete options');

        $obsolete_options = [
            'flx_woo_performance_history',     // Performance test history
            'flx_woo_last_health_check',       // Last health check timestamp
            'flx_woo_last_render_time',        // Last render timestamp
            'flx_woo_render_stats_24h',        // 24-hour render statistics
            'flx_woo_last_cron_test',          // Last cron test timestamp
        ];

        $removed_count = 0;
        foreach ($obsolete_options as $option) {
            if (delete_option($option)) {
                $removed_count++;
                Logger::debug('Removed obsolete option', ['option' => $option]);
            }
        }

        // Clear any scheduled cron jobs for performance testing
        $timestamp = wp_next_scheduled('flx_woo_performance_test');
        if ($timestamp) {
            wp_unschedule_event($timestamp, 'flx_woo_performance_test');
            Logger::debug('Unscheduled performance test cron job');
        }

        Logger::info('Migration to v2.3.0 completed', [
            'removed_options' => $removed_count,
            'cron_cleared' => $timestamp ? true : false,
        ]);
    }

}
