<?php
/**
 * Feature Flags Settings Page
 *
 * Admin page for managing feature flags.
 * Allows enabling/disabling analytics tracking, benchmarking, A/B testing, and plugin compatibility.
 *
 * @package FlxWoo
 * @version 2.3.0
 */

namespace FlxWoo\Admin;

use FlxWoo\FeatureFlags\FeatureManager;
use FlxWoo\FeatureFlags\ActivityLogger;
use FlxWoo\FeatureFlags\RetentionManager;

if (!defined('ABSPATH')) {
	exit;
}

class FeatureFlagsPage {
	/**
	 * Render the feature flags settings page
	 */
	public function render() {
		// Check user capabilities
		if (!\current_user_can('manage_woocommerce')) {
			\wp_die(\__('You do not have sufficient permissions to access this page.', 'flx-woo'));
		}

		// Handle form submission
		if (isset($_POST['flx_woo_save_feature_flags']) && \check_admin_referer('flx_woo_feature_flags_nonce')) {
			$this->save_feature_flags();
		}

		// Handle retention settings submission
		if (isset($_POST['flx_woo_save_retention']) && \check_admin_referer('flx_woo_retention_settings')) {
			$this->save_retention_settings();
		}

		// Get current feature flags
		$flags = FeatureManager::get_flags();
		$store_id = FeatureManager::get_store_id();
		$enabled_features = FeatureManager::get_enabled_features();
		$kill_switch_active = FeatureManager::is_kill_switch_active();

		// Load the view template
		include __DIR__ . '/views/feature-flags.php';
	}

	/**
	 * Save feature flags from form submission
	 */
	private function save_feature_flags() {
		$success = true;
		$errors = [];

		// Track old kill switch value for logging
		$old_kill_switch = FeatureManager::is_kill_switch_active();

		// Update global kill switch
		// Note: Unchecked checkboxes don't send POST data, so we check if form was submitted
		// If checkbox is unchecked, isset() returns false and we set kill switch to false
		$kill_switch = isset($_POST['kill_switch']) && $_POST['kill_switch'] === '1';
		FeatureManager::set_kill_switch($kill_switch);

		// Log kill switch change
		if ($old_kill_switch !== $kill_switch) {
			ActivityLogger::log_change(
				'kill_switch',
				$kill_switch ? 'kill_switch_on' : 'kill_switch_off',
				$old_kill_switch,
				$kill_switch
			);
		}

		// Update each feature flag
		$feature_names = [
			FeatureManager::ANALYTICS_TRACKING,
			FeatureManager::BENCHMARKING_DASHBOARD,
			FeatureManager::AB_TESTING,
			FeatureManager::PLUGIN_COMPATIBILITY,
		];

		// Get all current flags
		$flags = FeatureManager::get_flags();
		$flags_updated = false;

		foreach ($feature_names as $flag_name) {
			$flag = FeatureManager::get_flag($flag_name);
			if (!$flag) {
				continue;
			}

			// Track old values for logging
			$old_enabled = $flag['enabled_globally'] ?? false;
			$old_rollout = $flag['rollout_percentage'] ?? 0;

			// Get form data
			$enabled_globally = isset($_POST["flag_{$flag_name}_enabled"]) && $_POST["flag_{$flag_name}_enabled"] === '1';
			$rollout_percentage = isset($_POST["flag_{$flag_name}_rollout"]) ? intval($_POST["flag_{$flag_name}_rollout"]) : 0;
			$rollout_percentage = max(0, min(100, $rollout_percentage)); // Clamp to 0-100

			// Validate dependencies
			if ($enabled_globally) {
				$current_enabled = [];
				foreach ($feature_names as $check_flag) {
					if (isset($_POST["flag_{$check_flag}_enabled"]) && $_POST["flag_{$check_flag}_enabled"] === '1') {
						$current_enabled[] = $check_flag;
					}
				}

				$validation_error = FeatureManager::validate_dependencies($flag_name, $current_enabled);
				if ($validation_error) {
					$errors[] = $validation_error;
					$success = false;
					continue;
				}
			}

			// Update flag configuration in memory
			$flags[$flag_name]['enabled_globally'] = $enabled_globally;
			$flags[$flag_name]['rollout_percentage'] = $rollout_percentage;
			$flags_updated = true;

			// Log changes
			if ($old_enabled !== $enabled_globally) {
				ActivityLogger::log_change(
					$flag_name,
					$enabled_globally ? 'enabled' : 'disabled',
					$old_enabled,
					$enabled_globally
				);
			}

			if ($old_rollout !== $rollout_percentage) {
				ActivityLogger::log_change(
					$flag_name,
					'rollout_changed',
					$old_rollout,
					$rollout_percentage
				);
			}
		}

		// Save all flags at once (single database write)
		if ($flags_updated) {
			if (!\update_option(FeatureManager::OPTION_KEY, $flags)) {
				// Check if it's because the value didn't change
				$existing_flags = \get_option(FeatureManager::OPTION_KEY, []);
				if ($existing_flags !== $flags) {
					$errors[] = \__('Failed to save feature flags to database.', 'flx-woo');
					$success = false;
				}
				// If values are identical, treat as success (no change needed)
			}
		}

		// Show admin notice
		if ($success && empty($errors)) {
			\add_settings_error(
				'flx_woo_feature_flags',
				'settings_updated',
				\__('Feature flags saved successfully.', 'flx-woo'),
				'success'
			);
		} else {
			foreach ($errors as $error) {
				\add_settings_error(
					'flx_woo_feature_flags',
					'settings_error',
					$error,
					'error'
				);
			}
		}
	}

	/**
	 * Get feature flag status (enabled/disabled)
	 *
	 * @param string $flag_name Feature flag name
	 * @return bool True if enabled
	 */
	public function is_feature_enabled($flag_name) {
		$store_id = FeatureManager::get_store_id();
		return FeatureManager::is_enabled($flag_name, $store_id);
	}

	/**
	 * Get dependent features for a given feature
	 *
	 * @param string $flag_name Feature flag name
	 * @return array Array of dependent feature names
	 */
	public function get_dependent_features($flag_name) {
		return FeatureManager::get_dependent_features($flag_name);
	}

	/**
	 * Get feature overview statistics
	 *
	 * @return array Statistics for dashboard
	 */
	public function get_feature_overview() {
		$flags = FeatureManager::get_flags();
		$enabled = FeatureManager::get_enabled_features();

		$stats = [
			'total_features' => count($flags),
			'enabled_count' => count($enabled),
			'disabled_count' => count($flags) - count($enabled),
			'with_dependencies' => 0,
			'kill_switch_active' => FeatureManager::is_kill_switch_active(),
		];

		// Count features with dependencies
		foreach ($flags as $flag) {
			if (!empty($flag['dependencies'])) {
				$stats['with_dependencies']++;
			}
		}

		return $stats;
	}

	/**
	 * Get feature status list for overview section
	 *
	 * @return array Feature status items
	 */
	public function get_feature_status_list() {
		$flags = FeatureManager::get_flags();
		$enabled_features = FeatureManager::get_enabled_features();
		$status_list = [];

		foreach ($flags as $flag_name => $flag_config) {
			$is_enabled = in_array($flag_name, $enabled_features, true);
			$has_deps = !empty($flag_config['dependencies']);

			$status_list[] = [
				'name' => $flag_config['display_name'],
				'enabled' => $is_enabled,
				'has_dependencies' => $has_deps,
				'dependency_names' => $has_deps ? $this->get_dependency_names($flag_config['dependencies']) : [],
			];
		}

		return $status_list;
	}

	/**
	 * Get dependency tree structure (hierarchical)
	 *
	 * @return array Tree structure with root features and nested children
	 */
	public function get_dependency_tree() {
		$flags = FeatureManager::get_flags();
		$enabled_features = FeatureManager::get_enabled_features();
		$tree = [];

		// Find root features (no dependencies)
		foreach ($flags as $flag_name => $flag_config) {
			if (empty($flag_config['dependencies'])) {
				$tree[$flag_name] = [
					'display_name' => $flag_config['display_name'],
					'enabled' => in_array($flag_name, $enabled_features, true),
					'children' => $this->get_dependent_tree($flag_name, $enabled_features),
				];
			}
		}

		return $tree;
	}

	/**
	 * Get feature health status
	 *
	 * @return array Health status for each feature
	 */
	public function get_feature_health() {
		$flags = FeatureManager::get_flags();
		$enabled_features = FeatureManager::get_enabled_features();
		$health = [];

		foreach ($flags as $flag_name => $flag_config) {
			$is_enabled = in_array($flag_name, $enabled_features, true);
			$issues = [];

			// Check dependencies
			if (!empty($flag_config['dependencies'])) {
				foreach ($flag_config['dependencies'] as $dep) {
					if (!in_array($dep, $enabled_features, true)) {
						if ($is_enabled) {
							$issues[] = 'Missing dependency: ' . ($flags[$dep]['display_name'] ?? $dep);
						}
					}
				}
			}

			// Determine status
			if ($is_enabled && empty($issues)) {
				$status = 'healthy';
				$status_label = 'Healthy';
			} elseif ($is_enabled && !empty($issues)) {
				$status = 'warning';
				$status_label = 'Warning';
			} else {
				$status = 'ready';
				$status_label = 'Ready';
			}

			$health[$flag_name] = [
				'display_name' => $flag_config['display_name'],
				'status' => $status,
				'status_label' => $status_label,
				'enabled' => $is_enabled,
				'issues' => !empty($issues) ? $issues : ['None'],
			];
		}

		return $health;
	}

	/**
	 * Get human-readable dependency names
	 *
	 * @param array $dependency_keys Feature flag keys
	 * @return array Display names
	 */
	private function get_dependency_names($dependency_keys) {
		$flags = FeatureManager::get_flags();
		return array_map(function($key) use ($flags) {
			return $flags[$key]['display_name'] ?? $key;
		}, $dependency_keys);
	}

	/**
	 * Get recent activity for display
	 *
	 * @param int $limit Maximum number of entries to return
	 * @return array Recent activity entries
	 */
	public function get_recent_activity($limit = 10) {
		return ActivityLogger::get_recent_activity($limit);
	}

	/**
	 * Get activity statistics
	 *
	 * @return array Activity statistics
	 */
	public function get_activity_stats() {
		return ActivityLogger::get_statistics();
	}

	/**
	 * Get dependent tree recursively
	 *
	 * @param string $parent_flag Parent feature flag name
	 * @param array $enabled_features Currently enabled features
	 * @return array Dependent features
	 */
	private function get_dependent_tree($parent_flag, $enabled_features) {
		$dependents = FeatureManager::get_dependent_features($parent_flag);
		$flags = FeatureManager::get_flags();
		$tree = [];

		foreach ($dependents as $dependent) {
			$tree[$dependent] = [
				'display_name' => $flags[$dependent]['display_name'] ?? $dependent,
				'enabled' => in_array($dependent, $enabled_features, true),
				'children' => $this->get_dependent_tree($dependent, $enabled_features),
			];
		}

		return $tree;
	}

	/**
	 * Save retention settings from form submission
	 */
	private function save_retention_settings() {
		// Validate and sanitize retention period
		$retention_period = isset($_POST['retention_period']) ? \absint($_POST['retention_period']) : 90;

		// Save retention period
		$saved = RetentionManager::set_retention_period($retention_period);

		if ($saved) {
			// Schedule cron if not already scheduled
			RetentionManager::schedule_cleanup_cron();

			\add_settings_error(
				'flx_woo_feature_flags',
				'retention_saved',
				\__('Retention settings saved successfully. Automated cleanup scheduled.', 'flx-woo'),
				'success'
			);
		} else {
			\add_settings_error(
				'flx_woo_feature_flags',
				'retention_error',
				\__('Failed to save retention settings. Please try again.', 'flx-woo'),
				'error'
			);
		}
	}

	/**
	 * AJAX handler for manual cleanup
	 */
	public function ajax_manual_cleanup() {
		// Verify nonce
		\check_ajax_referer('flx_manual_cleanup', 'nonce');

		// Check permissions
		if (!\current_user_can('manage_woocommerce')) {
			\wp_send_json_error([
				'message' => \__('You do not have permission to perform this action.', 'flx-woo'),
			]);
			return;
		}

		// Run cleanup
		$result = RetentionManager::cleanup_old_records();

		// Send response
		\wp_send_json_success([
			'message' => $result['message'],
			'deleted_count' => $result['deleted_count'],
			'retention_days' => $result['retention_days'],
			'cutoff_date' => $result['cutoff_date'],
		]);
	}
}
