<?php
/**
 * Plugin Compatibility Admin Page
 *
 * Displays compatibility information for installed plugins.
 *
 * @package FlxWoo
 * @version 2.4.0
 */

namespace FlxWoo\Admin;

use FlxWoo\Compatibility\Reporter;

/**
 * Class CompatibilityPage
 *
 * Manages the plugin compatibility admin page.
 */
class CompatibilityPage {
    /**
     * Reporter instance
     *
     * @var Reporter
     */
    private $reporter;

    /**
     * Constructor
     */
    public function __construct() {
        $this->reporter = new Reporter();
    }

    /**
     * Render the compatibility page
     */
    public function render(): void {
        // Handle form submissions
        $this->handle_actions();

        // Get compatibility data
        $compatibility_data = $this->reporter->get_installed_plugins_compatibility();

        // Include the view
        include __DIR__ . '/views/compatibility.php';
    }

    /**
     * Handle AJAX requests
     */
    public function handle_ajax(): void {
        // Verify nonce
        check_ajax_referer( 'flx_woo_compatibility', 'nonce' );

        // Verify user capabilities
        if ( ! current_user_can( 'manage_woocommerce' ) ) {
            wp_send_json_error( [ 'message' => 'Insufficient permissions' ] );
        }

        $action = isset( $_POST['compatibility_action'] ) ? sanitize_text_field( $_POST['compatibility_action'] ) : '';

        switch ( $action ) {
            case 'submit_report':
                $this->handle_submit_report();
                break;

            case 'refresh_data':
                $this->handle_refresh_data();
                break;

            default:
                wp_send_json_error( [ 'message' => 'Invalid action' ] );
        }
    }

    /**
     * Handle submit report action
     */
    private function handle_submit_report(): void {
        $result = $this->reporter->submit_compatibility_report();

        if ( is_wp_error( $result ) ) {
            wp_send_json_error( [
                'message' => $result->get_error_message(),
            ] );
        }

        wp_send_json_success( [
            'message'           => 'Compatibility report submitted successfully',
            'reports_processed' => $result['reports_processed'] ?? 0,
        ] );
    }

    /**
     * Handle refresh data action
     */
    private function handle_refresh_data(): void {
        $compatibility_data = $this->reporter->get_installed_plugins_compatibility();

        wp_send_json_success( [
            'message' => 'Compatibility data refreshed',
            'data'    => $compatibility_data,
        ] );
    }

    /**
     * Handle form actions
     */
    private function handle_actions(): void {
        if ( ! isset( $_POST['flx_woo_compatibility_action'] ) ) {
            return;
        }

        // Verify nonce
        if ( ! isset( $_POST['_wpnonce'] ) || ! wp_verify_nonce( $_POST['_wpnonce'], 'flx_woo_compatibility' ) ) {
            add_settings_error(
                'flx_woo_compatibility',
                'invalid_nonce',
                'Security check failed',
                'error'
            );
            return;
        }

        // Verify user capabilities
        if ( ! current_user_can( 'manage_woocommerce' ) ) {
            add_settings_error(
                'flx_woo_compatibility',
                'insufficient_permissions',
                'You do not have permission to perform this action',
                'error'
            );
            return;
        }

        $action = sanitize_text_field( $_POST['flx_woo_compatibility_action'] );

        switch ( $action ) {
            case 'submit_report':
                $result = $this->reporter->submit_compatibility_report();

                if ( is_wp_error( $result ) ) {
                    add_settings_error(
                        'flx_woo_compatibility',
                        'submit_error',
                        'Error submitting report: ' . $result->get_error_message(),
                        'error'
                    );
                } else {
                    add_settings_error(
                        'flx_woo_compatibility',
                        'submit_success',
                        sprintf(
                            'Successfully submitted compatibility report for %d plugin(s)',
                            $result['reports_processed'] ?? 0
                        ),
                        'success'
                    );
                }
                break;
        }
    }

    /**
     * Get score color class
     *
     * @param int $score Compatibility score (0-100)
     * @return string CSS class
     */
    public static function get_score_color_class( int $score ): string {
        if ( $score >= 90 ) {
            return 'score-verified';
        }
        if ( $score >= 70 ) {
            return 'score-compatible';
        }
        if ( $score >= 40 ) {
            return 'score-issues';
        }
        return 'score-incompatible';
    }

    /**
     * Get score badge text
     *
     * @param int $score Compatibility score (0-100)
     * @return string Badge text
     */
    public static function get_score_badge( int $score ): string {
        if ( $score >= 90 ) {
            return '✓ Verified Compatible';
        }
        if ( $score >= 70 ) {
            return '✓ Compatible';
        }
        if ( $score >= 40 ) {
            return '⚠ Minor Issues';
        }
        return '✗ Incompatible';
    }

    /**
     * Get status icon
     *
     * @param string $status Status: verified|compatible|issues|incompatible|unknown
     * @return string HTML icon
     */
    public static function get_status_icon( string $status ): string {
        $icons = [
            'verified'     => '<span class="dashicons dashicons-yes-alt" style="color: #10b981;"></span>',
            'compatible'   => '<span class="dashicons dashicons-yes" style="color: #84cc16;"></span>',
            'issues'       => '<span class="dashicons dashicons-warning" style="color: #f59e0b;"></span>',
            'incompatible' => '<span class="dashicons dashicons-no-alt" style="color: #ef4444;"></span>',
            'unknown'      => '<span class="dashicons dashicons-editor-help" style="color: #94a3b8;"></span>',
        ];

        return $icons[ $status ] ?? $icons['unknown'];
    }
}
