<?php
/**
 * Benchmarking Dashboard Admin Page
 *
 * Displays store performance comparison to industry benchmarks.
 *
 * @package FlxWoo\Admin
 * @since 2.4.0
 */

namespace FlxWoo\Admin;

use FlxWoo\FeatureFlags\FeatureManager;

if (!defined('ABSPATH')) {
    exit;
}

class BenchmarkingPage {

    /**
     * Period options (days)
     */
    private const VALID_PERIODS = [7, 30, 90];
    private const DEFAULT_PERIOD = 30;

    /**
     * Constructor - Register AJAX handlers
     */
    public function __construct() {
        add_action('wp_ajax_flx_get_benchmark_data', [$this, 'ajax_get_benchmark_data']);
    }

    /**
     * Render the benchmarking dashboard page
     *
     * @return void
     */
    public function render(): void {
        // Permission check
        if (!current_user_can('manage_woocommerce')) {
            wp_die(esc_html__('You do not have sufficient permissions to access this page.', 'flx-woo'));
        }

        // Check feature flag
        $store_id = FeatureManager::get_store_id();
        if (!FeatureManager::is_enabled(FeatureManager::BENCHMARKING_DASHBOARD, $store_id)) {
            $this->render_feature_disabled_notice();
            return;
        }

        // Get initial benchmark data (30 days by default)
        $benchmark_data = $this->get_benchmark_data(self::DEFAULT_PERIOD);

        // Prepare data for view
        $overview_stats = $this->format_overview_stats($benchmark_data);
        $comparison_data = $this->format_comparison_data($benchmark_data);
        $insights = $this->extract_insights($benchmark_data);
        $period_options = self::VALID_PERIODS;

        // Include view template
        include __DIR__ . '/views/benchmarking.php';
    }

    /**
     * Get benchmark data from Next.js API
     *
     * @param int $period_days Period in days (7, 30, or 90)
     * @return array|null Benchmark data or null on error
     */
    private function get_benchmark_data(int $period_days): ?array {
        // Validate period
        if (!in_array($period_days, self::VALID_PERIODS, true)) {
            $period_days = self::DEFAULT_PERIOD;
        }

        // Get renderer URL from constant
        if (!defined('FLX_WOO_RENDERER_URL')) {
            return null;
        }
        $renderer_url = FLX_WOO_RENDERER_URL;

        // Get store ID
        $store_id = FeatureManager::get_store_id();

        // Build API URL
        $api_url = trailingslashit($renderer_url) . 'api/v1/analytics/benchmark';
        $api_url = add_query_arg([
            'store_id' => $store_id,
            'days' => $period_days
        ], $api_url);

        // Make HTTP request
        $response = wp_remote_get($api_url, [
            'timeout' => 10,
            'headers' => [
                'Accept' => 'application/json'
            ]
        ]);

        // Handle errors
        if (is_wp_error($response)) {
            error_log('FlxWoo Benchmarking API error: ' . $response->get_error_message());
            return null;
        }

        $status_code = wp_remote_retrieve_response_code($response);
        if ($status_code !== 200) {
            error_log('FlxWoo Benchmarking API returned status: ' . $status_code);
            return null;
        }

        // Parse JSON response
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);

        if (!isset($data['success']) || $data['success'] !== true) {
            error_log('FlxWoo Benchmarking API returned error');
            return null;
        }

        return $data['data'] ?? null;
    }

    /**
     * Format overview stats for display
     *
     * @param array|null $benchmark_data Benchmark data from API
     * @return array Overview statistics
     */
    private function format_overview_stats(?array $benchmark_data): array {
        if (!$benchmark_data || !isset($benchmark_data['store'])) {
            return [
                'conversion_rate' => 'N/A',
                'cart_value' => 'N/A',
                'page_load' => 'N/A',
                'has_data' => false
            ];
        }

        $store = $benchmark_data['store'];

        return [
            'conversion_rate' => number_format($store['conversion_rate_percent'] ?? 0, 1) . '%',
            'cart_value' => '$' . number_format(($store['avg_cart_value_cents'] ?? 0) / 100, 2),
            'page_load' => number_format(($store['avg_page_load_ms'] ?? 0) / 1000, 2) . 's',
            'total_revenue' => '$' . number_format(($store['total_revenue_cents'] ?? 0) / 100, 2),
            'checkouts_completed' => $store['checkouts_completed'] ?? 0,
            'has_data' => true
        ];
    }

    /**
     * Format comparison data for charts
     *
     * @param array|null $benchmark_data Benchmark data from API
     * @return array Comparison data for visualization
     */
    private function format_comparison_data(?array $benchmark_data): array {
        if (!$benchmark_data || !isset($benchmark_data['store'], $benchmark_data['industry'], $benchmark_data['comparison'])) {
            return ['has_data' => false];
        }

        $store = $benchmark_data['store'];
        $industry = $benchmark_data['industry'];
        $comparison = $benchmark_data['comparison'];

        return [
            'conversion_rate' => [
                'store' => $store['conversion_rate_percent'] ?? 0,
                'industry' => $industry['avg_conversion_rate_percent'] ?? 0,
                'diff' => $comparison['conversion_rate_diff_percent'] ?? 0,
                'percentile' => $comparison['conversion_rate_percentile'] ?? 0,
                'better' => ($comparison['conversion_rate_diff_percent'] ?? 0) > 0
            ],
            'cart_value' => [
                'store' => ($store['avg_cart_value_cents'] ?? 0) / 100,
                'industry' => ($industry['avg_cart_value_cents'] ?? 0) / 100,
                'diff_percent' => $comparison['cart_value_diff_percent'] ?? 0,
                'percentile' => $comparison['cart_value_percentile'] ?? 0,
                'better' => ($comparison['cart_value_diff_percent'] ?? 0) > 0
            ],
            'page_load' => [
                'store' => ($store['avg_page_load_ms'] ?? 0) / 1000,
                'industry' => ($industry['avg_page_load_ms'] ?? 0) / 1000,
                'diff_percent' => $comparison['page_load_diff_percent'] ?? 0,
                'percentile' => $comparison['page_load_percentile'] ?? 0,
                'better' => ($comparison['page_load_diff_percent'] ?? 0) < 0  // Lower is better for page load
            ],
            'has_data' => true
        ];
    }

    /**
     * Extract insights from benchmark data
     *
     * @param array|null $benchmark_data Benchmark data from API
     * @return array Array of insight strings
     */
    private function extract_insights(?array $benchmark_data): array {
        if (!$benchmark_data || !isset($benchmark_data['insights'])) {
            return [];
        }

        return $benchmark_data['insights'];
    }

    /**
     * AJAX handler for getting benchmark data with different periods
     *
     * @return void
     */
    public function ajax_get_benchmark_data(): void {
        // Security check
        check_ajax_referer('flx_benchmark_nonce', 'nonce');

        // Permission check
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error(['message' => __('Permission denied', 'flx-woo')]);
        }

        // Get period from request
        $period = isset($_POST['period']) ? intval($_POST['period']) : self::DEFAULT_PERIOD;

        // Fetch benchmark data
        $benchmark_data = $this->get_benchmark_data($period);

        if ($benchmark_data === null) {
            wp_send_json_error(['message' => __('Failed to fetch benchmark data', 'flx-woo')]);
        }

        // Format data for response
        $response = [
            'overview' => $this->format_overview_stats($benchmark_data),
            'comparison' => $this->format_comparison_data($benchmark_data),
            'insights' => $this->extract_insights($benchmark_data)
        ];

        wp_send_json_success($response);
    }

    /**
     * Render feature disabled notice
     *
     * @return void
     */
    private function render_feature_disabled_notice(): void {
        ?>
        <div class="wrap">
            <h1><?php esc_html_e('Benchmarking', 'flx-woo'); ?></h1>
            <div class="notice notice-warning">
                <p>
                    <?php esc_html_e('The Benchmarking feature is currently disabled. Please enable it in the Feature Flags settings.', 'flx-woo'); ?>
                </p>
                <p>
                    <a href="<?php echo esc_url(admin_url('admin.php?page=flx-woo-feature-flags')); ?>" class="button button-primary">
                        <?php esc_html_e('Go to Feature Flags', 'flx-woo'); ?>
                    </a>
                </p>
            </div>
        </div>
        <?php
    }
}
