<?php
/**
 * Admin Hooks
 *
 * Registers WordPress admin menu, settings, and hooks for FlxWoo admin interface.
 *
 * @package FlxWoo
 * @since 2.1.0
 */

namespace FlxWoo\Admin;

if (!defined('ABSPATH')) {
    exit;
}

class AdminHooks {
    /**
     * @var SettingsManager
     */
    private $settings_manager;

    /**
     * @var PerformanceDashboard
     */
    private $performance_dashboard;

    /**
     * @var FeatureFlagsPage
     */
    private $feature_flags_page;

    /**
     * @var ActivityAnalyticsPage
     */
    private $activity_analytics_page;

    /**
     * @var BenchmarkingPage
     */
    private $benchmarking_page;

    /**
     * @var ABTestingPage
     */
    private $ab_testing_page;

    /**
     * @var CompatibilityPage
     */
    private $compatibility_page;

    /**
     * Constructor
     */
    public function __construct() {
        $this->settings_manager = new SettingsManager();
        $this->performance_dashboard = new PerformanceDashboard();
        $this->feature_flags_page = new FeatureFlagsPage();
        $this->activity_analytics_page = new ActivityAnalyticsPage();
        $this->benchmarking_page = new BenchmarkingPage();
        $this->ab_testing_page = new ABTestingPage();
        $this->compatibility_page = new CompatibilityPage();

        // Register AJAX handlers for analytics
        add_action('wp_ajax_flx_get_timeline_data', [$this->activity_analytics_page, 'ajax_get_timeline_data']);
        add_action('wp_ajax_flx_get_feature_breakdown', [$this->activity_analytics_page, 'ajax_get_feature_breakdown']);
        add_action('wp_ajax_flx_get_user_activity', [$this->activity_analytics_page, 'ajax_get_user_activity']);
        add_action('wp_ajax_flx_get_action_distribution', [$this->activity_analytics_page, 'ajax_get_action_distribution']);
        add_action('wp_ajax_flx_export_csv', [$this->activity_analytics_page, 'ajax_export_csv']);
        add_action('wp_ajax_flx_get_filtered_data', [$this->activity_analytics_page, 'ajax_get_filtered_data']);

        // Register AJAX handler for manual cleanup
        add_action('wp_ajax_flx_manual_cleanup', [$this->feature_flags_page, 'ajax_manual_cleanup']);

        // Register AJAX handler for compatibility page
        add_action('wp_ajax_flx_woo_compatibility', [$this->compatibility_page, 'handle_ajax']);

        // Register cron action for automated cleanup
        add_action(\FlxWoo\FeatureFlags\RetentionManager::CRON_HOOK, [\FlxWoo\FeatureFlags\RetentionManager::class, 'cleanup_old_records']);
    }

    /**
     * Initialize hooks
     */
    public function init() {
        // Add admin menu
        add_action('admin_menu', [$this, 'add_admin_menu']);

        // Register settings
        add_action('admin_init', [$this, 'register_settings']);

        // Enqueue admin assets
        add_action('admin_enqueue_scripts', [$this, 'enqueue_admin_assets']);

        // Add settings link to plugins page
        add_filter('plugin_action_links_flx-woo/flx-woo.php', [$this, 'add_settings_link']);
    }

    /**
     * Add admin menu page
     */
    public function add_admin_menu() {
        // Add top-level FlxWoo menu (Settings Dashboard)
        add_menu_page(
            __('FlxWoo Settings', 'flx-woo'),              // Page title
            __('FlxWoo', 'flx-woo'),                       // Menu title
            'manage_woocommerce',                           // Capability
            'flx-woo-settings',                             // Menu slug
            [$this->performance_dashboard, 'render'],      // Callback
            'dashicons-admin-settings',                     // Icon
            56                                              // Position (after WooCommerce at 55)
        );

        // Override the automatic first submenu item to rename it to "Settings"
        add_submenu_page(
            'flx-woo-settings',                             // Parent slug
            __('FlxWoo Settings', 'flx-woo'),              // Page title
            __('Settings', 'flx-woo'),                     // Menu title (changed from "FlxWoo")
            'manage_woocommerce',                           // Capability
            'flx-woo-settings',                             // Menu slug (same as parent to override)
            [$this->performance_dashboard, 'render']       // Callback
        );

        // Add Feature Flags submenu (v2.3.0)
        add_submenu_page(
            'flx-woo-settings',                             // Parent slug
            __('Feature Flags', 'flx-woo'),                // Page title
            __('Feature Flags', 'flx-woo'),                // Menu title
            'manage_woocommerce',                           // Capability
            'flx-woo-feature-flags',                        // Menu slug
            [$this->feature_flags_page, 'render']          // Callback
        );

        // Add Activity Analytics submenu (v2.4.0)
        add_submenu_page(
            'flx-woo-settings',                             // Parent slug
            __('Analytics', 'flx-woo'),                    // Page title
            __('Analytics', 'flx-woo'),                    // Menu title
            'manage_woocommerce',                           // Capability
            'flx-woo-analytics',                            // Menu slug
            [$this->activity_analytics_page, 'render']     // Callback
        );

        // Add Benchmarking Dashboard submenu (v2.4.0)
        add_submenu_page(
            'flx-woo-settings',                             // Parent slug
            __('Benchmarking', 'flx-woo'),                 // Page title
            __('Benchmarking', 'flx-woo'),                 // Menu title
            'manage_woocommerce',                           // Capability
            'flx-woo-benchmarking',                         // Menu slug
            [$this->benchmarking_page, 'render']           // Callback
        );

        // Add A/B Testing submenu (v2.5.0)
        add_submenu_page(
            'flx-woo-settings',                             // Parent slug
            __('A/B Testing', 'flx-woo'),                  // Page title
            __('A/B Testing', 'flx-woo'),                  // Menu title
            'manage_woocommerce',                           // Capability
            'flx-woo-ab-testing',                           // Menu slug
            [$this->ab_testing_page, 'render']             // Callback
        );

        // Add Plugin Compatibility submenu (v2.4.0)
        add_submenu_page(
            'flx-woo-settings',                             // Parent slug
            __('Compatibility', 'flx-woo'),                // Page title
            __('Compatibility', 'flx-woo'),                // Menu title
            'manage_woocommerce',                           // Capability
            'flx-woo-compatibility',                        // Menu slug
            [$this->compatibility_page, 'render']          // Callback
        );
    }

    /**
     * Register settings with WordPress Settings API
     */
    public function register_settings() {
        // Register setting group
        register_setting(
            'flx_woo_settings_group',           // Option group
            SettingsManager::OPTION_NAME,        // Option name
            [
                'type' => 'array',
                'description' => __('FlxWoo plugin settings', 'flx-woo'),
                'sanitize_callback' => [$this, 'sanitize_settings'],
                'default' => $this->settings_manager->get_default_settings(),
            ]
        );

        // =====================================================================
        // Renderer Configuration Section - REMOVED for SaaS Model
        // =====================================================================
        // In the SaaS model, renderer configuration (URL, timeout, version)
        // is hardcoded in Constants.php and managed by the SaaS provider.
        // Users cannot modify these values for security and consistency.
        //
        // Renderer Status monitoring is available in the Performance Dashboard
        // via PerformanceDashboard::get_renderer_status().
        // =====================================================================

        // NOTE: The following fields have been removed from the UI:
        // - renderer_url (hardcoded in Constants.php)
        // - renderer_timeout (hardcoded in Constants.php)
        // - renderer_version (hardcoded in Constants.php)

        // =====================================================================
        // Cache Settings Section - REMOVED (Not Applicable to FlxWoo)
        // =====================================================================
        // FlxWoo renders Cart, Checkout, and Thank You pages - all of which are:
        // - User-specific (different cart per user)
        // - Dynamic (prices, inventory, shipping rates change in real-time)
        // - Transaction-specific (unique order numbers)
        //
        // Caching these pages would show stale/incorrect data and break e-commerce
        // functionality. Therefore, caching is NOT implemented in FlxWoo.
        // =====================================================================

        // =====================================================================
        // Debug & Development Section - REMOVED
        // =====================================================================
        // Debug settings removed because:
        // 1. "Enable Debug Logging" - Not implemented, does nothing
        // 2. "Development Mode" - Replaced with automatic localhost detection
        //
        // HTTP is now automatically allowed for localhost (127.0.0.1, ::1)
        // and HTTPS is required for all other domains (security by default).
        // =====================================================================
    }

    /**
     * Render text field
     */
    public function render_text_field($args) {
        $settings = $this->settings_manager->get_all_settings();
        $value = $settings[$args['field_name']] ?? '';

        echo '<input type="text" ';
        echo 'id="' . esc_attr($args['label_for']) . '" ';
        echo 'name="' . esc_attr(SettingsManager::OPTION_NAME . '[' . $args['field_name'] . ']') . '" ';
        echo 'value="' . esc_attr($value) . '" ';
        echo 'class="regular-text" ';

        if (isset($args['placeholder'])) {
            echo 'placeholder="' . esc_attr($args['placeholder']) . '" ';
        }

        echo '/>';

        if (isset($args['description'])) {
            echo '<p class="description">' . esc_html($args['description']) . '</p>';
        }
    }

    /**
     * Render number field
     */
    public function render_number_field($args) {
        $settings = $this->settings_manager->get_all_settings();
        $value = $settings[$args['field_name']] ?? '';

        echo '<input type="number" ';
        echo 'id="' . esc_attr($args['label_for']) . '" ';
        echo 'name="' . esc_attr(SettingsManager::OPTION_NAME . '[' . $args['field_name'] . ']') . '" ';
        echo 'value="' . esc_attr($value) . '" ';
        echo 'class="small-text" ';

        if (isset($args['min'])) {
            echo 'min="' . esc_attr($args['min']) . '" ';
        }

        if (isset($args['max'])) {
            echo 'max="' . esc_attr($args['max']) . '" ';
        }

        echo '/>';

        if (isset($args['description'])) {
            echo '<p class="description">' . esc_html($args['description']) . '</p>';
        }
    }

    /**
     * Render select field
     */
    public function render_select_field($args) {
        $settings = $this->settings_manager->get_all_settings();
        $value = $settings[$args['field_name']] ?? '';

        echo '<select ';
        echo 'id="' . esc_attr($args['label_for']) . '" ';
        echo 'name="' . esc_attr(SettingsManager::OPTION_NAME . '[' . $args['field_name'] . ']') . '">';

        foreach ($args['options'] as $option_value => $option_label) {
            echo '<option value="' . esc_attr($option_value) . '" ';
            selected($value, $option_value);
            echo '>' . esc_html($option_label) . '</option>';
        }

        echo '</select>';

        if (isset($args['description'])) {
            echo '<p class="description">' . esc_html($args['description']) . '</p>';
        }
    }

    /**
     * Render checkbox field
     */
    public function render_checkbox_field($args) {
        $settings = $this->settings_manager->get_all_settings();
        $value = isset($settings[$args['field_name']]) ? (bool) $settings[$args['field_name']] : false;

        echo '<label for="' . esc_attr($args['label_for']) . '">';
        echo '<input type="checkbox" ';
        echo 'id="' . esc_attr($args['label_for']) . '" ';
        echo 'name="' . esc_attr(SettingsManager::OPTION_NAME . '[' . $args['field_name'] . ']') . '" ';
        echo 'value="1" ';
        checked($value, true);
        echo '/> ';

        if (isset($args['description'])) {
            echo esc_html($args['description']);
        }

        echo '</label>';
    }

    /**
     * Sanitize settings before saving
     *
     * @param array $input Raw input from form
     * @return array Sanitized settings
     */
    public function sanitize_settings($input) {
        // Let SettingsManager handle validation
        $result = $this->settings_manager->update_all_settings($input);

        if (is_array($result) && !empty($result)) {
            // Validation errors occurred
            foreach ($result as $field => $error) {
                add_settings_error(
                    SettingsManager::OPTION_NAME,
                    'flx_woo_' . $field . '_error',
                    $error,
                    'error'
                );
            }

            // Return current settings (don't save invalid data)
            return $this->settings_manager->get_all_settings();
        }

        // Success
        add_settings_error(
            SettingsManager::OPTION_NAME,
            'flx_woo_settings_updated',
            __('Settings saved successfully.', 'flx-woo'),
            'success'
        );

        return $this->settings_manager->get_all_settings();
    }

    /**
     * Enqueue admin assets
     *
     * @param string $hook Current admin page hook
     */
    public function enqueue_admin_assets($hook) {
        // Enqueue assets on settings dashboard page (top-level menu)
        if ($hook === 'toplevel_page_flx-woo-settings') {
            // Enqueue dashboard CSS
            wp_enqueue_style(
                'flx-woo-settings-dashboard',
                plugins_url('src/Admin/assets/css/performance-dashboard.css', dirname(dirname(__FILE__))),
                [],
                '2.3.2' // Version updated - inline documentation labels
            );

            // Enqueue jQuery explicitly (required for AJAX)
            wp_enqueue_script('jquery');

            // Enqueue dashboard JavaScript
            wp_enqueue_script(
                'flx-woo-settings-dashboard',
                plugins_url('src/Admin/assets/js/performance-dashboard.js', dirname(dirname(__FILE__))),
                ['jquery'],
                '2.3.0', // Version updated for simplified dashboard
                true
            );

            // Pass nonces and AJAX URL to JavaScript
            wp_localize_script(
                'flx-woo-settings-dashboard',
                'flxDashboard',
                [
                    'ajaxurl' => admin_url('admin-ajax.php'),
                    'nonces' => [
                        'dashboard' => wp_create_nonce('flx_dashboard_nonce'),
                        'settings' => wp_create_nonce('flx_save_settings'),
                    ],
                ]
            );
        }

        // Enqueue assets on Feature Flags page (v2.4.0)
        // Check both hook and page parameter for reliability across different WordPress configurations
        $is_feature_flags_page = ($hook === 'flx-woo-settings_page_flx-woo-feature-flags')
            || ($hook === 'flxwoo_page_flx-woo-feature-flags')
            || (strpos($hook, 'flx-woo-feature-flags') !== false)
            || (isset($_GET['page']) && $_GET['page'] === 'flx-woo-feature-flags');

        if ($is_feature_flags_page) {
            // Enqueue base dashboard CSS (for .flx-dashboard-section and .flx-info-grid)
            wp_enqueue_style(
                'flx-woo-settings-dashboard',
                plugins_url('src/Admin/assets/css/performance-dashboard.css', dirname(dirname(__FILE__))),
                [],
                '2.3.2'
            );

            // Enqueue Feature Flags CSS
            wp_enqueue_style(
                'flx-woo-feature-flags',
                plugins_url('src/Admin/assets/css/feature-flags.css', dirname(dirname(__FILE__))),
                ['flx-woo-settings-dashboard'],
                '2.4.0'
            );

            // Enqueue jQuery explicitly
            wp_enqueue_script('jquery');

            // Enqueue Feature Flags JavaScript
            wp_enqueue_script(
                'flx-woo-feature-flags',
                plugins_url('src/Admin/assets/js/feature-flags.js', dirname(dirname(__FILE__))),
                ['jquery'],
                '2.4.0',
                true
            );

            // Pass dependency data to JavaScript for validation
            $flags = \FlxWoo\FeatureFlags\FeatureManager::get_flags();
            $dependencies = [];
            foreach ($flags as $flag_name => $flag_config) {
                if (!empty($flag_config['dependencies'])) {
                    $dependencies[$flag_name] = $flag_config['dependencies'];
                }
            }

            wp_localize_script(
                'flx-woo-feature-flags',
                'flxFeatureDependencies',
                $dependencies
            );

            // Pass AJAX URL and nonce for manual cleanup
            wp_localize_script(
                'flx-woo-feature-flags',
                'flxAjax',
                [
                    'ajaxurl' => admin_url('admin-ajax.php'),
                    'nonce' => wp_create_nonce('flx_manual_cleanup'),
                ]
            );
        }

        // Enqueue assets on Benchmarking Dashboard page (v2.4.0)
        $is_benchmarking_page = ($hook === 'flx-woo-settings_page_flx-woo-benchmarking')
            || (isset($_GET['page']) && $_GET['page'] === 'flx-woo-benchmarking');

        if ($is_benchmarking_page) {
            // Enqueue base dashboard CSS (shared styles)
            wp_enqueue_style(
                'flx-woo-settings-dashboard',
                plugins_url('src/Admin/assets/css/performance-dashboard.css', dirname(dirname(__FILE__))),
                [],
                '2.3.2'
            );

            // Enqueue Benchmarking CSS
            wp_enqueue_style(
                'flx-woo-benchmarking',
                plugins_url('src/Admin/assets/css/benchmarking.css', dirname(dirname(__FILE__))),
                ['flx-woo-settings-dashboard'],
                '2.4.0'
            );

            // Enqueue Chart.js library (shared with Analytics page)
            wp_enqueue_script(
                'chartjs',
                plugins_url('src/Admin/assets/js/chart.min.js', dirname(dirname(__FILE__))),
                [],
                '4.4.0',
                true
            );

            // Enqueue jQuery explicitly
            wp_enqueue_script('jquery');

            // Enqueue Benchmarking JavaScript
            wp_enqueue_script(
                'flx-woo-benchmarking',
                plugins_url('src/Admin/assets/js/benchmarking.js', dirname(dirname(__FILE__))),
                ['jquery', 'chartjs'],
                '2.4.0',
                true
            );

            // Pass AJAX URL and nonce to JavaScript
            wp_localize_script(
                'flx-woo-benchmarking',
                'flxBenchmarkData',
                [
                    'ajaxurl' => admin_url('admin-ajax.php'),
                    'nonce' => wp_create_nonce('flx_benchmark_nonce'),
                ]
            );
        }

        // Enqueue assets on A/B Testing page (v2.5.0)
        $is_ab_testing_page = ($hook === 'flx-woo-settings_page_flx-woo-ab-testing')
            || (isset($_GET['page']) && $_GET['page'] === 'flx-woo-ab-testing');

        if ($is_ab_testing_page) {
            // Enqueue base dashboard CSS (shared styles)
            wp_enqueue_style(
                'flx-woo-settings-dashboard',
                plugins_url('src/Admin/assets/css/performance-dashboard.css', dirname(dirname(__FILE__))),
                [],
                '2.3.2'
            );

            // Enqueue A/B Testing CSS
            wp_enqueue_style(
                'flx-woo-ab-testing',
                plugins_url('src/Admin/assets/css/ab-testing.css', dirname(dirname(__FILE__))),
                ['flx-woo-settings-dashboard'],
                '2.5.0'
            );

            // Enqueue jQuery explicitly
            wp_enqueue_script('jquery');

            // Enqueue A/B Testing JavaScript
            wp_enqueue_script(
                'flx-woo-ab-testing',
                plugins_url('src/Admin/assets/js/ab-testing.js', dirname(dirname(__FILE__))),
                ['jquery'],
                '2.5.0',
                true
            );

            // Pass AJAX URL and nonce to JavaScript (data is passed in view template)
        }

        // Enqueue assets on Activity Analytics page (v2.4.0)
        $is_analytics_page = ($hook === 'flx-woo-settings_page_flx-woo-analytics')
            || (isset($_GET['page']) && $_GET['page'] === 'flx-woo-analytics');

        if ($is_analytics_page) {
            // Enqueue base dashboard CSS (shared styles)
            wp_enqueue_style(
                'flx-woo-settings-dashboard',
                plugins_url('src/Admin/assets/css/performance-dashboard.css', dirname(dirname(__FILE__))),
                [],
                '2.3.2'
            );

            // Enqueue Activity Analytics CSS
            wp_enqueue_style(
                'flx-woo-analytics',
                plugins_url('src/Admin/assets/css/activity-analytics.css', dirname(dirname(__FILE__))),
                ['flx-woo-settings-dashboard'],
                '2.4.0'
            );

            // Enqueue Chart.js library
            wp_enqueue_script(
                'chartjs',
                plugins_url('src/Admin/assets/js/chart.min.js', dirname(dirname(__FILE__))),
                [],
                '4.4.0',
                true
            );

            // Enqueue jQuery explicitly
            wp_enqueue_script('jquery');

            // Enqueue Activity Analytics JavaScript
            wp_enqueue_script(
                'flx-woo-analytics',
                plugins_url('src/Admin/assets/js/activity-analytics.js', dirname(dirname(__FILE__))),
                ['jquery', 'chartjs'],
                '2.4.0',
                true
            );

            // Enqueue Activity Filters JavaScript (v2.3.0 - CSV Export & Advanced Filtering)
            wp_enqueue_script(
                'flx-woo-activity-filters',
                plugins_url('src/Admin/assets/js/activity-filters.js', dirname(dirname(__FILE__))),
                ['jquery', 'flx-woo-analytics'],
                '2.3.0',
                true
            );

            // Localize script with AJAX URL and nonce (shared by both scripts)
            wp_localize_script(
                'flx-woo-analytics',
                'flxAnalytics',
                [
                    'ajaxurl' => admin_url('admin-ajax.php'),
                    'nonce' => wp_create_nonce('flx_analytics_nonce'),
                ]
            );
        }
    }

    /**
     * Add dashboard link to plugins page
     *
     * @param array $links Existing plugin action links
     * @return array Modified links
     */
    public function add_settings_link($links) {
        // Add Settings Dashboard link
        $dashboard_link = sprintf(
            '<a href="%s">%s</a>',
            admin_url('admin.php?page=flx-woo-settings'),
            __('Settings', 'flx-woo')
        );

        array_unshift($links, $dashboard_link);

        return $links;
    }
}
