<?php
/**
 * Activity Analytics Page
 *
 * Admin page for visualizing feature flag activity with interactive charts.
 * Displays timeline, feature breakdown, user activity, and action distribution.
 *
 * @package FlxWoo\Admin
 * @since 2.4.0
 */

namespace FlxWoo\Admin;

use FlxWoo\Database\ActivityRepository;
use FlxWoo\Database\Migrator;
use FlxWoo\FeatureFlags\ActivityLogger;

if (!defined('ABSPATH')) {
	exit;
}

class ActivityAnalyticsPage {
	/**
	 * Render the analytics page
	 */
	public function render(): void {
		// Permission check
		if (!\current_user_can('manage_woocommerce')) {
			\wp_die(\__('You do not have sufficient permissions to access this page.', 'flx-woo'));
		}

		// Check migration status
		$migration_stats = $this->get_migration_status();

		// Get overview statistics
		$overview_stats = $this->get_overview_stats();

		// Check if this is a fresh installation with no data
		$is_empty_state = $this->is_empty_state();

		// Get next aggregation time if scheduled
		$next_aggregation = $this->get_next_data_collection();

		// Load view template
		include __DIR__ . '/views/analytics.php';
	}

	/**
	 * Get overview statistics
	 *
	 * @return array Overview stats
	 */
	private function get_overview_stats(): array {
		$stats = ActivityLogger::get_statistics();

		// Calculate average per day (last 7 days)
		$avg_per_day = $stats['last_7d'] > 0 ? round($stats['last_7d'] / 7, 1) : 0;

		return [
			'total_changes' => $stats['total_changes'],
			'last_24h' => $stats['last_24h'],
			'last_7d' => $stats['last_7d'],
			'avg_per_day' => $avg_per_day,
			'most_active_user' => $stats['most_active_user'],
			'most_changed_flag' => $stats['most_changed_flag'],
		];
	}

	/**
	 * Check if this is an empty state (no data collected yet)
	 *
	 * @return bool True if no activity data exists
	 */
	private function is_empty_state(): bool {
		try {
			$repo = new ActivityRepository();
			$entries = $repo->get_by_filters([], 1); // Get just 1 entry to check
			return empty($entries);
		} catch (\Exception $e) {
			// If database table doesn't exist or query fails, treat as empty state
			// This is normal for fresh installations before database migration
			return true;
		}
	}

	/**
	 * Get next data collection time information
	 *
	 * @return array|null Next collection info or null
	 */
	private function get_next_data_collection(): ?array {
		// For activity analytics, data is collected in real-time
		// But we can check when the next cron cleanup is scheduled
		$next_cleanup = \wp_next_scheduled(\FlxWoo\FeatureFlags\RetentionManager::CRON_HOOK);

		if (!$next_cleanup) {
			return null;
		}

		return [
			'timestamp' => $next_cleanup,
			'human' => \human_time_diff($next_cleanup, time()),
		];
	}

	/**
	 * Get migration status information
	 *
	 * @since 2.4.1
	 * @return array Migration status details
	 */
	private function get_migration_status(): array {
		return Migrator::get_migration_stats();
	}

	/**
	 * AJAX: Get timeline data
	 */
	public function ajax_get_timeline_data(): void {
		// Security checks
		\check_ajax_referer('flx_analytics_nonce', 'nonce');

		if (!\current_user_can('manage_woocommerce')) {
			\wp_send_json_error(['message' => \__('Permission denied', 'flx-woo')]);
		}

		$date_range = isset($_POST['date_range']) ? (int) $_POST['date_range'] : 30;

		$data = $this->prepare_timeline_data($date_range);

		\wp_send_json_success($data);
	}

	/**
	 * AJAX: Get feature breakdown data
	 */
	public function ajax_get_feature_breakdown(): void {
		// Security checks
		\check_ajax_referer('flx_analytics_nonce', 'nonce');

		if (!\current_user_can('manage_woocommerce')) {
			\wp_send_json_error(['message' => \__('Permission denied', 'flx-woo')]);
		}

		$data = $this->prepare_feature_breakdown();

		\wp_send_json_success($data);
	}

	/**
	 * AJAX: Get user activity data
	 */
	public function ajax_get_user_activity(): void {
		// Security checks
		\check_ajax_referer('flx_analytics_nonce', 'nonce');

		if (!\current_user_can('manage_woocommerce')) {
			\wp_send_json_error(['message' => \__('Permission denied', 'flx-woo')]);
		}

		$data = $this->prepare_user_activity();

		\wp_send_json_success($data);
	}

	/**
	 * AJAX: Get action distribution data
	 */
	public function ajax_get_action_distribution(): void {
		// Security checks
		\check_ajax_referer('flx_analytics_nonce', 'nonce');

		if (!\current_user_can('manage_woocommerce')) {
			\wp_send_json_error(['message' => \__('Permission denied', 'flx-woo')]);
		}

		$data = $this->prepare_action_distribution();

		\wp_send_json_success($data);
	}

	/**
	 * Prepare timeline data for line chart
	 *
	 * @param int $days Number of days to show
	 * @return array Chart.js data format
	 */
	private function prepare_timeline_data(int $days = 30): array {
		$repo = new ActivityRepository();

		// Get entries for the date range
		$start_date = gmdate('Y-m-d', strtotime("-{$days} days"));
		$entries = $repo->get_by_filters([
			'start_date' => $start_date,
		], 10000); // Large limit to get all

		// Group by date
		$date_counts = [];
		$current_date = strtotime($start_date);
		$end_date = strtotime('today');

		// Initialize all dates with 0
		while ($current_date <= $end_date) {
			$date_key = gmdate('Y-m-d', $current_date);
			$date_counts[$date_key] = 0;
			$current_date = strtotime('+1 day', $current_date);
		}

		// Count entries by date
		foreach ($entries as $entry) {
			$date_key = gmdate('Y-m-d', $entry['timestamp']);
			if (isset($date_counts[$date_key])) {
				$date_counts[$date_key]++;
			}
		}

		// Sort by date
		ksort($date_counts);

		// Format labels (shorter for display)
		$labels = array_map(function($date) {
			return gmdate('M j', strtotime($date));
		}, array_keys($date_counts));

		return [
			'labels' => array_values($labels),
			'datasets' => [[
				'label' => \__('Changes', 'flx-woo'),
				'data' => array_values($date_counts),
				'borderColor' => '#2271b1',
				'backgroundColor' => 'rgba(34, 113, 177, 0.1)',
				'tension' => 0.3,
				'fill' => true,
			]],
		];
	}

	/**
	 * Prepare feature breakdown data for horizontal bar chart
	 *
	 * @return array Chart.js data format
	 */
	private function prepare_feature_breakdown(): array {
		$repo = new ActivityRepository();

		// Get all entries
		$entries = $repo->get_by_filters([], 10000);

		// Count by feature flag
		$flag_counts = [];
		foreach ($entries as $entry) {
			$flag = $entry['flag_name'];
			if (!isset($flag_counts[$flag])) {
				$flag_counts[$flag] = 0;
			}
			$flag_counts[$flag]++;
		}

		// Sort by count (descending)
		arsort($flag_counts);

		// Take top 10
		$flag_counts = array_slice($flag_counts, 0, 10, true);

		// Format labels (human-readable)
		$labels = array_map(function($flag) {
			return str_replace('_', ' ', ucwords($flag, '_'));
		}, array_keys($flag_counts));

		// Colors (cycle through WordPress colors)
		$colors = ['#2271b1', '#46b450', '#f0b849', '#d63638', '#8c8f94'];
		$backgrounds = array_map(function($i) use ($colors) {
			return $colors[$i % count($colors)];
		}, array_keys($labels));

		return [
			'labels' => array_values($labels),
			'datasets' => [[
				'label' => \__('Changes', 'flx-woo'),
				'data' => array_values($flag_counts),
				'backgroundColor' => $backgrounds,
			]],
		];
	}

	/**
	 * Prepare user activity data for pie chart
	 *
	 * @return array Chart.js data format
	 */
	private function prepare_user_activity(): array {
		$repo = new ActivityRepository();

		// Get all entries
		$entries = $repo->get_by_filters([], 10000);

		// Count by user
		$user_counts = [];
		foreach ($entries as $entry) {
			$user = $entry['user_login'];
			if (!isset($user_counts[$user])) {
				$user_counts[$user] = 0;
			}
			$user_counts[$user]++;
		}

		// Sort by count (descending)
		arsort($user_counts);

		// Take top 5, group rest as "Others"
		$top_users = array_slice($user_counts, 0, 5, true);
		$others_count = array_sum(array_slice($user_counts, 5));

		if ($others_count > 0) {
			$top_users['Others'] = $others_count;
		}

		// Colors
		$colors = ['#2271b1', '#46b450', '#f0b849', '#d63638', '#8c8f94', '#c3c4c7'];

		return [
			'labels' => array_keys($top_users),
			'datasets' => [[
				'data' => array_values($top_users),
				'backgroundColor' => array_slice($colors, 0, count($top_users)),
			]],
		];
	}

	/**
	 * Prepare action distribution data for doughnut chart
	 *
	 * @return array Chart.js data format
	 */
	private function prepare_action_distribution(): array {
		$repo = new ActivityRepository();

		// Get all entries
		$entries = $repo->get_by_filters([], 10000);

		// Count by action
		$action_counts = [];
		foreach ($entries as $entry) {
			$action = $entry['action'];
			if (!isset($action_counts[$action])) {
				$action_counts[$action] = 0;
			}
			$action_counts[$action]++;
		}

		// Format labels (human-readable)
		$labels = [];
		foreach (array_keys($action_counts) as $action) {
			$labels[] = ActivityLogger::get_action_label($action);
		}

		// Colors by action type
		$action_colors = [
			'enabled' => '#46b450',
			'disabled' => '#d63638',
			'rollout_changed' => '#f0b849',
			'kill_switch_on' => '#d63638',
			'kill_switch_off' => '#46b450',
			'updated' => '#2271b1',
		];

		$backgrounds = [];
		foreach (array_keys($action_counts) as $action) {
			$backgrounds[] = $action_colors[$action] ?? '#8c8f94';
		}

		return [
			'labels' => $labels,
			'datasets' => [[
				'data' => array_values($action_counts),
				'backgroundColor' => $backgrounds,
			]],
		];
	}

	/**
	 * AJAX: Export activity log to CSV
	 *
	 * @since 2.3.0
	 */
	public function ajax_export_csv(): void {
		// Security checks
		\check_ajax_referer('flx_analytics_nonce', 'nonce');

		if (!\current_user_can('manage_woocommerce')) {
			\wp_die(\__('You do not have sufficient permissions to export data.', 'flx-woo'));
		}

		// Get filters from request
		$filters = $this->get_filters_from_request();

		// Get repository
		$repo = new ActivityRepository();

		// Fetch entries with filters
		$entries = $repo->get_by_filters($filters, 10000); // Max 10k entries

		// Set CSV headers
		header('Content-Type: text/csv; charset=utf-8');
		header('Content-Disposition: attachment; filename="flxwoo-activity-' . gmdate('Y-m-d-His') . '.csv"');
		header('Pragma: no-cache');
		header('Expires: 0');

		// Open output stream
		$output = fopen('php://output', 'w');

		// Write UTF-8 BOM for Excel compatibility
		fprintf($output, chr(0xEF) . chr(0xBB) . chr(0xBF));

		// Write CSV header row
		fputcsv($output, [
			\__('Date/Time', 'flx-woo'),
			\__('Feature Flag', 'flx-woo'),
			\__('Action', 'flx-woo'),
			\__('Old Value', 'flx-woo'),
			\__('New Value', 'flx-woo'),
			\__('User', 'flx-woo'),
			\__('IP Address', 'flx-woo'),
		]);

		// Write data rows
		foreach ($entries as $entry) {
			$flag_data = \FlxWoo\FeatureFlags\FeatureManager::get_flag($entry['flag_name']);
			fputcsv($output, [
				gmdate('Y-m-d H:i:s', $entry['timestamp']),
				$flag_data['display_name'] ?? str_replace('_', ' ', ucwords($entry['flag_name'], '_')),
				ActivityLogger::get_action_label($entry['action']),
				$this->format_csv_value($entry['old_value']),
				$this->format_csv_value($entry['new_value']),
				$entry['user_login'],
				$entry['ip_address'],
			]);
		}

		fclose($output);
		exit;
	}

	/**
	 * AJAX: Get filtered activity data
	 *
	 * @since 2.3.0
	 */
	public function ajax_get_filtered_data(): void {
		// Security checks
		\check_ajax_referer('flx_analytics_nonce', 'nonce');

		if (!\current_user_can('manage_woocommerce')) {
			\wp_send_json_error(['message' => \__('Permission denied', 'flx-woo')]);
		}

		// Get filters from request
		$filters = $this->get_filters_from_request();

		// Get pagination
		$page = isset($_POST['page']) ? max(1, (int) $_POST['page']) : 1;
		$per_page = isset($_POST['per_page']) ? max(10, min(100, (int) $_POST['per_page'])) : 25;

		// Get repository
		$repo = new ActivityRepository();

		// Fetch entries with filters
		$entries = $repo->get_by_filters($filters, $per_page, ($page - 1) * $per_page);

		// Get total count for pagination
		$total_count = $repo->count_by_filters($filters);

		// Format entries for display
		$formatted_entries = array_map(function($entry) {
			$flag_data = \FlxWoo\FeatureFlags\FeatureManager::get_flag($entry['flag_name']);
			return [
				'id' => $entry['id'],
				'timestamp' => gmdate('Y-m-d H:i:s', $entry['timestamp']),
				'timestamp_human' => \human_time_diff($entry['timestamp'], time()) . ' ago',
				'flag_name' => $flag_data['display_name'] ?? str_replace('_', ' ', ucwords($entry['flag_name'], '_')),
				'flag_name_raw' => $entry['flag_name'],
				'action' => ActivityLogger::get_action_label($entry['action']),
				'action_raw' => $entry['action'],
				'old_value' => $this->format_display_value($entry['old_value']),
				'new_value' => $this->format_display_value($entry['new_value']),
				'user_login' => $entry['user_login'],
				'user_id' => $entry['user_id'],
				'ip_address' => $entry['ip_address'],
			];
		}, $entries);

		\wp_send_json_success([
			'entries' => $formatted_entries,
			'total_count' => $total_count,
			'page' => $page,
			'per_page' => $per_page,
			'total_pages' => ceil($total_count / $per_page),
		]);
	}

	/**
	 * Get filters from POST request
	 *
	 * @since 2.3.0
	 * @return array Filters for repository
	 */
	private function get_filters_from_request(): array {
		$filters = [];

		// Date range
		if (!empty($_POST['start_date'])) {
			$filters['start_date'] = sanitize_text_field($_POST['start_date']);
		}
		if (!empty($_POST['end_date'])) {
			$filters['end_date'] = sanitize_text_field($_POST['end_date']);
		}

		// Date range preset (7/30/90 days)
		if (!empty($_POST['date_range'])) {
			$days = (int) $_POST['date_range'];
			$filters['start_date'] = gmdate('Y-m-d', strtotime("-{$days} days"));
		}

		// Feature flags (array)
		if (!empty($_POST['flag_names']) && is_array($_POST['flag_names'])) {
			$filters['flag_names'] = array_map('sanitize_text_field', $_POST['flag_names']);
		}

		// Actions (array)
		if (!empty($_POST['actions']) && is_array($_POST['actions'])) {
			$filters['actions'] = array_map('sanitize_text_field', $_POST['actions']);
		}

		// User filter
		if (!empty($_POST['user_id'])) {
			$filters['user_id'] = (int) $_POST['user_id'];
		}

		// IP address filter
		if (!empty($_POST['ip_address'])) {
			$filters['ip_address'] = sanitize_text_field($_POST['ip_address']);
		}

		return $filters;
	}

	/**
	 * Format value for CSV export
	 *
	 * @since 2.3.0
	 * @param mixed $value Value to format
	 * @return string Formatted value
	 */
	private function format_csv_value($value): string {
		if ($value === null) {
			return '';
		}

		if (is_bool($value)) {
			return $value ? 'true' : 'false';
		}

		if (is_array($value) || is_object($value)) {
			return wp_json_encode($value);
		}

		return (string) $value;
	}

	/**
	 * Format value for display in UI
	 *
	 * @param mixed $value Value to format
	 * @return string Formatted value
	 */
	private function format_display_value($value): string {
		if ($value === null) {
			return '—';
		}

		if (is_bool($value)) {
			return $value ? \__('Yes', 'flx-woo') : \__('No', 'flx-woo');
		}

		if (is_array($value) || is_object($value)) {
			return wp_json_encode($value, JSON_PRETTY_PRINT);
		}

		return \esc_html((string) $value);
	}
}
