<?php
/**
 * A/B Testing Admin Page
 *
 * Manage and view A/B test results.
 *
 * @package FlxWoo\Admin
 * @since 2.5.0
 */

namespace FlxWoo\Admin;

use FlxWoo\FeatureFlags\FeatureManager;

if (!defined('ABSPATH')) {
    exit;
}

class ABTestingPage {

    /**
     * Valid test statuses
     */
    private const VALID_STATUSES = ['draft', 'active', 'completed', 'archived'];

    /**
     * Constructor - Register AJAX handlers
     */
    public function __construct() {
        add_action('wp_ajax_flx_create_ab_test', [$this, 'ajax_create_test']);
        add_action('wp_ajax_flx_start_ab_test', [$this, 'ajax_start_test']);
        add_action('wp_ajax_flx_stop_ab_test', [$this, 'ajax_stop_test']);
        add_action('wp_ajax_flx_get_test_results', [$this, 'ajax_get_test_results']);
        add_action('wp_ajax_flx_delete_ab_test', [$this, 'ajax_delete_test']);
    }

    /**
     * Render the A/B testing page
     *
     * @return void
     */
    public function render(): void {
        // Permission check
        if (!current_user_can('manage_woocommerce')) {
            wp_die(esc_html__('You do not have sufficient permissions to access this page.', 'flx-woo'));
        }

        // Check feature flag
        $store_id = FeatureManager::get_store_id();
        if (!FeatureManager::is_enabled(FeatureManager::AB_TESTING, $store_id)) {
            $this->render_feature_disabled_notice();
            return;
        }

        // Get all tests
        $tests = $this->get_tests();

        // Include view template
        include __DIR__ . '/views/ab-testing.php';
    }

    /**
     * Get all tests from Next.js API
     *
     * @param string|null $status Optional status filter
     * @return array Tests or empty array on error
     */
    private function get_tests(?string $status = null): array {
        if (!defined('FLX_WOO_RENDERER_URL')) {
            return [];
        }
        $renderer_url = FLX_WOO_RENDERER_URL;

        $store_id = FeatureManager::get_store_id();

        // Build API URL
        $api_url = trailingslashit($renderer_url) . 'api/v1/ab-testing/tests';
        $args = ['store_id' => $store_id];
        if ($status) {
            $args['status'] = $status;
        }
        $api_url = add_query_arg($args, $api_url);

        // Make HTTP request
        $response = wp_remote_get($api_url, [
            'timeout' => 10,
            'headers' => ['Accept' => 'application/json']
        ]);

        // Handle errors
        if (is_wp_error($response)) {
            error_log('FlxWoo A/B Testing API error: ' . $response->get_error_message());
            return [];
        }

        $status_code = wp_remote_retrieve_response_code($response);
        if ($status_code !== 200) {
            error_log('FlxWoo A/B Testing API returned status: ' . $status_code);
            return [];
        }

        // Parse JSON response
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);

        if (!isset($data['success']) || $data['success'] !== true) {
            error_log('FlxWoo A/B Testing API returned error');
            return [];
        }

        return $data['tests'] ?? [];
    }

    /**
     * Get test results from Next.js API
     *
     * @param int $test_id Test ID
     * @return array|null Results or null on error
     */
    private function get_test_results(int $test_id): ?array {
        if (!defined('FLX_WOO_RENDERER_URL')) {
            return null;
        }
        $renderer_url = FLX_WOO_RENDERER_URL;

        // Build API URL
        $api_url = trailingslashit($renderer_url) . 'api/v1/ab-testing/results/' . $test_id;

        // Make HTTP request
        $response = wp_remote_get($api_url, [
            'timeout' => 10,
            'headers' => ['Accept' => 'application/json']
        ]);

        // Handle errors
        if (is_wp_error($response)) {
            error_log('FlxWoo A/B Testing Results API error: ' . $response->get_error_message());
            return null;
        }

        $status_code = wp_remote_retrieve_response_code($response);
        if ($status_code !== 200) {
            error_log('FlxWoo A/B Testing Results API returned status: ' . $status_code);
            return null;
        }

        // Parse JSON response
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);

        if (!isset($data['success']) || $data['success'] !== true) {
            return null;
        }

        return $data['data'] ?? null;
    }

    /**
     * AJAX handler: Create new test
     *
     * @return void
     */
    public function ajax_create_test(): void {
        // Security check
        check_ajax_referer('flx_ab_test_nonce', 'nonce');

        // Permission check
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error(['message' => __('Permission denied', 'flx-woo')]);
        }

        // Get form data
        $test_name = sanitize_text_field($_POST['test_name'] ?? '');
        $test_key = sanitize_key($_POST['test_key'] ?? '');
        $variants = json_decode(stripslashes($_POST['variants'] ?? '[]'), true);
        $traffic_split = json_decode(stripslashes($_POST['traffic_split'] ?? '{}'), true);

        if (empty($test_name) || empty($test_key) || empty($variants)) {
            wp_send_json_error(['message' => __('Missing required fields', 'flx-woo')]);
        }

        // Build request
        if (!defined('FLX_WOO_RENDERER_URL')) {
            wp_send_json_error(['message' => __('Renderer URL not configured', 'flx-woo')]);
        }
        $renderer_url = FLX_WOO_RENDERER_URL;
        $store_id = FeatureManager::get_store_id();

        $api_url = trailingslashit($renderer_url) . 'api/v1/ab-testing/tests';

        $response = wp_remote_post($api_url, [
            'timeout' => 10,
            'headers' => ['Content-Type' => 'application/json'],
            'body' => json_encode([
                'store_id' => $store_id,
                'test_name' => $test_name,
                'test_key' => $test_key,
                'variants' => $variants,
                'traffic_split' => $traffic_split,
            ]),
        ]);

        if (is_wp_error($response)) {
            wp_send_json_error(['message' => $response->get_error_message()]);
        }

        $status_code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);

        if ($status_code === 201 && $data['success']) {
            wp_send_json_success([
                'message' => __('Test created successfully', 'flx-woo'),
                'test_id' => $data['test_id'],
            ]);
        } else {
            wp_send_json_error(['message' => $data['error'] ?? __('Failed to create test', 'flx-woo')]);
        }
    }

    /**
     * AJAX handler: Start test
     *
     * @return void
     */
    public function ajax_start_test(): void {
        // Security check
        check_ajax_referer('flx_ab_test_nonce', 'nonce');

        // Permission check
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error(['message' => __('Permission denied', 'flx-woo')]);
        }

        $test_id = intval($_POST['test_id'] ?? 0);
        if ($test_id <= 0) {
            wp_send_json_error(['message' => __('Invalid test ID', 'flx-woo')]);
        }

        $this->update_test_status($test_id, 'start');
    }

    /**
     * AJAX handler: Stop test
     *
     * @return void
     */
    public function ajax_stop_test(): void {
        // Security check
        check_ajax_referer('flx_ab_test_nonce', 'nonce');

        // Permission check
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error(['message' => __('Permission denied', 'flx-woo')]);
        }

        $test_id = intval($_POST['test_id'] ?? 0);
        if ($test_id <= 0) {
            wp_send_json_error(['message' => __('Invalid test ID', 'flx-woo')]);
        }

        $this->update_test_status($test_id, 'stop');
    }

    /**
     * Update test status via API
     *
     * @param int $test_id Test ID
     * @param string $action Action ('start' or 'stop')
     * @return void
     */
    private function update_test_status(int $test_id, string $action): void {
        if (!defined('FLX_WOO_RENDERER_URL')) {
            wp_send_json_error(['message' => __('Renderer URL not configured', 'flx-woo')]);
        }
        $renderer_url = FLX_WOO_RENDERER_URL;

        $api_url = trailingslashit($renderer_url) . 'api/v1/ab-testing/tests';

        $response = wp_remote_request($api_url, [
            'method' => 'PUT',
            'timeout' => 10,
            'headers' => ['Content-Type' => 'application/json'],
            'body' => json_encode([
                'test_id' => $test_id,
                'action' => $action,
            ]),
        ]);

        if (is_wp_error($response)) {
            wp_send_json_error(['message' => $response->get_error_message()]);
        }

        $status_code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);

        if ($status_code === 200 && $data['success']) {
            wp_send_json_success([
                'message' => $data['message'] ?? sprintf(__('Test %sed successfully', 'flx-woo'), $action),
            ]);
        } else {
            wp_send_json_error(['message' => $data['error'] ?? __('Failed to update test', 'flx-woo')]);
        }
    }

    /**
     * AJAX handler: Get test results
     *
     * @return void
     */
    public function ajax_get_test_results(): void {
        // Security check
        check_ajax_referer('flx_ab_test_nonce', 'nonce');

        // Permission check
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error(['message' => __('Permission denied', 'flx-woo')]);
        }

        $test_id = intval($_POST['test_id'] ?? 0);
        if ($test_id <= 0) {
            wp_send_json_error(['message' => __('Invalid test ID', 'flx-woo')]);
        }

        $results = $this->get_test_results($test_id);

        if ($results === null) {
            wp_send_json_error(['message' => __('Failed to fetch results', 'flx-woo')]);
        }

        wp_send_json_success($results);
    }

    /**
     * AJAX handler: Delete test
     *
     * @return void
     */
    public function ajax_delete_test(): void {
        // Security check
        check_ajax_referer('flx_ab_test_nonce', 'nonce');

        // Permission check
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error(['message' => __('Permission denied', 'flx-woo')]);
        }

        $test_id = intval($_POST['test_id'] ?? 0);
        if ($test_id <= 0) {
            wp_send_json_error(['message' => __('Invalid test ID', 'flx-woo')]);
        }

        // Note: Delete not implemented in API yet, would need DELETE endpoint
        wp_send_json_error(['message' => __('Delete not yet implemented', 'flx-woo')]);
    }

    /**
     * Render feature disabled notice
     *
     * @return void
     */
    private function render_feature_disabled_notice(): void {
        ?>
        <div class="wrap">
            <h1><?php esc_html_e('A/B Testing', 'flx-woo'); ?></h1>
            <div class="notice notice-warning">
                <p>
                    <?php esc_html_e('The A/B Testing feature is currently disabled. Please enable it in the Feature Flags settings.', 'flx-woo'); ?>
                </p>
                <p>
                    <a href="<?php echo esc_url(admin_url('admin.php?page=flx-woo-feature-flags')); ?>" class="button button-primary">
                        <?php esc_html_e('Go to Feature Flags', 'flx-woo'); ?>
                    </a>
                </p>
            </div>
        </div>
        <?php
    }
}
