/**
 * Checkout Abandonment Tracker
 *
 * Detects when user leaves checkout page without completing purchase.
 * Tracks abandonment step (billing, shipping, payment, review).
 *
 * @package FlxWoo
 * @version 2.3.0
 */

(function ($) {
	'use strict';

	/**
	 * Abandonment tracker
	 */
	const AbandonmentTracker = {
		/**
		 * Current checkout step
		 */
		currentStep: 'unknown',

		/**
		 * Abandonment tracked flag
		 */
		tracked: false,

		/**
		 * Initialize tracker
		 */
		init: function () {
			// Detect current checkout step
			this.detectCheckoutStep();

			// Track abandonment on page unload
			this.bindUnloadEvent();

			// Update step on checkout update (AJAX)
			$(document.body).on('updated_checkout', function () {
				AbandonmentTracker.detectCheckoutStep();
			});
		},

		/**
		 * Detect which step of checkout user is on
		 */
		detectCheckoutStep: function () {
			// Check which fields are visible/focused
			if ($('#billing_first_name').length && $('#billing_first_name').is(':visible')) {
				this.currentStep = 'billing';
			}

			if ($('#shipping_first_name').length && $('#shipping_first_name').is(':visible')) {
				this.currentStep = 'shipping';
			}

			// Check if payment methods are visible
			if ($('.wc_payment_methods').length && $('.wc_payment_methods').is(':visible')) {
				this.currentStep = 'payment';
			}

			// Check if place order button is visible
			if ($('#place_order').length && $('#place_order').is(':visible')) {
				this.currentStep = 'review';
			}
		},

		/**
		 * Bind to page unload event
		 */
		bindUnloadEvent: function () {
			const self = this;

			// Use beforeunload to send beacon before page closes
			$(window).on('beforeunload', function (e) {
				// Don't track if already tracked
				if (self.tracked) {
					return;
				}

				// Don't track if user is submitting the form
				if (self.isSubmitting()) {
					return;
				}

				// Track abandonment
				self.trackAbandonment();
			});

			// Also track on visibilitychange (user switches tabs)
			document.addEventListener('visibilitychange', function () {
				if (document.hidden && !self.tracked && !self.isSubmitting()) {
					// User left the page - track after 30 seconds
					setTimeout(function () {
						if (document.hidden && !self.tracked && !self.isSubmitting()) {
							self.trackAbandonment();
						}
					}, 30000); // 30 seconds
				}
			});
		},

		/**
		 * Check if checkout form is being submitted
		 */
		isSubmitting: function () {
			// Check if place order button was clicked recently
			const placeOrderBtn = $('#place_order');
			if (placeOrderBtn.length && placeOrderBtn.hasClass('processing')) {
				return true;
			}

			// Check if WooCommerce is processing
			if ($('body').hasClass('woocommerce-checkout-processing')) {
				return true;
			}

			return false;
		},

		/**
		 * Track abandonment via AJAX
		 */
		trackAbandonment: function () {
			// Mark as tracked
			this.tracked = true;

			// Send beacon (non-blocking, works even if page is closing)
			if (navigator.sendBeacon) {
				// Use sendBeacon for reliable delivery
				const formData = new FormData();
				formData.append('action', 'flxwoo_track_abandonment');
				formData.append('nonce', flxwooAbandonmentTracker.nonce);
				formData.append('step', this.currentStep);

				navigator.sendBeacon(flxwooAbandonmentTracker.ajaxUrl, formData);
			} else {
				// Fallback to sync AJAX (for older browsers)
				$.ajax({
					url: flxwooAbandonmentTracker.ajaxUrl,
					type: 'POST',
					async: false, // Synchronous to ensure it completes
					data: {
						action: 'flxwoo_track_abandonment',
						nonce: flxwooAbandonmentTracker.nonce,
						step: this.currentStep,
					},
				});
			}
		},
	};

	// Initialize on document ready
	$(document).ready(function () {
		AbandonmentTracker.init();
	});
})(jQuery);
