<?php
/**
 * Plugin Name: Flux Payments Gateway for WooCommerce
 * Plugin URI: https://fluxpayments.com
 * Description: Accept credit cards, ACH, and cryptocurrency payments with lower fees and zero chargeback risk on crypto transactions.
 * Version: 1.7.2
 * Author: Flux Payments
 * License: GPL-2.0-or-later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: flux-payments-gateway
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) exit;
add_action('plugins_loaded', 'init_flux_gateway_class');

function init_flux_gateway_class()
{
    if (!class_exists('WC_Payment_Gateway')) {
        return;
    }

    class WC_Gateway_Flux extends WC_Payment_Gateway
    {
        public function __construct()
        {



            $this->id = 'flux_gateway';
            $this->method_title = 'Flux Payments';
            $this->has_fields = false;
            $this->method_description = 'Accept payments with Flux Payments.';
            $this->has_fields = false;
            $this->icon = plugins_url('assets/logo.png', __FILE__);


            $this->init_form_fields();
            $this->init_settings();

            $this->title = $this->get_option('title');
            $this->description = $this->get_option('description');
            $this->enabled = $this->get_option('enabled');

            if ($this->enabled !== "yes") {
                return;
            }



            // Side cart Pay Now → wc-ajax=checkout (Flux) with icon + loader
            add_action('wp_enqueue_scripts', function () {
                // Enqueue CSS
                wp_enqueue_style(
                    'flux-paynow-styles',
                    plugins_url('assets/flux-paynow.css', __FILE__),
                    [],
                    '1.0.0'
                );

                // Enqueue JavaScript
                wp_enqueue_script(
                    'flux-paynow-script',
                    plugins_url('assets/flux-paynow.js', __FILE__),
                    ['jquery'],
                    '1.0.0',
                    true
                );

                // Expose endpoint + nonce + fallback checkout URL + icon
                $boot = [
                    'ajax_url' => \WC_AJAX::get_endpoint('checkout'),                 // /?wc-ajax=checkout
                    'nonce' => wp_create_nonce('woocommerce-process_checkout'),    // same as #place_order
                    'checkout_url' => wc_get_checkout_url(),
                    'icon' => plugins_url('assets/flux-icon.svg', __FILE__),      // <-- your mark
                    'prefill' => [
                        'billing_email' => WC()->customer ? WC()->customer->get_billing_email() : '',
                        'billing_first_name' => WC()->customer ? WC()->customer->get_billing_first_name() : '',
                        'billing_last_name' => WC()->customer ? WC()->customer->get_billing_last_name() : '',
                    ],
                ];
                wp_add_inline_script('flux-paynow-script', 'window.FLUX_PAYNOW=' . wp_json_encode($boot) . ';', 'before');
            });


            add_action('woocommerce_update_product', [$this, 'put_or_post_product'], 10, 1);
            add_action('woocommerce_update_options_payment_gateways_' . $this->id, [$this, 'process_admin_options']);
            add_action('woocommerce_api_flux_gateway_response', [$this, 'handle_flux_response']);
        }

        public function put_or_post_product($product_id)
        {
            $product = wc_get_product($product_id);
            if (!$product) {
                return;
            }

            if (get_transient("flux_processed_product_{$product_id}")) {
                return;
            }


            set_transient("flux_processed_product_{$product_id}", true, 10);

            $flux_product_id = $product->get_meta('flux_product_id');
            $image_url = $this->get_product_image_url($product);
            $product_data = $this->build_flux_product_data($product, $flux_product_id, $image_url);

            try {
                $response = $this->sync_product_with_flux($product_data, $flux_product_id, $image_url);
            } catch (Exception $e) {
                $product->delete(true);
                throw $e;
            }

            if (!$flux_product_id && isset($response['createdObjects'][0]['id'])) {
                $product->update_meta_data('flux_product_id', $response['createdObjects'][0]['id']);
                $product->save();
            }
        }

        private function get_product_image_url($product)
        {
            $image_id = $product->get_image_id();
            if ($image_id) {
                $image_url = wp_get_attachment_url($image_id);
                if ($image_url) {
                    return $image_url;
                }
            } else {
            }
            return null;
        }

        private function build_flux_product_data($product, $flux_product_id, $image_url)
        {

            $is_virtual = $product->is_virtual();
            $is_downloadable = $product->is_downloadable();

            $is_physical = !$is_virtual && !$is_downloadable;



            $sale_price = $product->get_sale_price();
            $regular_price = $product->get_regular_price();

            $price_to_use = !empty($sale_price) ? $sale_price : $regular_price;
            
            // Ensure price is valid - convert to float and handle edge cases
            $price_to_use = floatval($price_to_use);
            if ($price_to_use <= 0) {
                error_log("Flux: Product {$product->get_name()} (ID: {$product->get_id()}) has invalid price: {$price_to_use}. Setting to 0.01");
                $price_to_use = 0.01; // Minimum price for Flux Payments
            }

            $data = [
                "name" => $product->get_name(),
                "sku" => $product->get_sku(),
                "price" => $price_to_use,
                "objectType" => "product",
                "description" => $product->get_description(),
                "type" => $is_physical ? "PHYSICAL_PRODUCT" : "SERVICE"
            ];

            if ($flux_product_id) {
                $data["id"] = $flux_product_id;
            }
            if ($image_url) {
                $data["imageUrl"] = basename(wp_parse_url($image_url, PHP_URL_PATH));
            }

            return $data;
        }

        private function sync_product_with_flux($product_data, $flux_product_id, $full_image_url)
        {
            $url = 'https://api.fluxpayments.com/api/product';
            $method = $flux_product_id ? 'PUT' : 'POST';
            $body = json_encode(
                !$flux_product_id
                ? ["createdObjects" => [$product_data]]
                : ["updatedObjects" => [$product_data]]
            );


            $response = $this->flux_api_request($url, $method, $body);

            if (!is_array($response)) {
                throw new Exception("Invalid response from Flux Payments: " . esc_textarea($response));
            }

            if (!$flux_product_id && !isset($response['createdObjects'][0]['id'])) {
                throw new Exception("Invalid response structure from Flux Payments");
            }

            if (empty($product_data['imageUrl'])) {
                return $response;
            }

            $image_url = $full_image_url;
            $image_response = wp_remote_get($image_url);

            if (is_wp_error($image_response)) {
                return $response;
            }

            $image_contents = wp_remote_retrieve_body($image_response);

            if (empty($image_contents)) {
                return $response;
            }

            $finfo = new finfo(FILEINFO_MIME_TYPE);
            $mime_type = $finfo->buffer($image_contents);

            $upload_url = $response['createdObjects'][0]['documentMap'][0]['uploadUrl'] ?? $response['updatedObjects'][0]['documentMap'][0]['uploadUrl'];


            if (!$upload_url) {
                return $response;
            }

            $upload_args = [
                'timeout' => 30,
                'method' => 'PUT',
                'headers' => [
                    'Content-Type' => $mime_type,
                    'Content-Length' => strlen($image_contents)
                ],
                'body' => $image_contents
            ];

            $upload_response = wp_remote_request($upload_url, $upload_args);

            if (is_wp_error($upload_response)) {
                return $response;
            }

            $http_code = wp_remote_retrieve_response_code($upload_response);

            if ($http_code >= 200 && $http_code < 300) {
            } else {
            }

            return $response;
        }
        private function flux_api_request($url, $method, $body)
        {
            $args = [
                'timeout' => 15000,
                'method' => $method,
                'headers' => [
                    'Content-Type' => 'application/json',
                    'Authorization' => 'Basic ' . base64_encode($this->get_option('flux_api_key'))
                ],
            ];

            if ($method !== 'GET' && !empty($body)) {
                $args['body'] = $body;
            }

            $response = wp_remote_request($url, $args);


            if (is_wp_error($response)) {
                throw new Exception("Failed to communicate with Flux Payments");
            }

            $httpCode = wp_remote_retrieve_response_code($response);
            $body = wp_remote_retrieve_body($response);

            error_log('Flux API Response Code: ' . $httpCode);
            error_log('Flux API Response Body: ' . $body);

            if ($httpCode !== 200) {
                throw new Exception("Failed to communicate with Flux Payments: " . esc_textarea($body));
            }

            error_log('Flux API Response: ' . print_r($body, true));

            return json_decode($body, true);
        }

        private function get_currencies()
        {
            $url = 'https://api.fluxpayments.com/api/currency';
            $response = $this->flux_api_request($url, 'GET', []);
            return $response;
        }

        private function get_wallets()
        {
            $url = 'https://api.fluxpayments.com/api/wallet';
            $response = $this->flux_api_request($url, 'GET', []);
            return $response;
        }


        public function init_form_fields()
        {

            $this->form_fields = [
                'enabled' => [
                    'title' => 'Enable/Disable',
                    'type' => 'checkbox',
                    'label' => 'Enable Flux Payments',
                    'default' => 'yes'
                ],
                'button_sync_products' => [
                    'title' => 'Sync Products',
                    'type' => 'button',
                    'default' => 'Sync Products',
                    'label' => 'Sync Products',
                    'description' => 'Click to synchronize products with Flux Payments'
                ],
                'flux_api_key' => [
                    'title' => 'Flux API Key',
                    'type' => 'password',
                    'default' => ''
                ],
                'title' => [
                    'title' => 'Title',
                    'type' => 'text',
                    'default' => 'Flux Payments'
                ],
                'crypto_enabled' => [
                    'title' => 'Enable Crypto',
                    'type' => 'checkbox',
                    'label' => 'Offer crypto at checkout',
                    'default' => 'yes'
                ]
            ];


            $crypto_enabled = $this->get_option('crypto_enabled');
            if ($crypto_enabled === 'yes') {


                $wallets = $this->get_wallets();


                if (isset($wallets['count']) && $wallets['count'] == 0) {
                    $wallet_fields = [
                        'crypto_wallets_title' => [
                            'title' => 'No wallets found',
                            'type' => 'title',
                            'description' => 'No wallets found. Please create a wallet in the Flux Payments dashboard before setting up crypto.',
                        ]
                    ];
                    $this->form_fields = array_merge($this->form_fields ?? [], $wallet_fields);
                    return;
                }

                $wallet_fields = [
                    'crypto_wallets_title' => [
                        'title' => 'Allowed Wallets',
                        'type' => 'title',
                        'description' => 'Choose wallets you want to enable/monitor.',
                    ],
                ];

                if (isset($wallets['objects']) && is_array($wallets['objects'])) {
                    foreach ($wallets['objects'] as $wallet) {
                        $id = $wallet['id'] ?? '';
                        $address = substr($wallet['publicAddress'] ?? '', 0, 5);
                        $chain = $wallet['chain'] ?? '';
                        $memoTag = $wallet['memoTag'] ?? null;

                        $memoPart = $memoTag ? " - {$memoTag}" : "";
                        $label = "{$id} - {$address}{$memoPart} - {$chain}";

                        $wallet_fields["crypto_wallet_{$id}"] = [
                            'title' => $label,
                            'type' => 'checkbox',
                            'label' => 'Enable',
                            'default' => 'no',
                            'class' => 'crypto-wallet-row',
                        ];
                    }
                }

                $this->form_fields = array_merge($this->form_fields ?? [], $wallet_fields);

                $currencies = $this->get_currencies();

                $asset_fields = [
                    'crypto_assets_title' => [
                        'title' => 'Allowed Crypto Assets',
                        'type' => 'title',
                        'description' => 'Choose assets you support.',
                    ],
                ];

                if (isset($currencies['objects']) && is_array($currencies['objects'])) {
                    foreach ($currencies['objects'] as $currency) {
                        $id = $currency['id'] ?? '';
                        $shortName = $currency['shortName'] ?? '';
                        $issuer = $currency['issuer'] ?: '';
                        $chain = $currency['chain'] ?? '';

                        $label = "{$id} - {$shortName} - {$issuer} - {$chain}";

                        $asset_fields["crypto_asset_{$id}"] = [
                            'title' => $label,
                            'type' => 'checkbox',
                            'label' => 'Enable',
                            'default' => 'no',
                        ];
                    }
                }


                $this->form_fields = array_merge($this->form_fields ?? [], $asset_fields);



            }

        }

        private function get_order_flux_product_ids($order)
        {
            $product_ids = [];
            foreach ($order->get_items() as $item) {
                $product_id = $item->get_product_id();
                $product = wc_get_product($product_id);
                if ($product) {
                    $flux_id = $product->get_meta('flux_product_id');
                    if ($flux_id) {
                        $product_ids[] = $flux_id;
                    }
                }
            }
            return array_unique($product_ids);
        }


        public function process_payment($order_id)
        {

            $order = wc_get_order($order_id);
            $order->set_payment_method('flux_gateway');
            $order->save();

            $redirect_url = wc_get_checkout_url();







            $response_nonce = wp_create_nonce('flux_response_' . $order->get_id());
            $return_url = add_query_arg([
                'wc-api' => 'flux_gateway_response',
                'flux_nonce' => $response_nonce,
                'key' => $order->get_order_key()
            ], home_url('/'));
            
            $params = array(
                'order_id' => $order->get_id(),
                'amount' => $order->get_total(),
                'currency' => get_woocommerce_currency(),
                'return_url' => esc_url_raw($return_url)
            );


            $url = esc_url_raw($redirect_url . '?' . http_build_query($params));



            WC()->session->set('flux_redirect_url', $url);





            $redirect_url = $this->flux_cart_redirect($order->get_order_number());


            return array(
                'result' => 'success',
                'redirect' => $redirect_url
            );
        }
        public function flux_cart_redirect($order_number)
        {
            $cart_items = WC()->cart->get_cart();
            $product_ids = [];
            foreach ($cart_items as $cart_item) {
                $product_id = $cart_item['product_id'];
                $quantity = $cart_item['quantity']; // Get the quantity of the product in cart
                $product = wc_get_product($product_id);
                if ($product) {
                    $flux_id = $product->get_meta('flux_product_id');
                    if ($flux_id) {
                        $product_ids[] = [
                            'id' => $flux_id,
                            'quantity' => $quantity
                        ];
                    }
                }
            }

            $product_json = $this->build_flux_product_json($product_ids);

            try {
                $response = $this->create_flux_payment_link($product_json, $order_number);
            } catch (Exception $e) {
                echo '<p>Failed to create Flux payment link.</p>';
                return;
            }


            $payment_link = $response['createdObjects'][0]['paymentLink'] ?? null;
            $redirect_url = $payment_link ? "https://payments.fluxpayments.com/" . $payment_link : null;
            $payment_link_id = $response['createdObjects'][0]['id'] ?? null;

            $order = wc_get_order($order_number);
            if ($order) {
                $order->update_meta_data('flux_payment_link_id', $payment_link_id);
                $order->save();
            }

            return $redirect_url;
        }
        private function create_flux_payment_link($product_json, $order_number)
        {
            $order = wc_get_order($order_number);
            $nonce = wp_create_nonce('flux_transaction_' . $order_number);
            
            // Build the redirect URL properly
            $redirect_url = add_query_arg([
                'key' => $order->get_order_key(),
                'flux_nonce' => $nonce
            ], wc_get_endpoint_url('order-received', $order_number, wc_get_page_permalink('checkout')));
            
            // For JSON encoding, we need to escape the URL properly
            $redirect_url = str_replace(['/', '?', '=', '&'], ['\/', '\?', '\=', '\&'], $redirect_url);

            $enabled_wallet_ids = [];
            $enabled_asset_ids = [];


            foreach (array_keys($this->form_fields ?? []) as $key) {
                $value = $this->get_option($key);

                if ($value === 'yes') {
                    if (strpos($key, 'crypto_wallet_') === 0) {
                        $enabled_wallet_ids[] = (int) str_replace('crypto_wallet_', '', $key);
                    } elseif (strpos($key, 'crypto_asset_') === 0) {
                        $enabled_asset_ids[] = (int) str_replace('crypto_asset_', '', $key);
                    }
                }
            }


            $body = [
                "createdObjects" => [
                    [
                        "objectType" => "payment_link",
                        "products" => $product_json,
                        "emailNotificationDisabled" => true,
                        "isReusable" => true,
                        "redirectUrl" => $redirect_url,
                        "minutesToExpire" => 30
                    ]
                ]
            ];

            $wallet_json = [];
            $asset_json = [];

            foreach ($enabled_wallet_ids as $wallet_id) {
                $wallet_json[] = [
                    "objectType" => "wallet",
                    "id" => (int) $wallet_id
                ];
            }


            foreach ($enabled_asset_ids as $asset_id) {
                $asset_json[] = [
                    "objectType" => "currency",
                    "id" => (int) $asset_id
                ];
            }

            if (count($wallet_json) > 0) {
                $body['createdObjects'][0]['wallets'] = $wallet_json;
            }

            if (count($asset_json) > 0) {
                $body['createdObjects'][0]['currencies'] = $asset_json;
            }


            // Debug: Log the body to WordPress debug log
            error_log('Flux Payment Link Body: ' . print_r($body, true));
            
            // Debug: Also log to a custom file for easier viewing
            file_put_contents(WP_CONTENT_DIR . '/flux-debug.log', date('Y-m-d H:i:s') . " - Body: " . print_r($body, true) . "\n", FILE_APPEND);
            $url = 'https://api.fluxpayments.com/api/paymentLink';
            $body = json_encode($body);
            $response = $this->flux_api_request($url, 'POST', $body);
            error_log('Flux Payment Link Response: ' . print_r($response, true));
            return $response;
        }

        private function build_flux_product_json($product_ids)
        {
            $product_json = [];
            foreach ($product_ids as $product_id) {
                $product_json[] = [
                    "objectType" => "product",
                    "id" => (int) $product_id['id'],
                    "orderQuantity" => (int) $product_id['quantity']
                ];
            }
            return $product_json;
        }


        public function get_flux_transaction($transaction_id)
        {
            $url = 'https://api.fluxpayments.com/api/transaction/' . $transaction_id;
            $response = $this->flux_api_request($url, 'GET', []);
            return $response;
        }


        /**
         * 
         * TODO: UPDATE THE SHIPPING ADDRESS
         * 
         * 
         * @param mixed $order
         * @param mixed $transaction_id
         * @return void
         */
        public function thankyou_page($order, $transaction_id)
        {
            $transaction = $this->get_flux_transaction($transaction_id);


            $transaction_status = null;
            if (isset($transaction['objects'][0]['approvalStatus'])) {
                $transaction_status = $transaction['objects'][0]['approvalStatus'];
            }

            if ($transaction_status === 'FINALIZED') {
                $order->update_status('completed');
                $order->add_order_note('Payment completed via Flux.');
            } else {
                $order->update_status('failed');
                $order->add_order_note('Payment pending via Flux.');
            }

            $order->update_meta_data('flux_transaction_id', $transaction_id);


        }


        public function handle_flux_response()
        {
            $order_id = isset($_GET['order_id']) ? absint($_GET['order_id']) : 0;
            $nonce = isset($_GET['flux_nonce']) ? sanitize_text_field(wp_unslash($_GET['flux_nonce'])) : '';
            
            // Verify nonce for security
            if (!wp_verify_nonce($nonce, 'flux_response_' . $order_id)) {
                wp_die(__('Security check failed. Invalid response verification.', 'flux-payments-gateway'));
            }
            
            $order = wc_get_order($order_id);

            if (!$order) {
                wp_die(__('Invalid order.', 'flux-payments-gateway'));
            }

            // Check user permissions
            if (!current_user_can('edit_shop_orders') && $order->get_billing_email() !== wp_get_current_user()->user_email) {
                $order_key = isset($_GET['key']) ? sanitize_text_field(wp_unslash($_GET['key'])) : '';
                if (!$order_key || !hash_equals($order->get_order_key(), $order_key)) {
                    wp_die(__('You do not have permission to view this order.', 'flux-payments-gateway'));
                }
            }

            $order->payment_complete();
            $order->add_order_note('Payment completed via Flux.');

            wp_redirect($this->get_return_url($order));
            exit;
        }

        public function get_icon()
        {
            $icon_url = plugins_url('assets/flux-icon.svg', __FILE__);
            $icon = '<img src="' . esc_url($icon_url) . '" alt="Flux" style="width:-webkit-fill-available;height:100px;border-radius:6px;vertical-align:middle;" />';
            return apply_filters('woocommerce_gateway_icon', $icon, $this->id);
        }

        public function payment_fields()
        {
            echo '<div style="display:none;">Pay Securely with Flux</div>';
        }


        public function render_flux_checkout_button(): void
        {
            // Re-use WC’s filter so merchants can still override text
            $label = apply_filters(
                'woocommerce_order_button_text',
                __('Pay With Flux', 'flux-payments-gateway')
            );

            printf(
                '<div id="flux-submit-button-container" style="display:none;">
                    <button id="place_order" type="button"
                        class="button alt flux-order-button wp-element-button">
                        %1$s
                    </button>
                </div>',
                esc_html($label)
            );
        }
    }

}


add_filter('woocommerce_payment_gateways', 'add_flux_gateway_class');
function add_flux_gateway_class($methods)
{
    $methods[] = 'WC_Gateway_Flux';
    return $methods;
}


add_filter('woocommerce_checkout_fields', function ($fields) {
    $fields['billing'] = [];
    $fields['shipping'] = [];
    if (isset($fields['order']['order_comments'])) {
        unset($fields['order']['order_comments']);
    }
    return $fields;
});

add_action('wp', function () {
    remove_action('woocommerce_before_checkout_form', 'woocommerce_checkout_coupon_form', 10);
    remove_action('woocommerce_checkout_order_review', 'woocommerce_order_review', 10);
});

add_filter('woocommerce_cart_needs_shipping', '__return_false');
add_filter('woocommerce_cart_needs_shipping_address', '__return_false');
add_filter('woocommerce_checkout_show_shipping', '__return_false');
add_filter('woocommerce_checkout_show_billing', '__return_false');

// This CSS is now included in flux-paynow.css

add_action('template_redirect', function () {
    // Only process on order-received pages
    if (!is_wc_endpoint_url('order-received')) {
        return;
    }

    $order_id = absint(get_query_var('order-received'));
    $transaction_id = isset($_GET['transactionId']) ? sanitize_text_field(wp_unslash($_GET['transactionId'])) : null;

    // If no transaction ID, this isn't a Flux callback
    if (!$order_id || !$transaction_id) {
        return;
    }

    $order = wc_get_order($order_id);
    if (!$order || !in_array($order->get_status(), ['pending', 'processing'])) {
        return;
    }

    // Verify order key for security (this is how WooCommerce normally validates order access)
    $order_key = isset($_GET['key']) ? sanitize_text_field(wp_unslash($_GET['key'])) : '';
    if (!$order_key || !hash_equals($order->get_order_key(), $order_key)) {
        wp_die(__('You do not have permission to view this order.', 'flux-payments-gateway'));
    }

    // Additional nonce verification if present (for extra security)
    $nonce = isset($_GET['flux_nonce']) ? sanitize_text_field(wp_unslash($_GET['flux_nonce'])) : '';
    if ($nonce && !wp_verify_nonce($nonce, 'flux_transaction_' . $order_id)) {
        wp_die(__('Security check failed. Invalid transaction verification.', 'flux-payments-gateway'));
    }

    if ($order->get_meta('_flux_thankyou_handled') === 'yes') {
        return;
    }

    if (class_exists('WC_Gateway_Flux')) {
        $gateway = new WC_Gateway_Flux();
        $gateway->thankyou_page($order, $transaction_id);
    }

    $order->update_meta_data('_flux_thankyou_handled', 'yes');
    $order->save();
});


add_action('admin_enqueue_scripts', function ($hook) {
    $screen = get_current_screen();
    if ($screen && $screen->id === 'woocommerce_page_wc-settings') {
        wp_enqueue_script(
            'flux-admin-script',
            plugins_url('assets/flux-admin.js', __FILE__),
            ['jquery'],
            '1.0.0',
            true
        );
        
        wp_localize_script('flux-admin-script', 'flux_admin', [
            'nonce' => wp_create_nonce('flux_sync_products'),
            'ajaxurl' => admin_url('admin-ajax.php')
        ]);
    }
});

add_action('wp_ajax_flux_sync_products', function () {
    check_ajax_referer('flux_sync_products');
    $args = array('post_type' => 'product', 'posts_per_page' => -1, 'fields' => 'ids');
    $products = get_posts($args);
    foreach ($products as $product_id) {
        (new WC_Gateway_Flux())->put_or_post_product($product_id);
    }
    wp_send_json_success('All products synced!');
});

add_action('wp_enqueue_scripts', function () {
    if (is_checkout()) {
        wp_enqueue_style(
            'flux-checkout',
            plugins_url('assets/flux-checkout.css', __FILE__),
            [],
            '1.0'
        );
    }
    
    // Enqueue scripts for cart page Flux button
    if (is_cart()) {
        wp_enqueue_script('jquery');
        // Make sure WooCommerce checkout params are available
        wp_localize_script('jquery', 'wc_checkout_params', [
            'ajax_url' => admin_url('admin-ajax.php')
        ]);
        
        // Add CSS to hide the default checkout button
        wp_add_inline_style('woocommerce-general', '
            .wc-proceed-to-checkout .checkout-button,
            .cart .checkout-button,
            .woocommerce-cart .checkout-button {
                display: none !important;
            }
        ');
    }
});


add_filter('woocommerce_order_button_text', function ($text) {
    return 'Pay With Flux';
});

add_filter('woocommerce_order_button_html', function ($button_html) {
    return str_replace('class="button', 'class="button flux-primary-btn', $button_html);
});

// Remove the default "Proceed to Checkout" button from cart page
add_action('wp', function() {
    if (is_cart()) {
        // Remove the checkout button action
        remove_action('woocommerce_proceed_to_checkout', 'woocommerce_button_proceed_to_checkout', 20);
    }
});

// Disable all coupon functionality
add_filter('woocommerce_coupons_enabled', '__return_false');

// Remove coupon form from cart page
add_action('wp', function() {
    remove_action('woocommerce_before_checkout_form', 'woocommerce_checkout_coupon_form', 10);
    // Don't remove cart totals - we need those for the Pay Now button
});

// Remove coupon field from checkout
add_filter('woocommerce_checkout_coupon_message', '__return_empty_string');

// Hide coupon-related elements with CSS
add_action('wp_head', function() {
    if (is_cart() || is_checkout()) {
        echo '<style>
            .woocommerce-form-coupon-toggle,
            .woocommerce-form-coupon,
            .checkout_coupon,
            .coupon,
            .woocommerce-remove-coupon,
            .cart-discount,
            .order-discount,
            .wc-block-components-totals-coupon,
            .wc-block-components-totals-discount,
            .woocommerce-cart-form .coupon,
            .woocommerce-checkout .coupon,
            .cart_totals .coupon,
            .shop_table .coupon,
            .woocommerce-coupon-data,
            .woocommerce-coupon-description,
            .coupon-info,
            .discount-info,
            .wc-coupon-form,
            .apply-coupon {
                display: none !important;
            }
        </style>';
    }
});

// Remove coupon functionality from admin
add_action('admin_init', function() {
    // Remove coupon menu from admin
    remove_menu_page('edit.php?post_type=shop_coupon');
});

// Disable coupon creation and usage completely
add_filter('wc_coupons_enabled', '__return_false');
add_filter('woocommerce_cart_totals_coupon_html', '__return_empty_string');
add_filter('woocommerce_cart_totals_coupon_label', '__return_empty_string');

// Add Flux Payments button to cart page
add_action('woocommerce_proceed_to_checkout', 'flux_add_cart_checkout_button');

function flux_add_cart_checkout_button() {
    // Only show if Flux gateway is enabled
    $gateways = WC()->payment_gateways->get_available_payment_gateways();
    if (!isset($gateways['flux_gateway']) || $gateways['flux_gateway']->enabled !== 'yes') {
        return;
    }

    // Create a nonce for security
    $nonce = wp_create_nonce('flux_cart_checkout');
    
    echo '<a href="#" 
            id="flux-cart-checkout-btn" 
            class="button alt flux-cart-checkout" 
            style="margin-left:10px;background:#b6ff00;color:#000;font-weight:bold;"
            data-nonce="' . esc_attr($nonce) . '">
            Pay Now
          </a>';
    
    // Add JavaScript to handle the button click
    ?>
    <script type="text/javascript">
    jQuery(document).ready(function($) {
        var $button = $('#flux-cart-checkout-btn');
        var originalText = 'Pay Now'; // Set the original text
        
        // Reset button state on page load (in case user navigated back)
        $button.text(originalText).prop('disabled', false);
        
        $button.on('click', function(e) {
            e.preventDefault();
            
            // Show loading state
            $button.text('Processing...').prop('disabled', true);
            
            // Create order via AJAX and redirect to Flux
            $.ajax({
                url: wc_checkout_params.ajax_url,
                type: 'POST',
                data: {
                    action: 'flux_create_cart_order',
                    nonce: $button.data('nonce')
                },
                success: function(response) {
                    if (response.success && response.data.redirect_url) {
                        // Redirect to Flux Payments
                        window.location.href = response.data.redirect_url;
                    } else {
                        alert('Error: ' + (response.data.message || 'Failed to create order'));
                        $button.text(originalText).prop('disabled', false);
                    }
                },
                error: function() {
                    alert('Network error. Please try again.');
                    $button.text(originalText).prop('disabled', false);
                }
            });
        });
        
        // Also reset on browser back/forward navigation
        $(window).on('pageshow', function(event) {
            if (event.originalEvent.persisted) {
                $button.text(originalText).prop('disabled', false);
            }
        });
    });
    </script>
    <?php
}

// Handle AJAX request to create order from cart
add_action('wp_ajax_flux_create_cart_order', 'flux_handle_cart_checkout');
add_action('wp_ajax_nopriv_flux_create_cart_order', 'flux_handle_cart_checkout');

function flux_handle_cart_checkout() {
    // Verify nonce
    if (!wp_verify_nonce($_POST['nonce'], 'flux_cart_checkout')) {
        wp_send_json_error(['message' => 'Security check failed']);
        return;
    }
    
    // Check if cart is not empty
    if (WC()->cart->is_empty()) {
        wp_send_json_error(['message' => 'Cart is empty']);
        return;
    }
    
    try {
        // Create order from cart
        $checkout = WC()->checkout();
        
        // Set payment method to Flux
        $_POST['payment_method'] = 'flux_gateway';
        
        // Create the order
        $order_id = $checkout->create_order([
            'payment_method' => 'flux_gateway'
        ]);
        
        if (is_wp_error($order_id)) {
            wp_send_json_error(['message' => $order_id->get_error_message()]);
            return;
        }
        
        // Get the Flux gateway instance
        $gateways = WC()->payment_gateways->get_available_payment_gateways();
        $flux_gateway = $gateways['flux_gateway'];
        
        // Process payment (this will create the Flux payment link)
        $result = $flux_gateway->process_payment($order_id);
        

        wp_send_json_success(['message' => 'Payment processed successfully', 'redirect_url' => $result['redirect'] ?? null, 'order_id' => $order_id ]);
    } catch (Exception $e) {
        wp_send_json_error(['message' => $e->getMessage(), 'order_id' => $order_id ]);
    }
}
