<?php
/**
 * WCFP_Pricing_Manager.
 *
 * @since 1.0.0
 * @package WCFP_Pricing_Manager
 */

defined( 'ABSPATH' ) || exit;
if ( ! class_exists( 'WCFP_Pricing_Manager' ) ) {
	/**
	 * Pricing Manager.
	 *
	 * @since 1.0.0
	 */
	class WCFP_Pricing_Manager {
		/**
		 * API client object.
		 *
		 * @var class
		 * @since 1.0.0
		 */
		private static $api_client;
		/**
		 * Modifying product prices based on rulesets.
		 *
		 * @since 1.0.0
		 */
		public function __construct() {
			$api_client       = new WCFP_API_Client();
			self::$api_client = $api_client;
			add_action( 'woocommerce_init', array( __CLASS__, 'wcfp_set_session_prices' ) );
			add_action( 'woocommerce_before_calculate_totals', array( __CLASS__, 'wcfp_update_cart_item_prices' ), 10, 1 );
			add_filter( 'woocommerce_product_get_price', array( __CLASS__, 'wcfp_adjust_price' ), 9999, 2 );
			add_filter( 'woocommerce_product_get_regular_price', array( __CLASS__, 'wcfp_adjust_price' ), 9999, 2 );
			add_filter( 'woocommerce_product_variation_get_price', array( __CLASS__, 'wcfp_adjust_price' ), 999, 2 );
			add_filter( 'woocommerce_variable_price_html', array( __CLASS__, 'wcfp_variable_prices' ), 999, 2 );
			add_filter( 'woocommerce_product_variation_get_regular_price', array( __CLASS__, 'wcfp_adjust_price' ), 999, 2 );
			add_filter( 'woocommerce_variation_prices_price', array( __CLASS__, 'wcfp_variation_price' ), 999, 3 );
			add_filter( 'woocommerce_variation_prices_regular_price', array( __CLASS__, 'wcfp_variation_price' ), 999, 3 );
		}
		/**
		 * Modifying variable product prices.
		 *
		 * @param string $price_html variable price HTML.
		 * @param object $product product object.
		 *
		 * @since 1.0.0
		 */
		public static function wcfp_variable_prices( $price_html, $product ) {
			$visible_children = $product->get_visible_children();
			$variation_prices = array();
			if ( ! empty( $visible_children ) ) {
				foreach ( $visible_children as $variation_id ) {
					$variation_prices[] = wc_get_product( $variation_id )->get_price();
				}
			}
			sort( $variation_prices );
			$min = current( $variation_prices );
			$max = end( $variation_prices );
			if ( $min === $max ) {
				return wc_price( $min );
			}
			return sprintf( '%s - %s', wc_price( $min ), wc_price( $max ) );
		}
		/**
		 * WooCommerce session for dynamic pricing API call.
		 *
		 * @since 1.0.0
		 */
		public static function wcfp_set_session_prices() {
			$minutes                         = get_option( 'wcfp_pricing_refresh_interval' );
			$minutes                         = ! empty( $minutes ) ? $minutes : 0;
			$current_time                    = gmdate( 'Y-m-d H:i:s' );
			$api_pricing_last_refreshed_date = wcfp_get_api_pricing_last_refreshed_date_from_session();
			if ( $api_pricing_last_refreshed_date ) { // if already visited to the store.
				$time_difference       = strtotime( $current_time ) - strtotime( $api_pricing_last_refreshed_date );
				$difference_in_minutes = round( $time_difference / 60, 2 );
				if ( $difference_in_minutes >= $minutes ) {
					$dynamic_prices = self::$api_client->wcfp_fetch_dynamic_prices();
					wcfp_set_dynamic_prices_in_session( $dynamic_prices );
					wcfp_set_api_pricing_last_refreshed_date_in_session( $current_time );
				}
			} else { // if visit the store for the first time.
				$dynamic_prices = self::$api_client->wcfp_fetch_dynamic_prices();
				wcfp_set_dynamic_prices_in_session( $dynamic_prices );
				wcfp_set_api_pricing_last_refreshed_date_in_session( $current_time );
			}
		}
		/**
		 * Modifying cart items prices.
		 *
		 * @param object $cart cart object.
		 *
		 * @since 1.0.0
		 */
		public static function wcfp_update_cart_item_prices( $cart ) {
			if ( 2 <= did_action( 'woocommerce_before_calculate_totals' ) ) {
				return;
			}
			if ( 0 >= count( $cart->get_cart() ) ) {
				return;
			}
			$dynamic_prices = wcfp_get_dynamic_prices_from_session();
			if ( ! $dynamic_prices ) {
				return;
			}
			foreach ( $cart->get_cart() as $cart_item_key => $cart_item ) {
				$product_id = $cart_item['data']->get_id();
				$new_price  = isset( $dynamic_prices[ $product_id ] ) ? $dynamic_prices[ $product_id ] : '';
				if ( $new_price ) {
					$cart_item['data']->set_price( $new_price['dynamic_price'] );
					$cart_item['data']->set_regular_price( $new_price['dynamic_price'] );
				}
			}
		}
		/**
		 * Modifying simple product prices.
		 *
		 * @param string $price variation price.
		 * @param object $product product object.
		 *
		 * @since 1.0.0
		 */
		public static function wcfp_adjust_price( $price, $product ) {
			// If this is a Woocommerce REST API request, return the original price.
			if ( defined( 'REST_REQUEST' ) && REST_REQUEST ) {
				return $price;
			}
			$dynamic_prices = wcfp_get_dynamic_prices_from_session();
			if ( ! $dynamic_prices ) {
				return $price;
			}
			$product_id = $product->get_id();
			if ( isset( $dynamic_prices[ $product_id ] ) ) {
				return $dynamic_prices[ $product_id ]['dynamic_price'];
			}
			return $price;
		}
		/**
		 * Modifying variation product prices.
		 *
		 * @param string $price variation price.
		 * @param object $variation variation object.
		 * @param object $product product object.
		 *
		 * @since 1.0.0
		 */
		public static function wcfp_variation_price( $price, $variation, $product ) {
			if ( defined( 'REST_REQUEST' ) && REST_REQUEST ) {
				return $price;
			}
			$dynamic_prices = wcfp_get_dynamic_prices_from_session();
			if ( ! $dynamic_prices ) {
				return $price;
			}
			$product_id = $product->get_id();
			if ( isset( $dynamic_prices[ $product_id ] ) ) {
				return $dynamic_prices[ $product_id ]['dynamic_price'];
			}
			return $price;
		}
	}
	new WCFP_Pricing_Manager();
}
