<?php
/**
 * Handle API requests.
 *
 * @since 1.0.0
 * @package WCFP_API_Client
 */

defined( 'ABSPATH' ) || exit;
if ( ! class_exists( 'WCFP_API_Client' ) ) {
	/**
	 * REST API configurations.
	 *
	 * @since 1.0.0
	 */
	class WCFP_API_Client {
		/**
		 * Initializing REST API.
		 *
		 * @since 1.0.0
		 */
		public function __construct() {
			add_action( 'rest_api_init', array( __CLASS__, 'wcfp_rest_result' ) );
		}
		/**
		 * Endpoints restriction.
		 *
		 * @param object $request an object of endpoints.
		 * @since 1.0.0
		 */
		public static function wcfp_has_access_to_custom_fp_endpoint( $request ) {
			$store_api_key = get_option( 'wcfp_api_key', '' );

			if ( empty( $store_api_key ) ) {
				return new WP_Error( 'rest_forbidden', 'Missing key', array( 'status' => 401 ) );
			}
			// Just a smoke check, no store api key should be less than 30 characters.
			if ( strlen( $store_api_key ) < 30 ) {
				return new WP_Error( 'rest_forbidden', 'Invalid key', array( 'status' => 401 ) );
			}
			$header_store_api_key = $request->get_header( 'X-Store-API-Key' );
			if ( empty( $header_store_api_key ) ) {
				return new WP_Error( 'rest_forbidden', 'Missing auth header', array( 'status' => 401 ) );
			}

			return $store_api_key === $header_store_api_key;
		}
		/**
		 * Registered some endpoints.
		 *
		 * @since 1.0.0
		 */
		public static function wcfp_rest_result() {
			register_rest_route(
				'fp-api/v1',
				'/all-products-with-variants',
				array(
					'methods'             => 'GET',
					'callback'            => array( __CLASS__, 'wcfp_get_all_products_with_variants' ),
					'permission_callback' => array( __CLASS__, 'wcfp_has_access_to_custom_fp_endpoint' ),
				)
			);
			register_rest_route(
				'fp-api/v1',
				'/all-orders',
				array(
					'methods'             => 'GET',
					'callback'            => array( __CLASS__, 'wcfp_get_all_orders' ),
					'permission_callback' => array( __CLASS__, 'wcfp_has_access_to_custom_fp_endpoint' ),
				)
			);
			register_rest_route(
				'fp-api/v1',
				'/test-connection',
				array(
					'methods'             => 'GET',
					'callback'            => array( __CLASS__, 'wcfp_test_rust_api_connection' ),
					'permission_callback' => array( __CLASS__, 'wcfp_has_access_to_custom_fp_endpoint' ),
				)
			);
		}
		/**
		 * Get all products data.
		 *
		 * @since 1.0.0
		 */
		public static function wcfp_get_all_products_with_variants() {
			$products = wc_get_products(
				array(
					'status' => 'publish',
				 	'limit'  => -1,
				)
			);
			$data     = array();
			foreach ( $products as $product ) {
				$product_data = $product->get_data();
				if ( $product->is_type( 'variable' ) ) {
					$variants                 = wc_get_products(
						array(
							'status' => 'publish',
							'type'   => 'variation',
							'parent' => $product->get_id(),
							'limit'  => -1,
						)
					);
					$product_data['variants'] = array_map(
						function ( $variant ) {
							return $variant->get_data();
						},
						$variants
					);
				}
				$data[] = $product_data;
			}
			return new WP_REST_Response( $data, 200 );
		}
		/**
		 * Retrieving all WooCommerce orders.
		 *
		 * @since 1.0.0
		 */
		public static function wcfp_get_all_orders() {
			$orders     = wc_get_orders( array( 'limit' => -1 ) );
			$order_data = array();
			foreach ( $orders as $order ) {
				$data      = $order->get_data();
				$item_data = array();
				foreach ( $order->get_items() as $item_id => $item ) {
					$item_data[ $item_id ]['product_id']   = $item->get_product_id();
					$item_data[ $item_id ]['variation_id'] = $item->get_variation_id();
					$item_data[ $item_id ]['product_name'] = $item->get_name();
					$item_data[ $item_id ]['quantity']     = $item->get_quantity();
					$item_data[ $item_id ]['subtotal']     = $item->get_subtotal();
					$item_data[ $item_id ]['total']        = $item->get_total();
					$item_data[ $item_id ]['tax']          = $item->get_subtotal_tax();
					$item_data[ $item_id ]['tax_class']    = $item->get_tax_class();
					$item_data[ $item_id ]['tax_status']   = $item->get_tax_status();
					$item_data[ $item_id ]['product_meta'] = $item->get_meta_data();
				}
				$order_data[] = array(
					'items_data' => $item_data,
					'order_data' => $data,
				);
			}
			return new WP_REST_Response( $order_data, 200 );
		}
		/**
		 * Fetching dynamic prices based on rulesets.
		 *
		 * @since 1.0.0
		 */
		public function wcfp_fetch_dynamic_prices() {
			$ip_address = get_option( 'wcfp_test_ip_address' );
			if ( empty( $ip_address ) ) {
				$ip_address = WC_Geolocation::get_ip_address();
			}
			$url = get_option( 'wcfp_pricing_api_url' );
			if ( ! empty( $url ) ) {
				$response = wp_remote_post(
					$url,
					array(
						'headers' => array(
							'X-Store-API-Key' => get_option( 'wcfp_api_key', '' ),
							'Content-Type'    => 'application/json',
						),
						'body'    => wp_json_encode(
							array(
								'shopper_ip_address' => $ip_address,
								'shopper_user_agent' => ! empty( $_SERVER['HTTP_USER_AGENT'] ) ? sanitize_text_field( wp_unslash( $_SERVER['HTTP_USER_AGENT'] ) ) : '',

							)
						),
					)
				);
				if ( is_wp_error( $response ) ) {
					return array();
				}
				$body = wp_remote_retrieve_body( $response );
				return json_decode( $body, true );
			} else {
				return array();
			}
		}
		/**
		 * Testing RUST API connection.
		 *
		 * @since 1.0.0
		 */
		public static function wcfp_test_rust_api_connection() {
			$result = wcfp_check_connection_to_rust_api();
			if ( $result ) {
				$data = array( 'connection_status' => 'success' );
				return new WP_REST_Response( $data, 200 );
			} else {
				$data = array( 'connection_status' => 'failed' );
				return new WP_REST_Response( $data, 200 );
			}
		}
	}
}
