<?php
if ( ! defined( 'ABSPATH' ) ) exit;

class FLASHFOLDERS_REST {
    const NAMESPACE = 'flashfolders/v1';

    public static function init() {
        add_action( 'rest_api_init', array( __CLASS__, 'register_routes' ) );
    }

    public static function register_routes() {
        // Media routes
        register_rest_route( self::NAMESPACE, '/media', array(
            'methods' => 'GET',
            'callback' => array( __CLASS__, 'get_media' ),
            'permission_callback' => array( __CLASS__, 'check_permissions' ),
        ) );

        register_rest_route( self::NAMESPACE, '/media/(?P<id>\d+)', array(
            'methods' => 'GET',
            'callback' => array( __CLASS__, 'get_media_item' ),
            'permission_callback' => array( __CLASS__, 'check_permissions' ),
        ) );

        register_rest_route( self::NAMESPACE, '/media/(?P<id>\d+)', array(
            'methods' => 'DELETE',
            'callback' => array( __CLASS__, 'delete_media' ),
            'permission_callback' => array( __CLASS__, 'check_permissions' ),
        ) );

        register_rest_route( self::NAMESPACE, '/media/(?P<id>\d+)/replace', array(
            'methods' => 'POST',
            'callback' => array( __CLASS__, 'replace_media' ),
            'permission_callback' => array( __CLASS__, 'check_permissions' ),
        ) );

        register_rest_route( self::NAMESPACE, '/media/(?P<id>\d+)/usage', array(
            'methods' => 'GET',
            'callback' => array( __CLASS__, 'get_media_usage' ),
            'permission_callback' => array( __CLASS__, 'check_permissions' ),
        ) );

        register_rest_route( self::NAMESPACE, '/media/(?P<id>\d+)/folder', array(
            'methods' => 'PUT',
            'callback' => array( __CLASS__, 'assign_folder' ),
            'permission_callback' => array( __CLASS__, 'check_permissions' ),
        ) );

        // Folder routes
        register_rest_route( self::NAMESPACE, '/folders', array(
            'methods' => 'GET',
            'callback' => array( __CLASS__, 'get_folders' ),
            'permission_callback' => array( __CLASS__, 'check_permissions' ),
        ) );

        register_rest_route( self::NAMESPACE, '/folders', array(
            'methods' => 'POST',
            'callback' => array( __CLASS__, 'create_folder' ),
            'permission_callback' => array( __CLASS__, 'check_permissions' ),
        ) );

        register_rest_route( self::NAMESPACE, '/folders/(?P<id>\d+)', array(
            'methods' => 'PUT',
            'callback' => array( __CLASS__, 'update_folder' ),
            'permission_callback' => array( __CLASS__, 'check_permissions' ),
        ) );

        register_rest_route( self::NAMESPACE, '/folders/(?P<id>\d+)', array(
            'methods' => 'DELETE',
            'callback' => array( __CLASS__, 'delete_folder' ),
            'permission_callback' => array( __CLASS__, 'check_permissions' ),
        ) );

        register_rest_route( self::NAMESPACE, '/folders/reorder', array(
            'methods' => 'POST',
            'callback' => array( __CLASS__, 'reorder_folders' ),
            'permission_callback' => array( __CLASS__, 'check_permissions' ),
        ) );
    }

    public static function check_permissions() {
        return current_user_can( 'upload_files' );
    }

    // Media endpoints
    public static function get_media( $request ) {
        $params = array(
            'posts_per_page' => $request->get_param( 'per_page' ) ?: 50,
            'paged' => $request->get_param( 'page' ) ?: 1,
            'search' => $request->get_param( 'search' ),
            'mime_type' => $request->get_param( 'mime_type' ),
            'orderby' => $request->get_param( 'orderby' ) ?: 'date',
            'order' => $request->get_param( 'order' ) ?: 'DESC',
        );

        $folder_id = $request->get_param( 'folder_id' );
        if ( $folder_id !== null ) {
            $params['folder_id'] = intval( $folder_id );
        }


        return rest_ensure_response( FLASHFOLDERS_Media::get_media( $params ) );
    }

    public static function get_media_item( $request ) {
        $item = FLASHFOLDERS_Media::get_media_item( $request['id'] );
        if ( ! $item ) {
            return new WP_Error( 'not_found', __( 'Media not found', 'flashfolders' ), array( 'status' => 404 ) );
        }
        return rest_ensure_response( $item );
    }

    public static function delete_media( $request ) {
        $result = FLASHFOLDERS_Media::delete_media( $request['id'], true );
        if ( ! $result ) {
            return new WP_Error( 'delete_failed', __( 'Could not delete', 'flashfolders' ) );
        }
        return rest_ensure_response( array( 'success' => true ) );
    }

    public static function replace_media( $request ) {
        $files = $request->get_file_params();
        if ( empty( $files['file'] ) ) {
            return new WP_Error( 'no_file', __( 'No file uploaded', 'flashfolders' ) );
        }

        $result = FLASHFOLDERS_Replace::replace_file( $request['id'], $files['file'] );
        
        if ( is_wp_error( $result ) ) {
            return $result;
        }

        return rest_ensure_response( $result );
    }

    public static function get_media_usage( $request ) {
        FLASHFOLDERS_Usage::clear_cache( $request['id'] ); // ← AGGIUNGI QUESTA RIGA
        $usage = FLASHFOLDERS_Usage::get_usage( $request['id'] );
        return rest_ensure_response( $usage );
    }

    public static function assign_folder( $request ) {
        $folder_id = $request->get_param( 'folder_id' );
        
        // Converte a int (anche 0 è valido!)
        $folder_id = intval( $folder_id );
        
        
        $result = FLASHFOLDERS_Folders::assign_media_to_folder( $request['id'], $folder_id );
        
        if ( ! $result ) {
            return new WP_Error( 'assign_failed', __( 'Could not assign folder', 'flashfolders' ) );
        }


        return rest_ensure_response( array( 
            'success' => true,
            'folder_id' => $folder_id,
            'media_id' => $request['id']
        ) );
    }

    // Folder endpoints
    public static function get_folders( $request ) {
        $tree = $request->get_param( 'tree' );
        
        if ( $tree ) {
            $folders = FLASHFOLDERS_Folders::get_folder_tree();
        } else {
            $folders = FLASHFOLDERS_Folders::get_folders();
        }

        return rest_ensure_response( $folders );
    }

    public static function create_folder( $request ) {
        $name = $request->get_param( 'name' );
        $parent = $request->get_param( 'parent' ) ?: 0;

        if ( empty( $name ) ) {
            return new WP_Error( 'no_name', __( 'Folder name required', 'flashfolders' ) );
        }

        $folder = FLASHFOLDERS_Folders::create_folder( $name, $parent );
        
        if ( is_wp_error( $folder ) ) {
            return $folder;
        }

        return rest_ensure_response( $folder );
    }

    public static function update_folder( $request ) {
        $data = array(
            'name' => $request->get_param( 'name' ),
            'parent' => $request->get_param( 'parent' ),
        );

        $folder = FLASHFOLDERS_Folders::update_folder( $request['id'], $data );
        
        if ( is_wp_error( $folder ) ) {
            return $folder;
        }

        return rest_ensure_response( $folder );
    }

    public static function delete_folder( $request ) {
        $result = FLASHFOLDERS_Folders::delete_folder( $request['id'] );
        
        if ( is_wp_error( $result ) || ! $result ) {
            return new WP_Error( 'delete_failed', __( 'Could not delete folder', 'flashfolders' ) );
        }

        return rest_ensure_response( array( 'success' => true ) );
    }

    public static function reorder_folders( $request ) {
        $folder_ids = $request->get_param( 'folder_ids' );
        
        if ( empty( $folder_ids ) || ! is_array( $folder_ids ) ) {
            return new WP_Error( 'invalid_data', __( 'Invalid folder order data', 'flashfolders' ) );
        }

        $result = FLASHFOLDERS_Folders::reorder_folders( $folder_ids );
        
        if ( ! $result ) {
            return new WP_Error( 'reorder_failed', __( 'Could not reorder folders', 'flashfolders' ) );
        }

        return rest_ensure_response( array( 'success' => true ) );
    }
}