<?php
if ( ! defined( 'ABSPATH' ) ) exit;

class FLASHFOLDERS_Media {
    
    public static function get_media( $args = array() ) {
        global $wpdb;

        $defaults = array(
            'post_type' => 'attachment',
            'post_status' => 'inherit',
            'posts_per_page' => 50,
            'paged' => 1,
            'orderby' => 'date',
            'order' => 'DESC',
            'suppress_filters' => false,
        );
        $args = wp_parse_args( $args, $defaults );

        // ✅ RESET tax_query to ignore other plugins
        $args['tax_query'] = array();

        // Folder management - ONLY our taxonomy
        if ( isset( $args['folder_id'] ) ) {
            $folder_id = intval( $args['folder_id'] );
            
            if ( $folder_id > 0 ) {
                // ✅ FIX: Use include_children = false to get EXACT folder matches only
                $args['tax_query'] = array( 
                    array(
                        'taxonomy' => FLASHFOLDERS_Folders::TAXONOMY,
                        'field' => 'term_id',
                        'terms' => $folder_id,
                        'include_children' => false, // ← THIS IS THE KEY!
                    )
                );
                
            } else {
                // ROOT (folder_id = 0): show ONLY files without our taxonomy
                $media_with_folders = $wpdb->get_col( 
                    $wpdb->prepare(
                        "SELECT DISTINCT tr.object_id 
                        FROM {$wpdb->term_relationships} tr
                        INNER JOIN {$wpdb->term_taxonomy} tt ON tr.term_taxonomy_id = tt.term_taxonomy_id
                        WHERE tt.taxonomy = %s",
                        FLASHFOLDERS_Folders::TAXONOMY
                    )
                );
                
                if ( ! empty( $media_with_folders ) ) {
                    $args['post__not_in'] = $media_with_folders;
                }
            }
            unset( $args['folder_id'] );
        }

        // Search
        if ( isset( $args['search'] ) && ! empty( $args['search'] ) ) {
            $args['s'] = $args['search'];
            unset( $args['search'] );
        }

        // MIME type
        if ( isset( $args['mime_type'] ) && ! empty( $args['mime_type'] ) ) {
            $args['post_mime_type'] = $args['mime_type'];
            unset( $args['mime_type'] );
        }
        
        $query = new WP_Query( $args );

        return array(
            'items' => array_map( array( __CLASS__, 'format_media_item' ), $query->posts ),
            'total' => $query->found_posts,
            'total_pages' => $query->max_num_pages,
            'current_page' => $args['paged'],
        );
    }

    /**
     * Get single media item
     */
    public static function get_media_item( $attachment_id ) {
        $attachment = get_post( $attachment_id );
        if ( ! $attachment || $attachment->post_type !== 'attachment' ) {
            return null;
        }
        return self::format_media_item( $attachment );
    }

    /**
     * Format media item for output
     * ✅ FIX: Full support for PDF previews
     */
    public static function format_media_item( $attachment ) {
        $metadata = wp_get_attachment_metadata( $attachment->ID );
        $file_path = get_attached_file( $attachment->ID );
        $file_size = $file_path && file_exists( $file_path ) ? filesize( $file_path ) : 0;
        
        // ✅ Get folder ONLY from our taxonomy
        $folder = FLASHFOLDERS_Folders::get_media_folder( $attachment->ID );
        
        // ✅ CACHE BUSTING: File version to force browser refresh
        $version = get_post_meta( $attachment->ID, '_flashfolder_file_version', true );
        
        // If no saved version, use file modification timestamp
        if ( empty( $version ) && $file_path && file_exists( $file_path ) ) {
            $version = filemtime( $file_path );
        }
        
        // Cache busting parameter
        $cache_buster = $version ? '?v=' . $version : '';
        
        // ✅ THUMBNAIL: Try wp_get_attachment_image_url first (for images)
        $thumbnail = wp_get_attachment_image_url( $attachment->ID, 'thumbnail' );
        
        // If no thumbnail (e.g. PDF, video, documents), try wp_get_attachment_thumb_url
        // This function also works for PDFs and other non-image files
        if ( ! $thumbnail ) {
            $thumbnail = wp_get_attachment_thumb_url( $attachment->ID );
        }
        
        // Add cache buster if thumbnail exists
        if ( $thumbnail ) {
            $thumbnail .= $cache_buster;
        }
        
        // Medium size (mainly for images)
        $medium = wp_get_attachment_image_url( $attachment->ID, 'medium' );
        if ( $medium ) {
            $medium .= $cache_buster;
        }
        
        return array(
            'id' => $attachment->ID,
            'title' => $attachment->post_title,
            'filename' => basename( $file_path ),
            'url' => wp_get_attachment_url( $attachment->ID ) . $cache_buster,
            'mime_type' => $attachment->post_mime_type,
            'size' => $file_size,
            'size_human' => size_format( $file_size ),
            'width' => isset( $metadata['width'] ) ? $metadata['width'] : null,
            'height' => isset( $metadata['height'] ) ? $metadata['height'] : null,
            'date' => $attachment->post_date,
            'modified' => $attachment->post_modified,
            'alt' => get_post_meta( $attachment->ID, '_wp_attachment_image_alt', true ),
            'folder_id' => $folder ? $folder->term_id : null,
            'folder_name' => $folder ? $folder->name : null,
            'thumbnail' => $thumbnail, // ✅ Now includes PDF previews
            'medium' => $medium,
        );
    }

    /**
     * Delete media
     */
    public static function delete_media( $attachment_id, $force = false ) {
        if ( class_exists( 'FLASHFOLDERS_Usage' ) ) {
            FLASHFOLDERS_Usage::clear_cache( $attachment_id );
        }

        return wp_delete_attachment( $attachment_id, $force );
    }
    
    /**
     * Direct SQL query to count media in a folder
     * Useful for debugging and performance
     */
    public static function count_media_in_folder( $folder_id ) {
        global $wpdb;
        
        if ( $folder_id > 0 ) {
            // Count in specific folder
            $count = $wpdb->get_var( $wpdb->prepare(
                "SELECT COUNT(DISTINCT p.ID)
                FROM {$wpdb->posts} p
                INNER JOIN {$wpdb->term_relationships} tr ON p.ID = tr.object_id
                INNER JOIN {$wpdb->term_taxonomy} tt ON tr.term_taxonomy_id = tt.term_taxonomy_id
                WHERE p.post_type = 'attachment'
                AND p.post_status = 'inherit'
                AND tt.taxonomy = %s
                AND tt.term_id = %d",
                FLASHFOLDERS_Folders::TAXONOMY,
                $folder_id
            ) );
        } else {
            // Count in root (without folders)
            $count = $wpdb->get_var( $wpdb->prepare(
                "SELECT COUNT(DISTINCT p.ID)
                FROM {$wpdb->posts} p
                LEFT JOIN {$wpdb->term_relationships} tr ON p.ID = tr.object_id
                LEFT JOIN {$wpdb->term_taxonomy} tt ON (tr.term_taxonomy_id = tt.term_taxonomy_id AND tt.taxonomy = %s)
                WHERE p.post_type = 'attachment'
                AND p.post_status = 'inherit'
                AND tt.term_id IS NULL",
                FLASHFOLDERS_Folders::TAXONOMY
            ) );
        }
        
        return intval( $count );
    }
    
    /**
     * Debug: Get info about all active filters
     */
    public static function debug_active_filters() {
        global $wp_filter;
        
        $hooks = array( 'posts_where', 'posts_join', 'posts_groupby', 'posts_orderby', 'pre_get_posts' );
        
        foreach ( $hooks as $hook ) {
            if ( isset( $wp_filter[$hook] ) ) {
                foreach ( $wp_filter[$hook]->callbacks as $priority => $callbacks ) {
                    foreach ( $callbacks as $callback ) {
                        $function_name = self::get_callback_name( $callback['function'] );
                    }
                }
            }
        }
    }
    
    /**
     * Helper to get readable name of a callback
     */
    private static function get_callback_name( $callback ) {
        if ( is_string( $callback ) ) {
            return $callback;
        }
        if ( is_array( $callback ) ) {
            $class = is_object( $callback[0] ) ? get_class( $callback[0] ) : $callback[0];
            return $class . '::' . $callback[1];
        }
        if ( is_object( $callback ) && ( $callback instanceof Closure ) ) {
            return 'Closure';
        }
        return 'Unknown';
    }
}