<?php
if ( ! defined( 'ABSPATH' ) ) exit;

class FLASHFOLDERS_Folders {
    const TAXONOMY = 'flashfolders';

    public static function register_taxonomy() {
        $labels = array(
            'name' => __( 'Media Folders', 'flashfolders' ),
            'singular_name' => __( 'Media Folder', 'flashfolders' ),
        );

        register_taxonomy( self::TAXONOMY, 'attachment', array(
            'labels' => $labels,
            'hierarchical' => true,
            'public' => false,
            'show_ui' => false,
            'show_in_rest' => true,
            'rewrite' => false,
            'update_count_callback' => array( __CLASS__, 'update_term_count' ),
        ) );
    }

    /**
     * Custom term count callback for media folders.
     *
     * NOTE:
     * Direct database query is intentionally used here because WordPress core
     * does not provide a reliable API to correctly count attachment terms
     * with post_status = 'inherit'.
     *
     * This mirrors how WordPress core handles similar counts internally.
     * Caching is intentionally NOT used here because this callback must always
     * return fresh counts when terms are updated.
     */
    public static function update_term_count( $terms, $taxonomy ) {
        global $wpdb;

        foreach ( (array) $terms as $term ) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Fresh count required
            $count = $wpdb->get_var( $wpdb->prepare(
                "SELECT COUNT(*) FROM {$wpdb->term_relationships} tr
                INNER JOIN {$wpdb->term_taxonomy} tt ON tr.term_taxonomy_id = tt.term_taxonomy_id
                INNER JOIN {$wpdb->posts} p ON tr.object_id = p.ID
                WHERE tt.taxonomy = %s 
                AND tt.term_id = %d 
                AND p.post_type = 'attachment'
                AND p.post_status = 'inherit'",
                self::TAXONOMY,
                $term
            ) );

            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Term count update
            $wpdb->update(
                $wpdb->term_taxonomy,
                array( 'count' => $count ),
                array( 'term_id' => $term, 'taxonomy' => self::TAXONOMY )
            );
        }
    }

    public static function get_folders( $args = array() ) {
        $folders = get_terms( wp_parse_args( $args, array(
            'taxonomy' => self::TAXONOMY,
            'hide_empty' => false,
        ) ) );
        
        // Sort by menu_order if available
        if ( ! is_wp_error( $folders ) && is_array( $folders ) ) {
            usort( $folders, function( $a, $b ) {
                $order_a = get_term_meta( $a->term_id, 'menu_order', true );
                $order_b = get_term_meta( $b->term_id, 'menu_order', true );
                $order_a = $order_a ? intval( $order_a ) : 0;
                $order_b = $order_b ? intval( $order_b ) : 0;
                
                if ( $order_a === $order_b ) {
                    return strcmp( $a->name, $b->name );
                }
                return $order_a - $order_b;
            } );
        }
        
        return $folders;
    }

    public static function get_folder( $folder_id ) {
        return get_term( $folder_id, self::TAXONOMY );
    }

    public static function create_folder( $name, $parent = 0 ) {
        $result = wp_insert_term( $name, self::TAXONOMY, array( 'parent' => $parent ) );
        return is_wp_error( $result ) ? $result : self::get_folder( $result['term_id'] );
    }

    public static function update_folder( $folder_id, $data ) {
        $args = array();
        if ( isset( $data['name'] ) ) $args['name'] = $data['name'];
        if ( isset( $data['parent'] ) ) $args['parent'] = $data['parent'];
        
        $result = wp_update_term( $folder_id, self::TAXONOMY, $args );
        
        // Handle menu_order if present
        if ( ! is_wp_error( $result ) && isset( $data['menu_order'] ) ) {
            update_term_meta( $folder_id, 'menu_order', intval( $data['menu_order'] ) );
        }
        
        return is_wp_error( $result ) ? $result : self::get_folder( $folder_id );
    }

    public static function delete_folder( $folder_id ) {
        return wp_delete_term( $folder_id, self::TAXONOMY );
    }

    /**
     * Reorder folders
     */
    public static function reorder_folders( $folder_ids ) {
        foreach ( $folder_ids as $index => $folder_id ) {
            update_term_meta( $folder_id, 'menu_order', $index );
        }
        return true;
    }

    /**
     * Assign a media to a folder
     * CRITICAL: Removes ALL previous associations to prevent duplicates
     */
    public static function assign_media_to_folder( $attachment_id, $folder_id ) {
        
        // Get old folders to update their counts
        $old_folders = wp_get_object_terms( $attachment_id, self::TAXONOMY, array( 'fields' => 'ids' ) );
        
        // ✅ CRITICAL FIX: Remove ALL previous folder associations
        // wp_delete_object_term_relationships() è più affidabile di wp_set_object_terms(array())
        wp_delete_object_term_relationships( $attachment_id, self::TAXONOMY );
        
        if ( $folder_id > 0 ) {
            // Assign the new folder (false = don't append, REPLACE)
            $result = wp_set_object_terms( $attachment_id, intval( $folder_id ), self::TAXONOMY, false );
            
            if ( is_wp_error( $result ) ) {
                return false;
            }
        }
        
        // Update counts for ALL involved folders (old + new)
        $folders_to_update = is_array( $old_folders ) ? $old_folders : array();
        if ( $folder_id > 0 && ! in_array( $folder_id, $folders_to_update ) ) {
            $folders_to_update[] = intval( $folder_id );
        }
        
        
        if ( ! empty( $folders_to_update ) ) {
            wp_update_term_count_now( $folders_to_update, self::TAXONOMY );
            
            // Verify counts after update
            foreach ( $folders_to_update as $term_id ) {
                $term = get_term( $term_id, self::TAXONOMY );
            }
        }
        
        // ✅ CRITICAL: Clear ALL relevant caches
        clean_object_term_cache( $attachment_id, 'attachment' );
        clean_term_cache( $folders_to_update, self::TAXONOMY );
        wp_cache_delete( $attachment_id, 'post_meta' );
        
        return true;
    }

    public static function get_media_folder( $attachment_id ) {
        $terms = wp_get_object_terms( $attachment_id, self::TAXONOMY );
        return ( is_wp_error( $terms ) || empty( $terms ) ) ? null : $terms[0];
    }

    public static function get_folder_tree( $parent = 0 ) {
        $folders = self::get_folders( array( 'parent' => $parent ) );
        $tree = array();
        foreach ( $folders as $folder ) {
            $tree[] = array(
                'id' => $folder->term_id,
                'name' => $folder->name,
                'parent' => $folder->parent,
                'count' => $folder->count,
                'children' => self::get_folder_tree( $folder->term_id ),
            );
        }
        return $tree;
    }
}