<?php // phpcs:disable
/**
 * Fizen Payment Gateway.
 *
 * Provides a Fizen Payment Gateway.
 *
 * @class       Fizen_Pay
 * @extends     WC_Payment_Gateway
 * @since       1.0.0
 * @package     WooCommerce/Classes/Payment
 * @author      WooThemes
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Fizen_Pay Class.
 */
class Fizen_Pay extends WC_Payment_Gateway {

	/** @var bool Whether or not logging is enabled */
	public static $log_enabled = false;

	/** @var WC_Logger Logger instance */
	public static $log = false;

	/**
	 * Constructor for the gateway.
	 */
	public function __construct() {
		$this->id                 = 'fizen';
		$this->has_fields         = false;
		$this->order_button_text  = __( 'Complete Order', 'fizen' );
		$this->method_title       = __( 'Fizen Pay - Crypto Payment Gateway', 'fizen' );
		$this->method_description = '<p>' .
			// translators: Introduction text at top of Fizen Payment Gateway settings page.
			__( 'Accept cryptocurrencies from your customers securely and seemlessly with just a few click setup.', 'fizen' )
			. '</p><p>' .
			sprintf(
				// translators: Introduction text at top of Fizen Payment Gateway settings page. Includes external URL.
				__( 'You can access your existing Fizen Pay account, or set up a new one at: %s', 'fizen' ),
				'<a target="_blank" href="https://merchant.fizen.io/">https://merchant.fizen.io/</a>'
			);

		// Timeout after 3 days. Default to 3 days as pending Bitcoin txns
		// are usually forgotten after 2-3 days.
		$this->timeout = ( new WC_DateTime() )->sub( new DateInterval( 'P3D' ) );

		// Load the settings.
		$this->init_form_fields();
		$this->init_settings();

		// Define user set variables.
		$this->title       = $this->get_option( 'title' );
		$this->description = $this->get_option( 'description' );
		$this->debug       = 'yes' === $this->get_option( 'debug', 'no' );

		self::$log_enabled = $this->debug;

		add_action( 'woocommerce_update_options_payment_gateways_' . $this->id, array( $this, 'process_admin_options' ) );
		add_filter( 'woocommerce_order_data_store_cpt_get_orders_query', array( $this, '_custom_query_var' ), 10, 2 );
		add_action( 'woocommerce_api_fizen_pay', array( $this, 'handle_webhook' ) );
	}

	/**
	 * Logging method.
	 *
	 * @param string $message Log message.
	 * @param string $level   Optional. Default 'info'.
	 *     emergency|alert|critical|error|warning|notice|info|debug
	 */
	public static function log( $message, $level = 'info' ) {
		if ( self::$log_enabled ) {
			if ( empty( self::$log ) ) {
				self::$log = wc_get_logger();
			}
			self::$log->log( $level, $message, array( 'source' => 'fizen' ) );
		}
	}

	/**
	 * Get gateway icon.
	 * 
	 * @return string
	 */
	public function get_icon() {
		// if ( $this->get_option( 'show_icons' ) === 'no' ) {
		// 	return '';
		// }

		$image_path = plugin_dir_path( __FILE__ ) . 'assets/images';
		$icon_html  = '';
		$methods    = array( 'bitcoin', 'dai', 'ethereum', 'bnb', 'usdt', 'usdc', 'trx', 'dodge' );

		foreach ( $methods as $m ) {
			$path = realpath( $image_path . '/' . $m . '.png' );
			if ( $path && dirname( $path ) === $image_path && is_file( $path ) ) {
				$url        = WC_HTTPS::force_https_url( plugins_url( '/assets/images/' . $m . '.png', __FILE__ ) );
				$icon_html .= '<img width="26" src="' . esc_attr( $url ) . '" alt="' . esc_attr__( $m, 'coinbase' ) . '" />';
			}
		}

		return apply_filters( 'woocommerce_gateway_icon', $icon_html, $this->id );
	}

	/**
	 * Initialise Gateway Settings Form Fields.
	 */
	public function init_form_fields() {
		$this->form_fields = array(
			'enabled'        => array(
				'title'   => __( 'Enable/Disable', 'woocommerce' ),
				'type'    => 'checkbox',
				'label'   => __( 'Enable Fizen Pay', 'fizen' ),
				'default' => 'yes',
			),
			'sandbox'        => array(
				'title'   => __( 'Test mode', 'woocommerce' ),
				'type'    => 'checkbox',
				'label'   => __( 'Enable test mode', 'fizen' ),
				'default' => 'no',
				'description' => sprintf(
					__(
						'Testmode is a staging environment using testnet blockchain, so that you can confirm that your integration works correctly, and simulate transactions without moving any real money. 
						After enabling sandbox environment, you must integrate this Woocommerce shop to a Fizen Pay sandbox account. You can access your existing account, or set up a new one at: %s',
						'fizen'
					),
					esc_url( 'https://merchant-testmode.fizen.io' )
				),
			),
			'title'          => array(
				'title'       => __( 'Title', 'woocommerce' ),
				'type'        => 'text',
				'description' => __( 'This controls the title which the user sees during checkout.', 'woocommerce' ),
				'default'     => __( 'Pay with cryptocurrencies', 'fizen' ),
				'desc_tip'    => true,
			),
			'description'    => array(
				'title'       => __( 'Description', 'woocommerce' ),
				'type'        => 'text',
				'desc_tip'    => true,
				'description' => __( 'This controls the description which the user sees during checkout.', 'woocommerce' ),
				'default'     => __( 'After clicking "Complete Order", you will be redirected to Fizen Pay to make your payment securely', 'fizen' ),
			),
			'api_key'        => array(
				'title'       => __( 'API Key', 'fizen' ),
				'type'        => 'text',
				'default'     => '',
				'description' => sprintf(
					__(
						'You can manage your API keys within the Fizen Payment Gateway Settings page, available here: %s',
						'fizen'
					),
					esc_url( 'https://merchant.fizen.io/dashboard/settings' )
				),

			),
			'webhook_secret' => array(
				'title'       => __( 'Webhook Shared Secret', 'fizen' ),
				'type'        => 'text',
				'description' =>

				__( 'Using webhooks allows Fizen Pay to send payment confirmation messages to the website. To fill this out:', 'fizen' )

				. '<br /><br />' .

				__( '1. In your Fizen Pay settings page, scroll to the \'Webhook subscriptions\' section', 'fizen' )

				. '<br />' .

				sprintf( __( '2. Click \'Add an endpoint\' and paste the following <strong>URL</strong>: <em>%s</em>', 'fizen' ), add_query_arg( 'wc-api', 'Fizen_Pay', home_url( '/', 'https' ) ) )

				. '<br />' .

				__( '3. Click "Show shared secret" and paste into the box above.', 'fizen' ),

			),
			'debug'          => array(
				'title'       => __( 'Debug log', 'woocommerce' ),
				'type'        => 'checkbox',
				'label'       => __( 'Enable logging', 'woocommerce' ),
				'default'     => 'no',
				'description' => sprintf( __( 'Log Fizen API events inside %s', 'fizen' ), '<code>' . WC_Log_Handler_File::get_log_file_path( 'fizen' ) . '</code>' ),
			),
		);
	}

	/**
	 * Process the payment and return the result.
	 * @param  int $order_id
	 * @return array
	 */
	public function process_payment( $order_id ) {
		global $woocommerce;
		$order = wc_get_order( $order_id );

		// Create description for charge based on order's products. Ex: 1 x Product1, 2 x Product2
		try {
			$order_items = array_map( function( $item ) {
				return $item['quantity'] . ' x ' . $item['name'];
			}, $order->get_items() );

			$description = mb_substr( implode( ', ', $order_items ), 0, 200 );
		} catch ( Exception $e ) {
			$description = null;
		}

		$this->init_api();

		// Create a new charge.
		$metadata = array(
			'orderId'  => $order->get_id(),
			'orderKey' => $order->get_order_key(),
            		'source' => 'woocommerce'
		);
		$result   = Fizen_API_Handler::create_charge(
			$order->get_total(), get_woocommerce_currency(), $metadata,
			$this->get_return_url( $order ), null, $description,
			$this->get_cancel_url( $order )
		);

		if ( ! $result[0] ) {
			return array( 'result' => 'fail' );
		}

		// Remove cart
		$woocommerce->cart->empty_cart();
		
		$charge = $result[1]['data'];

		$order->update_meta_data( '_fizen_charge_id', $charge['code'] );
		$order->save();

		return array(
			'result'   => 'success',
			'redirect' => $charge['hostedUrl'],
		);
	}

	/**
	 * Get the cancel url.
	 *
	 * @param WC_Order $order Order object.
	 * @return string
	 */
	public function get_cancel_url( $order ) {
		$return_url = $order->get_cancel_order_url();

		if ( is_ssl() || get_option( 'woocommerce_force_ssl_checkout' ) == 'yes' ) {
			$return_url = str_replace( 'http:', 'https:', $return_url );
		}

		return apply_filters( 'woocommerce_get_cancel_url', $return_url, $order );
	}

	/**
	 * Check payment statuses on orders and update order statuses.
	 */
	public function check_orders() {
		$this->init_api();

		// Check the status of non-archived Fizen orders.
		$orders = wc_get_orders( array( 'fizen_archived' => false, 'status'   => array( 'wc-pending' ) ) );
		foreach ( $orders as $order ) {
			$charge_id = $order->get_meta( '_fizen_charge_id' );
			self::log( 'Check order with charge_id = ' . print_r( $charge_id, true ) );
			usleep( 300000 );  // Ensure we don't hit the rate limit.
			$result = Fizen_API_Handler::send_request( '/api/web-app-public/v1/charges/code/' . $charge_id );

			if ( ! $result[0] ) {
				self::log( 'Failed to fetch order updates for: ' . $order->get_id() );
				continue;
			}

			$timeline = $result[1]['data']['timeline'];
			$this->_update_order_status( $order, $timeline );
		}
	}

	/**
	 * Handle requests sent to webhook.
	 */
	public function handle_webhook() {
		$payload = file_get_contents( 'php://input' );
		if ( ! empty( $payload ) && $this->validate_webhook( $payload ) ) {
			$data       = json_decode( $payload, true );
			$event_data = $data['charge'];

			self::log( 'Webhook received event: ' . print_r( $data, true ) );

			if ( ! isset( $event_data['metadata']['orderId'] ) ) {
				exit;
			}

			$order_id = $event_data['metadata']['orderId'];
			if ( wc_get_order( $order_id ) ) {
				$this->_update_order_status( wc_get_order( $order_id ), $event_data['timeline'] );
			}

			exit;  // 200 response for acknowledgement.
		}

		wp_die( 'Fizen Webhook Request Failure', 'Fizen Webhook', array( 'response' => 500 ) );
	}

	/**
	 * Check Fizen webhook request is valid.
	 * @param  string $payload
	 */
	public function validate_webhook( $payload ) {
		self::log( 'Checking Webhook response is valid' );

		if ( ! isset( $_SERVER['HTTP_X_FP_WEBHOOK_SIGNATURE'] ) ) {
			return false;
		}

		$sig    = sanitize_text_field($_SERVER['HTTP_X_FP_WEBHOOK_SIGNATURE']);
		$secret = $this->get_option( 'webhook_secret' );

		$sig2 = hash_hmac( 'sha256', $payload, $secret );

		self::log( 'Signature from payload: ' . print_r( $sig, true ));
		self::log( 'Signature from hash: ' . print_r( $sig2, true ));

		if ( $sig === $sig2 ) {
			return true;
		}

		return false;
	}

	/**
	 * Init the API class and set the API key etc.
	 */
	protected function init_api() {
		include_once dirname( __FILE__ ) . '/includes/fizen-api-handler.php';

		Fizen_API_Handler::$log     = get_class( $this ) . '::log';
		Fizen_API_Handler::$api_key = $this->get_option( 'api_key' );
		$isSandBox = 'yes' === $this->get_option( 'sandbox', 'no' );
		if ($isSandBox) {
			Fizen_API_Handler::$api_url = 'https://pay-api-sandbox.fizen.io';
		} else {
			Fizen_API_Handler::$api_url = 'https://pay-api.fizen.io';
		}
	}

	/**
	 * Update the status of an order from a given timeline.
	 * @param  WC_Order $order
	 * @param  array    $timeline
	 */
	public function _update_order_status( $order, $timeline ) {
		$prev_status = $order->get_meta( '_fizen_status' );

		$last_update = end( $timeline );
		$status      = $last_update['status'];
		if ( $status !== $prev_status ) {
			$order->update_meta_data( '_fizen_status', $status );

			if ( 'EXPIRED' === $status && 'pending' == $order->get_status() ) {
				$order->update_status( 'cancelled', __( 'Fizen Pay payment link is expired.', 'fizen' ) );
			} elseif ( 'CANCELED' === $status ) {
				$order->update_status( 'cancelled', __( 'Fizen Pay payment link is cancelled by users.', 'fizen' ) );
			} elseif ( str_contains($status, 'UNRESOLVED') ) {
			    	if ($last_update['context'] === 'OVERPAID') {
                    			$order->update_status( 'processing', __( 'Payment was successfully received via Fizen Pay.', 'fizen' ) );
                    			$order->payment_complete();
                		} else {
                    			// translators: Fizen error status for "unresolved" payment. Includes error status.
                    			$order->update_status( 'failed', sprintf( __( 'Payment received via Fizen Pay is unresolved, reason: %s.', 'fizen' ), $last_update['context'] ) );
                		}
			} elseif ( 'PENDING' === $status ) {
				$order->update_status( 'blockchainpending', __( 'Payment detected via Fizen Pay, but awaiting blockchain confirmation.', 'fizen' ) );
			} elseif ( 'RESOLVED' === $status ) {
				// We don't know the resolution, so don't change order status.
				$order->add_order_note( __( 'Payment marked as resolved by merchant on Fizen Pay.', 'fizen' ) );
            		} elseif ( 'COMPLETED' === $status ) {
                		$order->update_status( 'processing', __( 'Payment was successfully received via Fizen Pay.', 'fizen' ) );
                		$order->payment_complete();
			}
		}

		// Archive if in a resolved state and idle more than timeout.
		if ( in_array( $status, array( 'EXPIRED', 'COMPLETED', 'RESOLVED' ), true ) &&
			$order->get_date_modified() < $this->timeout ) {
			self::log( 'Archiving order: ' . $order->get_order_number() );
			$order->update_meta_data( '_fizen_archived', true );
		}
	}

	/**
	 * Handle a custom 'fizen_archived' query var to get orders
	 * payed through Fizen with the '_fizen_archived' meta.
	 * @param array $query - Args for WP_Query.
	 * @param array $query_vars - Query vars from WC_Order_Query.
	 * @return array modified $query
	 */
	public function _custom_query_var( $query, $query_vars ) {
		if ( array_key_exists( 'fizen_archived', $query_vars ) ) {
			$query['meta_query'][] = array(
				'key'     => '_fizen_archived',
				'compare' => $query_vars['fizen_archived'] ? 'EXISTS' : 'NOT EXISTS',
			);
			// Limit only to orders payed through Fizen.
			$query['meta_query'][] = array(
				'key'     => '_fizen_charge_id',
				'compare' => 'EXISTS',
			);
		}

		return $query;
	}
}
