<?php
/*
Plugin Name: Fix It Easy Security Headers
Description: Configure various security headers. Security headers enhance the security and privacy of a WordPress website by instructing the browser on how to handle various aspects of web communication. Implementing these headers helps protect against common web vulnerabilities and attacks.
Version: 1.1
Author: WP Fix It
Author URI: https://www.wpfixit.com
License: GPLv2 or later
License URI: https://www.gnu.org/licenses/gpl-2.0.html
Text Domain: fix-it-easy-security-headers
*/

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

// --- constants --------------------------------------------------------------
define( 'ESH_OPTION', 'security_headers_settings' );
define( 'ESH_SLUG',   'fix-it-easy-security-headers' );

/**
 * --- NEW: CSP nonce & tag hooks -------------------------------------------
 * Generates a per-request nonce and attaches it to all front-end enqueued
 * scripts/styles (including inline). This lets us remove 'unsafe-inline'.
 */
if ( ! function_exists('esh_get_csp_nonce') ) {
    function esh_get_csp_nonce() {
        static $nonce = null;
        if ( $nonce === null ) {
            $bytes = function_exists('random_bytes') ? random_bytes(16) : openssl_random_pseudo_bytes(16);
            // URL-safe base64, trimmed padding
            $nonce = rtrim( strtr( base64_encode($bytes), '+/', '-_' ), '=' );
        }
        return $nonce;
    }
}

add_filter('script_loader_tag', function($tag, $handle, $src){
    if (is_admin()) return $tag; // keep wp-admin stable
    if (strpos($tag, ' nonce=') === false) {
        $tag = str_replace('<script ', '<script nonce="'. esc_attr(esh_get_csp_nonce()) .'" ', $tag);
    }
    return $tag;
}, 10, 3);

add_filter('style_loader_tag', function($tag, $handle, $href){
    if (is_admin()) return $tag;
    if (strpos($tag, ' nonce=') === false) {
        $tag = str_replace('<link ', '<link nonce="'. esc_attr(esh_get_csp_nonce()) .'" ', $tag);
    }
    return $tag;
}, 10, 3);

// Ensure nonce on inline <script> printed by WP APIs
add_filter('wp_inline_script_attributes', function($attrs){
    if (is_admin()) return $attrs;
    $attrs['nonce'] = esh_get_csp_nonce();
    return $attrs;
});

// --- activation: enable all + redirect -------------------------------------
register_activation_hook( __FILE__, 'esh_activate' );
function esh_activate() {
    // enable all checkboxes
    $defaults = array(
        'strict_transport_security' => 1,
        'content_security_policy'   => 1,
        'x_frame_options'           => 1,
        'x_content_type_options'    => 1,
        'referrer_policy'           => 1,
        'permissions_policy'        => 1,
    );

    $current = get_option( ESH_OPTION );
    if ( ! is_array( $current ) || empty( $current ) ) {
        update_option( ESH_OPTION, $defaults );
    } else {
        // ensure every key is enabled without losing any existing ones
        $merged = array_merge( $current, array_fill_keys( array_keys( $defaults ), 1 ) );
        update_option( ESH_OPTION, $merged );
    }

    // trigger one-time redirect after activation
    set_transient( 'esh_do_activation_redirect', 1, 30 );
}

add_action( 'load-plugins.php', 'esh_activation_redirect' );
function esh_activation_redirect() {
    if ( ! get_transient( 'esh_do_activation_redirect' ) ) return;
    delete_transient( 'esh_do_activation_redirect' );

    // don't redirect on network/bulk activations
    if ( is_network_admin() ) return;

    wp_safe_redirect( admin_url( 'tools.php?page=' . ESH_SLUG ) );
    exit;
}

// --- admin menu -------------------------------------------------------------
add_action('admin_menu', 'security_headers_menu');
function security_headers_menu() {
    add_management_page(
        'Easy Security Headers',
        'Easy Security Headers',
        'manage_options',
        ESH_SLUG,                        // <— unify slug
        'security_headers_options_page'
    );
}

// --- options page -----------------------------------------------------------
function esh_get_scan_url() {
    return add_query_arg( 'q', home_url(), 'https://securityheaders.com/' );
}

function security_headers_options_page() { ?>
    <div class="wrap">
        <?php settings_errors('security_headers_settings'); ?>
        <form method="post" action="options.php">
            <?php
            settings_fields('security_headers_settings_group');
            do_settings_sections( ESH_SLUG );
            ?>
            <div style="display:inline-flex; gap:8px; align-items:center; margin-top:33px">
                <?php submit_button( null, 'primary', 'submit', false ); ?>
                <a class="button button-secondary" target="_blank" rel="noopener"
                   href="<?php echo esc_url( esh_get_scan_url() ); ?>">
                    <?php echo esc_html__( 'Check Headers', 'fix-it-easy-security-headers' ); ?>
                </a>
            </div>
        </form>
    </div>
<?php }

// --- plugin row links -------------------------------------------------------
add_filter('plugin_action_links_' . plugin_basename(__FILE__), 'security_headers_action_links');
function security_headers_action_links( $links ) {
    $settings_link = sprintf(
        '<a href="%s">%s</a>',
        esc_url( admin_url( 'tools.php?page=' . ESH_SLUG ) ),
        esc_html__( 'Settings', 'fix-it-easy-security-headers' )
    );

    $check_headers_link = sprintf(
        '<a href="%s" target="_blank" rel="noopener">%s</a>',
        esc_url( esh_get_scan_url() ),
        esc_html__( 'Check Headers', 'fix-it-easy-security-headers' )
    );

    array_unshift( $links, $settings_link, $check_headers_link );
    return $links;
}

// --- settings API -----------------------------------------------------------
add_action('admin_init', 'security_headers_settings');
function security_headers_settings() {
    register_setting( 'security_headers_settings_group', ESH_OPTION, 'security_headers_validate_settings' );
    add_settings_section( 'security_headers_main_section', 'Configure Easy Security Headers', 'security_headers_section_description', ESH_SLUG );
}

function security_headers_section_description() {
    echo '<p>Security headers enhance the security and privacy of a WordPress website by instructing the browser on how to handle various aspects of web communication. Implementing these headers helps protect against common web vulnerabilities and attacks.</p>';

    $headers = array(
        'strict_transport_security' => array( 'Strict-Transport-Security', 'Helps to protect websites against man-in-the-middle attacks by enforcing the use of HTTPS.' ),
        'content_security_policy'   => array( 'Content-Security-Policy', 'Prevents a variety of attacks such as cross-site scripting (XSS) and other cross-site injections.' ),
        'x_frame_options'           => array( 'X-Frame-Options', 'Protects against clickjacking attacks by controlling whether the site can be framed.' ),
        'x_content_type_options'    => array( 'X-Content-Type-Options', 'Prevents MIME type sniffing which can reduce exposure to drive-by download attacks.' ),
        'referrer_policy'           => array( 'Referrer-Policy', 'Controls how much referrer information should be included with requests.' ),
        'permissions_policy'        => array( 'Permissions-Policy', 'Controls which features and APIs can be used in the browser.' ),
    );

    foreach ( $headers as $key => $header ) {
        add_settings_field( $key, $header[0], 'security_headers_checkbox_field', ESH_SLUG, 'security_headers_main_section', array( 'key' => $key, 'description' => $header[1] ) );
    }
}

function security_headers_checkbox_field( $args ) {
    $options     = get_option( ESH_OPTION );
    $key         = isset( $args['key'] ) ? sanitize_key( $args['key'] ) : '';
    $id          = 'security_headers_' . $key;
    $description = isset( $args['description'] ) ? $args['description'] : '';

    ?>
    <label for="<?php echo esc_attr( $id ); ?>">
        <input
            type="checkbox"
            id="<?php echo esc_attr( $id ); ?>"
            name="<?php echo esc_attr( ESH_OPTION . '[' . $key . ']' ); ?>"
            value="1"
            <?php checked( 1, isset( $options[ $key ] ) ? (int) $options[ $key ] : 0 ); ?>
        />
        <?php echo esc_html( $description ); ?>
    </label>
    <?php
}

function security_headers_validate_settings( $input ) {
    // sanitize checkboxes to integers
    $clean = array();
    foreach ( (array) $input as $k => $v ) {
        $clean[ sanitize_key( $k ) ] = (int) (bool) $v;
    }
    return $clean;
}

// --- send headers -----------------------------------------------------------
add_action('send_headers', 'security_headers_add_headers');
function security_headers_add_headers() {
    $options = get_option( ESH_OPTION );

    if ( ! is_array( $options ) ) return;

    if ( ! empty( $options['strict_transport_security'] ) ) {
        // Safer HSTS; only send on HTTPS to avoid mixed-mode issues
        if ( is_ssl() ) {
            header( 'Strict-Transport-Security: max-age=31536000; includeSubDomains; preload' );
        }
    }

    if ( ! empty( $options['content_security_policy'] ) ) {
        /**
         * --- NEW: Strict CSP using per-request nonce ------------------------
         * Replaces permissive wildcards/unsafe directives with explicit allowlists.
         * Extend allowlists via filters: esh_csp_* below.
         */
        if ( ! is_admin() ) {
            $nonce = esh_get_csp_nonce();

            // Allowlist pieces (filterable)
            $script_src = apply_filters('esh_csp_script_src', array(
                "'self'",
                "'nonce-{$nonce}'",
                'https://www.googletagmanager.com',
                'https://www.google-analytics.com',
            ));

            $style_src = apply_filters('esh_csp_style_src', array(
                "'self'",
                "'nonce-{$nonce}'",
                'https://fonts.googleapis.com',
            ));

            $img_src = apply_filters('esh_csp_img_src', array(
                "'self'",
                'data:',
                'https:',
            ));

            $font_src = apply_filters('esh_csp_font_src', array(
                "'self'",
                'https://fonts.gstatic.com',
                'data:',
            ));

            $connect_src = apply_filters('esh_csp_connect_src', array(
                "'self'",
                'https:',
            ));

            $frame_src = apply_filters('esh_csp_frame_src', array(
                "'self'",
                'https://www.youtube.com',
                'https://player.vimeo.com',
            ));

            // Base policy
            $base_policy = array(
                "default-src 'self'",
                "base-uri 'self'",
                "object-src 'none'",
                "frame-ancestors 'self'",
                "form-action 'self'",
                'upgrade-insecure-requests',
            );

            $directives = array_merge(
                $base_policy,
                array(
                    'script-src '  . implode(' ', array_unique($script_src)),
                    'style-src '   . implode(' ', array_unique($style_src)),
                    'img-src '     . implode(' ', array_unique($img_src)),
                    'font-src '    . implode(' ', array_unique($font_src)),
                    'connect-src ' . implode(' ', array_unique($connect_src)),
                    'frame-src '   . implode(' ', array_unique($frame_src)),
                )
            );

            $policy = implode('; ', $directives);

            // Report-Only support via filters
            $report_only = apply_filters('esh_csp_report_only', false);
            $report_to   = apply_filters('esh_csp_report_to', '');

            if ( $report_only ) {
                if ( $report_to ) {
                    $policy .= '; report-uri ' . esc_url_raw($report_to);
                }
                header( 'Content-Security-Policy-Report-Only: ' . $policy );
            } else {
                header( 'Content-Security-Policy: ' . $policy );
            }
        } else {
            // Keep wp-admin flexible; if you want CSP there too, remove the is_admin() check above.
            // (No header sent for admin by default.)
        }
    }

    if ( ! empty( $options['x_frame_options'] ) ) {
        header( 'X-Frame-Options: SAMEORIGIN' );
    }
    if ( ! empty( $options['x_content_type_options'] ) ) {
        header( 'X-Content-Type-Options: nosniff' );
    }
    if ( ! empty( $options['referrer_policy'] ) ) {
        // Safer default than no-referrer-when-downgrade
        header( 'Referrer-Policy: strict-origin-when-cross-origin' );
    }
    if ( ! empty( $options['permissions_policy'] ) ) {
        // Tighter baseline; adjust as needed
        header( 'Permissions-Policy: geolocation=(self), microphone=(), camera=(), fullscreen=(self)' );
    }
}