<?php
/**
 * Plugin Name: First Order Coupon
 * Plugin URI: https://wordpress.org/plugins/first-order-coupon/
 * Description: Create coupons for first-time buyers using their phone number
 * Version: 1.1.1
 * Author: Ehsan Didban
 * Author URI: https://ehsandidban.ir
 * Text Domain: first-order-coupon
 * Domain Path: /languages
 * License: GPLv2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Requires at least: 5.6
 * Tested up to: 6.8
 * Requires PHP: 7.2
 * WC requires at least: 4.0
 * WC tested up to: 9.9.3
 */

if (!defined('ABSPATH')) {
    exit;
}

// Add checkbox to coupon settings (Usage Restriction tab)
add_action('woocommerce_coupon_options_usage_restriction', function ($coupon_id) {
    woocommerce_wp_checkbox([
        'id' => 'first_order_only',
        'label' => __('Apply only for user\'s first order', 'first-order-coupon'),
        'description' => __('If enabled, this coupon will only be valid for the user\'s first order.', 'first-order-coupon')
    ]);
});

// Save coupon meta when coupon is saved
add_action('woocommerce_coupon_options_save', function ($coupon_id) {
    $is_first = isset($_POST['first_order_only']) ? 'yes' : 'no';
    update_post_meta($coupon_id, 'first_order_only', $is_first);
});

// Validate coupon usage based on user's phone number and order history
add_filter('woocommerce_coupon_is_valid', function ($valid, $coupon, $discount) {
    $only_first = get_post_meta($coupon->get_id(), 'first_order_only', true);
    if ($only_first === 'yes') {
        // If the user is not logged in, prevent coupon usage
        if (!is_user_logged_in()) {
            wc_add_notice(__('This coupon is only available for logged-in users on their first order.', 'first-order-coupon'), 'error');
            return false;
        }

        // Check if the logged-in user has previous orders
        $user_id = get_current_user_id();
        $orders = wc_get_orders([
            'customer_id' => $user_id,
            'limit' => 1,
            'status' => ['completed', 'processing']
        ]);

        // If a previous order exists, invalidate the coupon
        if (!empty($orders)) {
            wc_add_notice(__('This coupon can only be used for the first order.', 'first-order-coupon'), 'error');
            return false;
        }
    }
    return $valid;
}, 10, 3);