<?php
/**
 * Admin Page Template
 *
 * @package FindMissingMoreTags
 *
 * @var string                                       $current_tab  Current active tab.
 * @var int                                          $current_page Current page number.
 * @var array{missing: int, has_tag: int, ignored: int} $counts     Post counts.
 * @var array<\WP_Post>                              $posts        Posts for current view.
 * @var int                                          $total_posts  Total posts in current tab.
 * @var int                                          $total_pages  Total pages.
 * @var \FindMissingMoreTags\AdminPage               $admin_page   Admin page instance.
 */

declare(strict_types=1);

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}
?>

<div class="wrap fmmt-wrap">
	<h1 class="fmmt-title"><?php esc_html_e( 'Find Missing More Tags', 'find-missing-more-tags' ); ?></h1>

	<!-- Stats Cards -->
	<div class="fmmt-stats">
		<div class="fmmt-card fmmt-stat-card fmmt-stat-card--warning">
			<span class="fmmt-stat-card__icon dashicons dashicons-warning"></span>
			<div class="fmmt-stat-card__content">
				<span class="fmmt-stat-card__number"><?php echo esc_html( number_format_i18n( $counts['missing'] ) ); ?></span>
				<span class="fmmt-stat-card__label"><?php esc_html_e( 'Missing More Tag', 'find-missing-more-tags' ); ?></span>
			</div>
		</div>

		<div class="fmmt-card fmmt-stat-card fmmt-stat-card--success">
			<span class="fmmt-stat-card__icon dashicons dashicons-yes-alt"></span>
			<div class="fmmt-stat-card__content">
				<span class="fmmt-stat-card__number"><?php echo esc_html( number_format_i18n( $counts['has_tag'] ) ); ?></span>
				<span class="fmmt-stat-card__label"><?php esc_html_e( 'Has More Tag', 'find-missing-more-tags' ); ?></span>
			</div>
		</div>

		<div class="fmmt-card fmmt-stat-card fmmt-stat-card--ignored">
			<span class="fmmt-stat-card__icon dashicons dashicons-hidden"></span>
			<div class="fmmt-stat-card__content">
				<span class="fmmt-stat-card__number"><?php echo esc_html( number_format_i18n( $counts['ignored'] ) ); ?></span>
				<span class="fmmt-stat-card__label"><?php esc_html_e( 'Ignored', 'find-missing-more-tags' ); ?></span>
			</div>
		</div>
	</div>

	<!-- Tabs -->
	<div class="fmmt-tabs-wrapper">
		<nav class="fmmt-tabs" role="tablist">
			<a href="<?php echo esc_url( $admin_page->get_tab_url( 'missing' ) ); ?>" class="fmmt-tab <?php echo 'missing' === $current_tab ? 'fmmt-tab--active' : ''; ?>" role="tab" aria-selected="<?php echo 'missing' === $current_tab ? 'true' : 'false'; ?>">
				<?php
				printf(
					/* translators: %s: number of posts */
					esc_html__( 'Missing More Tag (%s)', 'find-missing-more-tags' ),
					esc_html( number_format_i18n( $counts['missing'] ) )
				);
				?>
			</a>
			<a href="<?php echo esc_url( $admin_page->get_tab_url( 'has-tag' ) ); ?>" class="fmmt-tab <?php echo 'has-tag' === $current_tab ? 'fmmt-tab--active' : ''; ?>" role="tab" aria-selected="<?php echo 'has-tag' === $current_tab ? 'true' : 'false'; ?>">
				<?php
				printf(
					/* translators: %s: number of posts */
					esc_html__( 'Has More Tag (%s)', 'find-missing-more-tags' ),
					esc_html( number_format_i18n( $counts['has_tag'] ) )
				);
				?>
			</a>
			<a href="<?php echo esc_url( $admin_page->get_tab_url( 'ignored' ) ); ?>" class="fmmt-tab <?php echo 'ignored' === $current_tab ? 'fmmt-tab--active' : ''; ?>" role="tab" aria-selected="<?php echo 'ignored' === $current_tab ? 'true' : 'false'; ?>">
				<?php
				printf(
					/* translators: %s: number of posts */
					esc_html__( 'Ignored (%s)', 'find-missing-more-tags' ),
					esc_html( number_format_i18n( $counts['ignored'] ) )
				);
				?>
			</a>
		</nav>

		<!-- Posts Table -->
		<div class="fmmt-card fmmt-table-card">
			<?php if ( empty( $posts ) ) : ?>
				<div class="fmmt-empty-state">
					<?php if ( 'missing' === $current_tab ) : ?>
						<span class="dashicons dashicons-yes-alt fmmt-empty-state__icon"></span>
						<p class="fmmt-empty-state__message">
							<?php esc_html_e( 'All posts have the more tag. Nice work!', 'find-missing-more-tags' ); ?>
						</p>
					<?php elseif ( 'ignored' === $current_tab ) : ?>
						<span class="dashicons dashicons-info fmmt-empty-state__icon"></span>
						<p class="fmmt-empty-state__message">
							<?php esc_html_e( 'No posts are being ignored.', 'find-missing-more-tags' ); ?>
						</p>
					<?php else : ?>
						<span class="dashicons dashicons-info fmmt-empty-state__icon"></span>
						<p class="fmmt-empty-state__message">
							<?php esc_html_e( 'No posts have the more tag yet.', 'find-missing-more-tags' ); ?>
						</p>
					<?php endif; ?>
				</div>
			<?php else : ?>
				<table class="fmmt-table widefat striped">
					<thead>
						<tr>
							<th class="fmmt-table__col-title"><?php esc_html_e( 'Title', 'find-missing-more-tags' ); ?></th>
							<th class="fmmt-table__col-status"><?php esc_html_e( 'Status', 'find-missing-more-tags' ); ?></th>
							<th class="fmmt-table__col-author"><?php esc_html_e( 'Author', 'find-missing-more-tags' ); ?></th>
							<th class="fmmt-table__col-date"><?php esc_html_e( 'Date', 'find-missing-more-tags' ); ?></th>
							<th class="fmmt-table__col-action"><?php esc_html_e( 'Action', 'find-missing-more-tags' ); ?></th>
						</tr>
					</thead>
					<tbody>
						<?php foreach ( $posts as $fmmt_post ) : ?>
							<?php
							$fmmt_author    = get_userdata( $fmmt_post->post_author );
							$fmmt_edit_link = get_edit_post_link( $fmmt_post->ID );
							$fmmt_post_date = 'future' === $fmmt_post->post_status
								? $fmmt_post->post_date
								: ( ! empty( $fmmt_post->post_modified ) ? $fmmt_post->post_modified : $fmmt_post->post_date );
							$fmmt_title     = ! empty( $fmmt_post->post_title ) ? $fmmt_post->post_title : __( '(No title)', 'find-missing-more-tags' );
							?>
							<tr>
								<td class="fmmt-table__col-title">
									<a href="<?php echo esc_url( $fmmt_edit_link ); ?>" target="_blank" rel="noopener noreferrer" class="fmmt-post-title">
										<?php echo esc_html( $fmmt_title ); ?>
									</a>
								</td>
								<td class="fmmt-table__col-status">
									<span class="fmmt-status <?php echo esc_attr( $admin_page->get_status_class( $fmmt_post->post_status ) ); ?>">
										<?php echo esc_html( $admin_page->get_status_label( $fmmt_post->post_status ) ); ?>
									</span>
								</td>
								<td class="fmmt-table__col-author">
									<?php echo esc_html( $fmmt_author ? $fmmt_author->display_name : __( 'Unknown', 'find-missing-more-tags' ) ); ?>
								</td>
								<td class="fmmt-table__col-date">
									<?php echo esc_html( mysql2date( get_option( 'date_format' ), $fmmt_post_date ) ); ?>
								</td>
								<td class="fmmt-table__col-action">
									<a href="<?php echo esc_url( $fmmt_edit_link ); ?>" target="_blank" rel="noopener noreferrer" class="button button-small">
										<?php esc_html_e( 'Edit', 'find-missing-more-tags' ); ?>
										<span class="dashicons dashicons-external" aria-hidden="true"></span>
									</a>
									<?php if ( 'missing' === $current_tab ) : ?>
										<a href="<?php echo esc_url( $admin_page->get_ignore_url( $fmmt_post->ID ) ); ?>" class="fmmt-action-link fmmt-action-link--ignore">
											<?php esc_html_e( 'Ignore', 'find-missing-more-tags' ); ?>
										</a>
									<?php elseif ( 'ignored' === $current_tab ) : ?>
										<a href="<?php echo esc_url( $admin_page->get_include_url( $fmmt_post->ID ) ); ?>" class="fmmt-action-link fmmt-action-link--include">
											<?php esc_html_e( 'Include', 'find-missing-more-tags' ); ?>
										</a>
									<?php endif; ?>
								</td>
							</tr>
						<?php endforeach; ?>
					</tbody>
				</table>

				<?php $admin_page->render_pagination( $current_page, $total_pages, $current_tab ); ?>
			<?php endif; ?>
		</div>
	</div>
</div>
