<?php
declare(strict_types=1);
/**
 * Post Scanner Class
 *
 * @package FindMissingMoreTags
 */

namespace FindMissingMoreTags;

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Handles detection of more tags in post content.
 */
class PostScanner {

	/**
	 * Post statuses to scan.
	 *
	 * @var array<string>
	 */
	private const SCANNABLE_STATUSES = array(
		'publish',
		'draft',
		'pending',
		'future',
		'private',
	);

	/**
	 * Posts per page for pagination.
	 *
	 * @var int
	 */
	private const POSTS_PER_PAGE = 20;

	/**
	 * Meta key for ignored posts.
	 *
	 * @var string
	 */
	public const IGNORED_META_KEY = '_fmmt_ignored';

	/**
	 * Check if a post contains a more tag.
	 *
	 * Detects both Classic Editor (<!--more-->) and Block Editor formats.
	 *
	 * @param \WP_Post $post The post to check.
	 * @return bool True if post has a more tag, false otherwise.
	 */
	public function has_more_tag( \WP_Post $post ): bool {
		return strpos( $post->post_content, '<!--more-->' ) !== false;
	}

	/**
	 * Get posts missing the more tag.
	 *
	 * @param int $page Current page number.
	 * @return array{posts: array<\WP_Post>, total: int, pages: int}
	 */
	public function get_posts_missing_more_tag( int $page = 1 ): array {
		return $this->get_filtered_posts( $page, false );
	}

	/**
	 * Get posts that have the more tag.
	 *
	 * @param int $page Current page number.
	 * @return array{posts: array<\WP_Post>, total: int, pages: int}
	 */
	public function get_posts_with_more_tag( int $page = 1 ): array {
		return $this->get_filtered_posts( $page, true );
	}

	/**
	 * Get filtered posts based on more tag presence.
	 *
	 * @param int  $page        Current page number.
	 * @param bool $has_tag     Whether to get posts with or without the tag.
	 * @param bool $exclude_ignored Whether to exclude ignored posts.
	 * @return array{posts: array<\WP_Post>, total: int, pages: int}
	 */
	private function get_filtered_posts( int $page, bool $has_tag, bool $exclude_ignored = true ): array {
		// Get all post IDs first for accurate counting.
		$all_posts = $this->get_all_scannable_posts();

		// Filter based on more tag presence.
		$filtered_posts = array_filter(
			$all_posts,
			function ( \WP_Post $post ) use ( $has_tag, $exclude_ignored ): bool {
				if ( $this->has_more_tag( $post ) !== $has_tag ) {
					return false;
				}
				if ( $exclude_ignored && $this->is_post_ignored( $post->ID ) ) {
					return false;
				}
				return true;
			}
		);

		$total = count( $filtered_posts );
		$pages = (int) ceil( $total / self::POSTS_PER_PAGE );

		// Slice for current page.
		$offset      = ( $page - 1 ) * self::POSTS_PER_PAGE;
		$paged_posts = array_slice( $filtered_posts, $offset, self::POSTS_PER_PAGE );

		return array(
			'posts' => array_values( $paged_posts ),
			'total' => $total,
			'pages' => max( 1, $pages ),
		);
	}

	/**
	 * Get all scannable posts.
	 *
	 * @return array<\WP_Post>
	 */
	private function get_all_scannable_posts(): array {
		static $cached_posts = null;

		if ( null !== $cached_posts ) {
			return $cached_posts;
		}

		$args = array(
			'post_type'      => 'post',
			'post_status'    => self::SCANNABLE_STATUSES,
			'posts_per_page' => -1,
			'orderby'        => 'date',
			'order'          => 'DESC',
		);

		$query = new \WP_Query( $args );

		$cached_posts = $query->posts;

		return $cached_posts;
	}

	/**
	 * Get total counts for all categories.
	 *
	 * @return array{missing: int, has_tag: int, ignored: int}
	 */
	public function get_counts(): array {
		$all_posts = $this->get_all_scannable_posts();

		$missing = 0;
		$has_tag = 0;
		$ignored = 0;

		foreach ( $all_posts as $post ) {
			if ( $this->has_more_tag( $post ) ) {
				++$has_tag;
			} elseif ( $this->is_post_ignored( $post->ID ) ) {
				++$ignored;
			} else {
				++$missing;
			}
		}

		return array(
			'missing' => $missing,
			'has_tag' => $has_tag,
			'ignored' => $ignored,
		);
	}

	/**
	 * Get posts per page setting.
	 *
	 * @return int
	 */
	public function get_posts_per_page(): int {
		return self::POSTS_PER_PAGE;
	}

	/**
	 * Check if a post is ignored.
	 *
	 * @param int $post_id Post ID.
	 * @return bool True if post is ignored, false otherwise.
	 */
	public function is_post_ignored( int $post_id ): bool {
		return '1' === get_post_meta( $post_id, self::IGNORED_META_KEY, true );
	}

	/**
	 * Get ignored posts (posts missing more tag that are marked as ignored).
	 *
	 * @param int $page Current page number.
	 * @return array{posts: array<\WP_Post>, total: int, pages: int}
	 */
	public function get_ignored_posts( int $page = 1 ): array {
		$all_posts = $this->get_all_scannable_posts();

		// Filter to posts missing more tag AND marked as ignored.
		$filtered_posts = array_filter(
			$all_posts,
			function ( \WP_Post $post ): bool {
				return ! $this->has_more_tag( $post ) && $this->is_post_ignored( $post->ID );
			}
		);

		$total = count( $filtered_posts );
		$pages = (int) ceil( $total / self::POSTS_PER_PAGE );

		// Slice for current page.
		$offset      = ( $page - 1 ) * self::POSTS_PER_PAGE;
		$paged_posts = array_slice( $filtered_posts, $offset, self::POSTS_PER_PAGE );

		return array(
			'posts' => array_values( $paged_posts ),
			'total' => $total,
			'pages' => max( 1, $pages ),
		);
	}

	/**
	 * Ignore a post (mark it as ignored).
	 *
	 * @param int $post_id Post ID.
	 * @return bool True on success, false on failure.
	 */
	public function ignore_post( int $post_id ): bool {
		return (bool) update_post_meta( $post_id, self::IGNORED_META_KEY, '1' );
	}

	/**
	 * Include a post (remove ignored flag).
	 *
	 * @param int $post_id Post ID.
	 * @return bool True on success, false on failure.
	 */
	public function include_post( int $post_id ): bool {
		return delete_post_meta( $post_id, self::IGNORED_META_KEY );
	}
}
