<?php
declare(strict_types=1);
/**
 * Main Plugin Class
 *
 * @package FindMissingMoreTags
 */

namespace FindMissingMoreTags;

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Plugin class implementing singleton pattern.
 */
class Plugin {

	/**
	 * Singleton instance.
	 *
	 * @var Plugin|null
	 */
	private static ?Plugin $instance = null;

	/**
	 * Admin page handler.
	 *
	 * @var AdminPage
	 */
	private AdminPage $admin_page;

	/**
	 * Post scanner.
	 *
	 * @var PostScanner
	 */
	private PostScanner $post_scanner;

	/**
	 * Get the singleton instance.
	 *
	 * @return Plugin
	 */
	public static function get_instance(): Plugin {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}

		return self::$instance;
	}

	/**
	 * Private constructor to enforce singleton.
	 */
	private function __construct() {
		$this->load_dependencies();
		$this->init_hooks();
	}

	/**
	 * Load required dependencies.
	 *
	 * @return void
	 */
	private function load_dependencies(): void {
		$this->post_scanner = new PostScanner();
		$this->admin_page   = new AdminPage( $this->post_scanner );
	}

	/**
	 * Initialize WordPress hooks.
	 *
	 * @return void
	 */
	private function init_hooks(): void {
		add_action( 'admin_menu', array( $this->admin_page, 'add_menu_page' ) );
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_assets' ) );
		add_action( 'admin_init', array( $this->admin_page, 'handle_ignore_action' ) );
	}

	/**
	 * Enqueue admin assets.
	 *
	 * @param string $hook_suffix The current admin page hook suffix.
	 * @return void
	 */
	public function enqueue_assets( string $hook_suffix ): void {
		// Only load on our plugin page.
		if ( 'posts_page_find-missing-more-tags' !== $hook_suffix ) {
			return;
		}

		wp_enqueue_style(
			'fmmt-admin',
			FMMT_PLUGIN_URL . 'assets/css/admin.css',
			array(),
			FMMT_VERSION
		);

		wp_enqueue_script(
			'fmmt-admin',
			FMMT_PLUGIN_URL . 'assets/js/admin.js',
			array(),
			FMMT_VERSION,
			true
		);
	}

	/**
	 * Get the post scanner instance.
	 *
	 * @return PostScanner
	 */
	public function get_post_scanner(): PostScanner {
		return $this->post_scanner;
	}

	/**
	 * Get the admin page instance.
	 *
	 * @return AdminPage
	 */
	public function get_admin_page(): AdminPage {
		return $this->admin_page;
	}
}
