<?php
declare(strict_types=1);
/**
 * Admin Page Class
 *
 * @package FindMissingMoreTags
 */

namespace FindMissingMoreTags;

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Handles admin menu registration and page rendering.
 */
class AdminPage {

	/**
	 * Post scanner instance.
	 *
	 * @var PostScanner
	 */
	private PostScanner $scanner;

	/**
	 * Constructor.
	 *
	 * @param PostScanner $scanner Post scanner instance.
	 */
	public function __construct( PostScanner $scanner ) {
		$this->scanner = $scanner;
	}

	/**
	 * Add the admin menu page.
	 *
	 * @return void
	 */
	public function add_menu_page(): void {
		add_posts_page(
			__( 'Missing More Tags', 'find-missing-more-tags' ),
			__( 'Missing More Tags', 'find-missing-more-tags' ),
			'edit_posts',
			'find-missing-more-tags',
			array( $this, 'render_page' )
		);
	}

	/**
	 * Handle ignore/include actions.
	 *
	 * @return void
	 */
	public function handle_ignore_action(): void {
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Only checking if action exists here.
		if ( ! isset( $_GET['action'] ) || ! isset( $_GET['page'] ) || 'find-missing-more-tags' !== $_GET['page'] ) {
			return;
		}

		// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Checking action type before nonce verification.
		$action = sanitize_key( $_GET['action'] );

		if ( ! in_array( $action, array( 'ignore', 'include' ), true ) ) {
			return;
		}

		// Verify nonce.
		if ( ! isset( $_GET['_wpnonce'] ) || ! wp_verify_nonce( sanitize_key( $_GET['_wpnonce'] ), 'fmmt_' . $action . '_post' ) ) {
			wp_die(
				esc_html__( 'Security check failed. Please try again.', 'find-missing-more-tags' ),
				esc_html__( 'Error', 'find-missing-more-tags' ),
				array( 'response' => 403 )
			);
		}

		// Get and validate post ID.
		$post_id = isset( $_GET['post_id'] ) ? absint( $_GET['post_id'] ) : 0;

		if ( 0 === $post_id ) {
			wp_die(
				esc_html__( 'Invalid post ID.', 'find-missing-more-tags' ),
				esc_html__( 'Error', 'find-missing-more-tags' ),
				array( 'response' => 400 )
			);
		}

		// Check capability.
		if ( ! current_user_can( 'edit_post', $post_id ) ) {
			wp_die(
				esc_html__( 'You do not have permission to modify this post.', 'find-missing-more-tags' ),
				esc_html__( 'Error', 'find-missing-more-tags' ),
				array( 'response' => 403 )
			);
		}

		// Perform action.
		if ( 'ignore' === $action ) {
			$this->scanner->ignore_post( $post_id );
			$redirect_tab = 'missing';
		} else {
			$this->scanner->include_post( $post_id );
			$redirect_tab = 'ignored';
		}

		// Redirect back.
		$redirect_url = add_query_arg(
			array(
				'page' => 'find-missing-more-tags',
				'tab'  => $redirect_tab,
			),
			admin_url( 'edit.php' )
		);

		wp_safe_redirect( $redirect_url );
		exit;
	}

	/**
	 * Render the admin page.
	 *
	 * @return void
	 */
	public function render_page(): void {
		// Verify user capability.
		if ( ! current_user_can( 'edit_posts' ) ) {
			wp_die(
				esc_html__( 'You do not have sufficient permissions to access this page.', 'find-missing-more-tags' )
			);
		}

		// Get current tab and page.
		$current_tab  = $this->get_current_tab();
		$current_page = $this->get_current_page();

		// Get counts for tab labels.
		$counts = $this->scanner->get_counts();

		// Get posts for current tab.
		if ( 'has-tag' === $current_tab ) {
			$result = $this->scanner->get_posts_with_more_tag( $current_page );
		} elseif ( 'ignored' === $current_tab ) {
			$result = $this->scanner->get_ignored_posts( $current_page );
		} else {
			$result = $this->scanner->get_posts_missing_more_tag( $current_page );
		}

		// Load the template.
		$template_vars = array(
			'current_tab'  => $current_tab,
			'current_page' => $current_page,
			'counts'       => $counts,
			'posts'        => $result['posts'],
			'total_posts'  => $result['total'],
			'total_pages'  => $result['pages'],
			'admin_page'   => $this,
		);

		// Extract variables for template.
		// phpcs:ignore WordPress.PHP.DontExtract.extract_extract
		extract( $template_vars );

		include FMMT_PLUGIN_DIR . 'templates/admin-page.php';
	}

	/**
	 * Get the current tab from query string.
	 *
	 * @return string
	 */
	private function get_current_tab(): string {
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only display parameter.
		$tab = isset( $_GET['tab'] ) ? sanitize_key( $_GET['tab'] ) : 'missing';

		return in_array( $tab, array( 'missing', 'has-tag', 'ignored' ), true ) ? $tab : 'missing';
	}

	/**
	 * Get the current page number from query string.
	 *
	 * @return int
	 */
	private function get_current_page(): int {
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only display parameter.
		$page = isset( $_GET['paged'] ) ? absint( $_GET['paged'] ) : 1;

		return max( 1, $page );
	}

	/**
	 * Get the URL for a specific tab.
	 *
	 * @param string $tab Tab identifier.
	 * @return string
	 */
	public function get_tab_url( string $tab ): string {
		return add_query_arg(
			array(
				'page' => 'find-missing-more-tags',
				'tab'  => $tab,
			),
			admin_url( 'edit.php' )
		);
	}

	/**
	 * Get the pagination URL.
	 *
	 * @param int    $page Page number.
	 * @param string $tab  Current tab.
	 * @return string
	 */
	public function get_pagination_url( int $page, string $tab ): string {
		return add_query_arg(
			array(
				'page'  => 'find-missing-more-tags',
				'tab'   => $tab,
				'paged' => $page,
			),
			admin_url( 'edit.php' )
		);
	}

	/**
	 * Get human-readable post status label.
	 *
	 * @param string $status Post status.
	 * @return string
	 */
	public function get_status_label( string $status ): string {
		$labels = array(
			'publish' => __( 'Published', 'find-missing-more-tags' ),
			'draft'   => __( 'Draft', 'find-missing-more-tags' ),
			'pending' => __( 'Pending', 'find-missing-more-tags' ),
			'future'  => __( 'Scheduled', 'find-missing-more-tags' ),
			'private' => __( 'Private', 'find-missing-more-tags' ),
		);

		return $labels[ $status ] ?? ucfirst( $status );
	}

	/**
	 * Get the CSS class for a post status.
	 *
	 * @param string $status Post status.
	 * @return string
	 */
	public function get_status_class( string $status ): string {
		$classes = array(
			'publish' => 'fmmt-status--published',
			'draft'   => 'fmmt-status--draft',
			'pending' => 'fmmt-status--pending',
			'future'  => 'fmmt-status--scheduled',
			'private' => 'fmmt-status--private',
		);

		return $classes[ $status ] ?? 'fmmt-status--default';
	}

	/**
	 * Render pagination.
	 *
	 * @param int    $current_page Current page number.
	 * @param int    $total_pages  Total number of pages.
	 * @param string $tab          Current tab.
	 * @return void
	 */
	public function render_pagination( int $current_page, int $total_pages, string $tab ): void {
		if ( $total_pages <= 1 ) {
			return;
		}

		echo '<div class="fmmt-pagination">';

		// Previous link.
		if ( $current_page > 1 ) {
			printf(
				'<a href="%s" class="fmmt-pagination__link fmmt-pagination__prev">&larr; %s</a>',
				esc_url( $this->get_pagination_url( $current_page - 1, $tab ) ),
				esc_html__( 'Previous', 'find-missing-more-tags' )
			);
		} else {
			printf(
				'<span class="fmmt-pagination__link fmmt-pagination__prev fmmt-pagination__link--disabled">&larr; %s</span>',
				esc_html__( 'Previous', 'find-missing-more-tags' )
			);
		}

		// Page numbers.
		echo '<span class="fmmt-pagination__numbers">';
		$this->render_page_numbers( $current_page, $total_pages, $tab );
		echo '</span>';

		// Next link.
		if ( $current_page < $total_pages ) {
			printf(
				'<a href="%s" class="fmmt-pagination__link fmmt-pagination__next">%s &rarr;</a>',
				esc_url( $this->get_pagination_url( $current_page + 1, $tab ) ),
				esc_html__( 'Next', 'find-missing-more-tags' )
			);
		} else {
			printf(
				'<span class="fmmt-pagination__link fmmt-pagination__next fmmt-pagination__link--disabled">%s &rarr;</span>',
				esc_html__( 'Next', 'find-missing-more-tags' )
			);
		}

		echo '</div>';
	}

	/**
	 * Render page number links.
	 *
	 * @param int    $current_page Current page.
	 * @param int    $total_pages  Total pages.
	 * @param string $tab          Current tab.
	 * @return void
	 */
	private function render_page_numbers( int $current_page, int $total_pages, string $tab ): void {
		$range = 2; // Number of pages to show on each side of current.

		for ( $i = 1; $i <= $total_pages; $i++ ) {
			// Always show first, last, and pages near current.
			$show_page = ( 1 === $i )
				|| ( $total_pages === $i )
				|| ( $i >= $current_page - $range && $i <= $current_page + $range );

			if ( ! $show_page ) {
				// Show ellipsis.
				if ( 2 === $i || $total_pages - 1 === $i ) {
					echo '<span class="fmmt-pagination__ellipsis">&hellip;</span>';
				}
				continue;
			}

			if ( $current_page === $i ) {
				printf(
					'<span class="fmmt-pagination__link fmmt-pagination__link--current">%d</span>',
					(int) $i
				);
			} else {
				printf(
					'<a href="%s" class="fmmt-pagination__link">%d</a>',
					esc_url( $this->get_pagination_url( $i, $tab ) ),
					(int) $i
				);
			}
		}
	}

	/**
	 * Get the URL to ignore a post.
	 *
	 * @param int $post_id Post ID.
	 * @return string
	 */
	public function get_ignore_url( int $post_id ): string {
		return wp_nonce_url(
			add_query_arg(
				array(
					'page'    => 'find-missing-more-tags',
					'action'  => 'ignore',
					'post_id' => $post_id,
				),
				admin_url( 'edit.php' )
			),
			'fmmt_ignore_post'
		);
	}

	/**
	 * Get the URL to include (un-ignore) a post.
	 *
	 * @param int $post_id Post ID.
	 * @return string
	 */
	public function get_include_url( int $post_id ): string {
		return wp_nonce_url(
			add_query_arg(
				array(
					'page'    => 'find-missing-more-tags',
					'action'  => 'include',
					'post_id' => $post_id,
				),
				admin_url( 'edit.php' )
			),
			'fmmt_include_post'
		);
	}
}
