<?php
/**
 * Public functions for Featured Products Carousel by Tag
 *
 * Registers/enqueues frontend assets and provides the shortcode [featured-products-carousel-tag].
 *
 * Location: public/featured-products-carousel-tag-functions.php
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Enqueue frontend scripts and styles (local files only, no CDN).
 * Simple and direct enqueuing as requested.
 */
add_action( 'wp_enqueue_scripts', 'fpct_enqueue_scripts', 999 );
function fpct_enqueue_scripts() {
    // JS (local files expected to exist in plugin public/js/)
    wp_register_script( 'fpct-owl-js', fpct_plugin_url . 'public/js/owl.carousel.min.js', array( 'jquery' ), '2.3.4', true );
    wp_enqueue_script( 'fpct-owl-js' );

    wp_register_script( 'fpct-scripts', fpct_plugin_url . 'public/js/scripts.js', array( 'jquery', 'fpct-owl-js' ), '1.0.0', true );
    wp_enqueue_script( 'fpct-scripts' );

    // CSS (local files expected to exist in plugin public/css/)
    wp_enqueue_style( 'fpct-owl-css', fpct_plugin_url . 'public/css/owl.carousel.min.css', array(), '2.3.4' );
    wp_enqueue_style( 'fpct-owl-theme', fpct_plugin_url . 'public/css/owl.theme.default.min.css', array( 'fpct-owl-css' ), '2.3.4' );
    wp_enqueue_style( 'fpct-styles', fpct_plugin_url . 'public/css/styles.css', array(), '1.0.0' );
}

/**
 * Shortcode: [featured-products-carousel-tag]
 *
 * Attributes:
 *  - per_page (int) number of products to query, default 12
 *  - items (int) number of items visible on desktop, default 4
 *  - autoplay (string) 'true'|'false', default 'true'
 *
 * Output: container with class fpct-carousel and owl-carousel, and items inside.
 */
add_shortcode( 'featured-products-carousel-tag', 'fpct_shortcode_carousel' );
function fpct_shortcode_carousel( $atts ) {
    // If the plugin option disables output, return empty
    if ( ! get_option( 'fpct_enabled', 1 ) ) {
        return '';
    }

    // Require WooCommerce
    if ( ! class_exists( 'WooCommerce' ) ) {
        return '<p>' . esc_html__( 'WooCommerce is not active.', 'featured-products-carousel-tag' ) . '</p>';
    }

    $atts = shortcode_atts(
        array(
            'per_page' => 12,
            'items'    => 4,
            'autoplay' => 'true',
            'cache_ttl' => 300, // segundos para cachear IDs de productos (transient)
        ),
        $atts,
        'featured-products-carousel-tag'
    );

    // Ensure attributes are valid
    $per_page  = max( 1, intval( $atts['per_page'] ) );
    $items     = max( 1, intval( $atts['items'] ) );
    $autoplay  = ( $atts['autoplay'] === 'true' ) ? 'true' : 'false';
    $cache_ttl = max( 0, intval( $atts['cache_ttl'] ) );

    // Build cache key based on slugs relevant here
    $cache_key = 'fpct_featured_prod_ids_v1';

    // Try get cached IDs first
    $product_ids = false;
    if ( $cache_ttl > 0 ) {
        $product_ids = get_transient( $cache_key );
        if ( ! empty( $product_ids ) && ! is_array( $product_ids ) ) {
            $product_ids = false;
        }
    }

    if ( $product_ids === false ) {
        // Gather IDs from product_tag terms 'featured' and 'destacado' (if exist)
        $slugs = array( 'featured', 'destacado' );
        $collected = array();

        foreach ( $slugs as $slug ) {
            $term = get_term_by( 'slug', $slug, 'product_tag' );
            if ( $term && ! is_wp_error( $term ) ) {
                // get_objects_in_term returns array of object IDs or WP_Error
                $objs = get_objects_in_term( intval( $term->term_id ), 'product_tag' );
                if ( is_array( $objs ) && ! empty( $objs ) ) {
                    $collected = array_merge( $collected, $objs );
                }
            }
        }

        // Normalize IDs
        $product_ids = array_map( 'intval', $collected );
        $product_ids = array_values( array_unique( $product_ids ) );

        // Cache results (even empty array for short time)
        if ( $cache_ttl > 0 ) {
            set_transient( $cache_key, $product_ids, $cache_ttl );
        }
    }

    // If no products found via tags, early return
    if ( empty( $product_ids ) ) {
        return '<p>' . esc_html__( 'No featured products found.', 'featured-products-carousel-tag' ) . '</p>';
    }

    // Build query: use post__in so we avoid large tax_query overhead
    $args = array(
        'post_type'      => 'product',
        'posts_per_page' => $per_page,
        'post_status'    => 'publish',
        'post__in'       => $product_ids,
        'orderby'        => 'date',
        'order'          => 'DESC',
        'no_found_rows'  => true,
    );

    $query = new WP_Query( $args );

    if ( ! $query->have_posts() ) {
        wp_reset_postdata();
        return '<p>' . esc_html__( 'No featured products found.', 'featured-products-carousel-tag' ) . '</p>';
    }

    // Unique id for instance
    $uniq = 'fpct-carousel-' . wp_unique_id();

    // Build HTML
    ob_start();
    ?>
    <div id="<?php echo esc_attr( $uniq ); ?>"
         class="fpct-carousel owl-carousel"
         data-fpct-autoplay="<?php echo esc_attr( $autoplay ); ?>"
         data-fpct-items="<?php echo esc_attr( $items ); ?>">
        <?php
        while ( $query->have_posts() ) :
            $query->the_post();

            $product_id = get_the_ID();
            $product    = wc_get_product( $product_id );

            if ( ! $product ) {
                continue;
            }

            $permalink  = get_permalink( $product_id );
            $title      = get_the_title( $product_id );
            $price_html = $product->get_price_html();

            // Thumbnail (with alt attribute)
            $thumbnail = get_the_post_thumbnail( $product_id, 'woocommerce_thumbnail', array( 'loading' => 'lazy', 'alt' => esc_attr( $title ) ) );
            ?>
            <div class="fpct-item" role="group" aria-label="<?php echo esc_attr( $title ); ?>">
                <?php
                /* translators: %s: product title used for aria-label on the product link */
                $aria_text = sprintf( __( 'View %s', 'featured-products-carousel-tag' ), $title );
                ?>
                <a href="<?php echo esc_url( $permalink ); ?>" class="fpct-thumb" aria-label="<?php echo esc_attr( $aria_text ); ?>">
                    <?php
                    if ( $thumbnail ) {
                        echo wp_kses_post( $thumbnail );
                    } else {
                        echo wp_kses_post( wc_placeholder_img( 'woocommerce_thumbnail' ) );
                    }
                    ?>
                </a>

                <a href="<?php echo esc_url( $permalink ); ?>" class="fpct-title">
                    <?php echo esc_html( $title ); ?>
                </a>

                <a href="<?php echo esc_url( $permalink ); ?>" class="fpct-price">
                    <?php echo wp_kses_post( $price_html ); ?>
                </a>

                <div class="fpct-add-to-cart">
                    <?php
                    // Provide global $product for the WC template function
                    $GLOBALS['product'] = $product;
                    woocommerce_template_loop_add_to_cart();
                    unset( $GLOBALS['product'] );
                    ?>
                </div>
            </div>
        <?php endwhile; ?>
    </div>
    <?php
    wp_reset_postdata();

    return ob_get_clean();
}
