<?php
/**
 * Admin page for Featured Products Carousel by Tag
 *
 * Fixes:
 *  - Remove deprecated FILTER_SANITIZE_STRING usage.
 *  - Use filter_input(..., FILTER_DEFAULT) then sanitize with WP functions.
 *  - check_admin_referer() executed before consuming $_POST values.
 *  - Safe reading of inputs using wp_unslash() when needed.
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Render admin page
 */
function fpct_admin_page() {
    if ( ! current_user_can( 'manage_options' ) ) {
        return;
    }

    // 1) Build filtered list of post types (for display & validation)
    $all_post_types = get_post_types( array(), 'objects' );

    $blacklist = array(
        'attachment','revision','nav_menu_item','custom_css','customize_changeset',
        'oembed_cache','user_request','wp_block','wp_template','wp_template_part',
        'wp_global_styles','wp_navigation','wp_font_family','wp_font_face',
        'product_variation','shop_order','shop_order_refund','shop_coupon'
    );

    $filtered_post_types = array();
    foreach ( $all_post_types as $pt_obj ) {
        $pt_name = (string) $pt_obj->name;

        if ( in_array( $pt_name, $blacklist, true ) ) {
            continue;
        }

        if ( 0 === strpos( $pt_name, 'wp_' ) ) {
            continue;
        }

        $has_ui   = isset( $pt_obj->show_ui ) ? (bool) $pt_obj->show_ui : false;
        $is_public = isset( $pt_obj->public ) ? (bool) $pt_obj->public : false;

        if ( ! $has_ui && ! $is_public ) {
            continue;
        }

        $filtered_post_types[ $pt_name ] = $pt_obj;
    }

    // Sort by human label
    uasort( $filtered_post_types, function( $a, $b ) {
        $la = ! empty( $a->labels->singular_name ) ? $a->labels->singular_name : $a->name;
        $lb = ! empty( $b->labels->singular_name ) ? $b->labels->singular_name : $b->name;
        return strcasecmp( $la, $lb );
    } );

    $allowed_names = array_keys( $filtered_post_types );

    // --------------------------
    // Process form submission
    // --------------------------
    // Detect submission: use filter_input with FILTER_DEFAULT (no deprecated filters)
    $form_submitted = filter_input( INPUT_POST, 'fpct_admin_submit', FILTER_DEFAULT );

    if ( $form_submitted !== null ) {
        // Verify nonce before reading any other $_POST values
        check_admin_referer( 'fpct_save_settings', 'fpct_nonce' );

        // Activar/desactivar (checkbox) - use filter_input boolean
        $enabled_bool = filter_input( INPUT_POST, 'fpct_enabled', FILTER_VALIDATE_BOOLEAN, FILTER_NULL_ON_FAILURE );
        $enabled = $enabled_bool ? 1 : 0;
        update_option( 'fpct_enabled', $enabled );

        // If later you add array inputs or text inputs that may contain slashes,
        // use wp_unslash() + sanitize_text_field() or esc_url_raw() as needed here.

        // Success notice
        echo '<div class="notice notice-success is-dismissible"><p>' . esc_html__( 'Ajustes guardados.', 'featured-products-carousel-tag' ) . '</p></div>';
    }

    // --------------------------
    // Load options for form (sanitized)
    // --------------------------
    $enabled_flag = intval( get_option( 'fpct_enabled', 1 ) );

    ?>
    <div class="wrap">
        <h1><?php echo esc_html__( 'Carrusel de productos destacados por etiqueta', 'featured-products-carousel-tag' ); ?></h1>

        <form method="post" action="">
            <?php wp_nonce_field( 'fpct_save_settings', 'fpct_nonce' ); ?>
            <input type="hidden" name="fpct_admin_submit" value="1" />

            <table class="form-table">
                <tr>
                    <th scope="row"><?php echo esc_html__( 'Activar carrusel', 'featured-products-carousel-tag' ); ?></th>
                    <td>
                        <label for="fpct_enabled">
                            <input type="checkbox" id="fpct_enabled" name="fpct_enabled" value="1" <?php checked( $enabled_flag, 1 ); ?> />
                            <?php echo esc_html__( 'Activar el shortcode [featured-products-carousel-tag]', 'featured-products-carousel-tag' ); ?>
                        </label>
                    </td>
                </tr>
            </table>

            <?php submit_button(); ?>
        </form>

        <hr>

        <h2 id="wpv-about"><?php echo esc_html__( 'Acerca de este plugin', 'featured-products-carousel-tag' ); ?></h2>
        <p><?php echo esc_html__( 'Este plugin muestra productos de WooCommerce con la etiqueta "featured" o "destacado" en un carrusel responsive usando OwlCarousel2. Usa el shortcode [featured-products-carousel-tag].', 'featured-products-carousel-tag' ); ?></p>
        <hr>

        <p><strong><?php echo esc_html__( 'Desarrollador', 'featured-products-carousel-tag' ); ?></strong></p>
        <ul>
            <li><a href="https://d3veloper.es/" target="_blank" rel="noopener noreferrer"><?php echo esc_html__( 'Sitio Web', 'featured-products-carousel-tag' ); ?></a></li>
            <li><a href="https://www.instagram.com/d3veloper.es/" target="_blank" rel="noopener noreferrer"><?php echo esc_html__( 'Instagram', 'featured-products-carousel-tag' ); ?></a></li>
            <li><a href="mailto:hola@d3veloper.es" target="_blank" rel="noopener noreferrer">hola@d3veloper.es</a></li>
        </ul>
        <hr>

        <h3><?php echo esc_html__( 'Cómo usar', 'featured-products-carousel-tag' ); ?></h3>
        <ol>
            <li><?php echo esc_html__( 'Marca tus productos con la etiqueta "featured" o "destacado".', 'featured-products-carousel-tag' ); ?></li>
            <li><?php echo esc_html__( 'Inserta el shortcode [featured-products-carousel-tag] en cualquier página o entrada.', 'featured-products-carousel-tag' ); ?></li>
        </ol>
    </div>
    <?php
}
