<?php
/**
 * Funciones públicas: registro de assets y shortcode
 *
 * Optimizado para evitar tax_query pesado: detecta taxonomías tipo "tag",
 * obtiene objetos en término y luego hace una única WP_Query con post__in.
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Registrar assets locales y setear transient si faltan
 */
if ( ! function_exists( 'fpct_pc_register_assets' ) ) {
    function fpct_pc_register_assets() {
        $missing = array();
        $css_dir = FPCT_PC_PLUGIN_DIR . 'public/css/';
        $js_dir  = FPCT_PC_PLUGIN_DIR . 'public/js/';

        if ( file_exists( $css_dir . 'owl.carousel.min.css' ) ) {
            wp_register_style( 'fpct_pc_owl_css', FPCT_PC_PLUGIN_URL . 'public/css/owl.carousel.min.css', array(), '2.3.4' );
        } else {
            $missing[] = 'public/css/owl.carousel.min.css';
        }

        if ( file_exists( $css_dir . 'owl.theme.default.min.css' ) ) {
            wp_register_style( 'fpct_pc_owl_theme', FPCT_PC_PLUGIN_URL . 'public/css/owl.theme.default.min.css', array( 'fpct_pc_owl_css' ), '2.3.4' );
        } else {
            $missing[] = 'public/css/owl.theme.default.min.css';
        }

        if ( file_exists( $css_dir . 'styles.css' ) ) {
            wp_register_style( 'fpct_pc_styles', FPCT_PC_PLUGIN_URL . 'public/css/styles.css', array(), '1.0.0' );
        } else {
            $missing[] = 'public/css/styles.css';
        }

        if ( file_exists( $js_dir . 'owl.carousel.min.js' ) ) {
            wp_register_script( 'fpct_pc_owl_js', FPCT_PC_PLUGIN_URL . 'public/js/owl.carousel.min.js', array( 'jquery' ), '2.3.4', true );
        } else {
            $missing[] = 'public/js/owl.carousel.min.js';
        }

        if ( file_exists( $js_dir . 'scripts.js' ) ) {
            wp_register_script( 'fpct_pc_scripts', FPCT_PC_PLUGIN_URL . 'public/js/scripts.js', array( 'jquery', 'fpct_pc_owl_js' ), '1.0.0', true );
        } else {
            $missing[] = 'public/js/scripts.js';
        }

        if ( ! empty( $missing ) ) {
            set_transient( 'fpct_pc_missing_assets', $missing, 5 * MINUTE_IN_SECONDS );
        } else {
            delete_transient( 'fpct_pc_missing_assets' );
        }
    }
}
add_action( 'wp_enqueue_scripts', 'fpct_pc_register_assets', 20 );

/**
 * Admin notice si faltan assets locales
 */
if ( ! function_exists( 'fpct_pc_admin_missing_assets_notice' ) ) {
    function fpct_pc_admin_missing_assets_notice() {
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }
        $missing = get_transient( 'fpct_pc_missing_assets' );
        if ( empty( $missing ) || ! is_array( $missing ) ) {
            return;
        }
        echo '<div class="notice notice-warning is-dismissible">';
        echo '<p><strong>' . esc_html__( 'Featured Post Carousel: faltan archivos locales', 'featured-post-carousel-tag' ) . '</strong></p>';
        echo '<p>' . esc_html__( 'Sube los archivos OwlCarousel en public/js/ y public/css/ dentro del plugin para activar el carrusel.', 'featured-post-carousel-tag' ) . '</p>';
        echo '<ul>';
        foreach ( $missing as $m ) {
            echo '<li>' . esc_html( $m ) . '</li>';
        }
        echo '</ul>';
        echo '</div>';
        delete_transient( 'fpct_pc_missing_assets' );
    }
}
add_action( 'admin_notices', 'fpct_pc_admin_missing_assets_notice' );

/**
 * Shortcode [featured-post-carousel-tag]
 * - Busca taxonomías "tipo tag" asociadas a los post types seleccionados,
 *   verifica si existen términos 'featured' o 'destacado', usa get_objects_in_term()
 *   para obtener IDs y luego consulta una sola vez con post__in.
 */
if ( ! function_exists( 'fpct_pc_shortcode_render' ) ) {
    function fpct_pc_shortcode_render( $atts ) {
        if ( ! get_option( 'fpct_pc_enabled', 1 ) ) {
            return '';
        }

        $atts = shortcode_atts( array(
            'per_page'  => 8,
            'items'     => 3,
            'autoplay'  => 'true',
            'cache_ttl' => 300, // 5 minutos
        ), $atts, 'featured-post-carousel-tag' );

        $per_page  = max( 1, intval( $atts['per_page'] ) );
        $items     = max( 1, intval( $atts['items'] ) );
        $autoplay  = ( $atts['autoplay'] === 'true' ) ? 'true' : 'false';
        $cache_ttl = max( 0, intval( $atts['cache_ttl'] ) );

        // Post types permitidos (sanear)
        $post_types = (array) get_option( 'fpct_pc_post_types', array( 'post', 'page', 'product' ) );
        $post_types = array_filter( array_map( 'sanitize_text_field', $post_types ) );
        if ( empty( $post_types ) ) {
            $post_types = array( 'post' );
        }

        // Cache key
        $cache_key = 'fpct_pc_ids_' . md5( implode( ',', $post_types ) . '|' . $per_page . '|' . $items . '|featured|destacado' );

        $post_ids = false;
        if ( $cache_ttl > 0 ) {
            $post_ids = get_transient( $cache_key );
            if ( ! empty( $post_ids ) && ! is_array( $post_ids ) ) {
                $post_ids = false;
            }
        }

        if ( $post_ids === false ) {
            $tag_taxonomies = array();
            foreach ( $post_types as $pt ) {
                $tax_objs = get_object_taxonomies( $pt, 'objects' );
                if ( empty( $tax_objs ) ) {
                    continue;
                }
                foreach ( $tax_objs as $tax_obj ) {
                    $has_ui    = isset( $tax_obj->show_ui ) ? (bool) $tax_obj->show_ui : false;
                    $is_public = isset( $tax_obj->public ) ? (bool) $tax_obj->public : false;
                    $is_hier   = isset( $tax_obj->hierarchical ) ? (bool) $tax_obj->hierarchical : false;

                    if ( ! $has_ui && ! $is_public ) {
                        continue;
                    }
                    if ( $is_hier ) {
                        continue;
                    }
                    $tname = (string) $tax_obj->name;
                    $tag_taxonomies[ $tname ] = true;
                }
            }
            $tag_taxonomies = array_keys( $tag_taxonomies );

            if ( empty( $tag_taxonomies ) && taxonomy_exists( 'post_tag' ) ) {
                $tag_taxonomies[] = 'post_tag';
            }

            if ( empty( $tag_taxonomies ) ) {
                return '<p>' . esc_html__( 'No hay taxonomías de etiquetas disponibles para los tipos seleccionados.', 'featured-post-carousel-tag' ) . '</p>';
            }

            $collected_ids = array();
            foreach ( $tag_taxonomies as $tax ) {
                $term_featured = get_term_by( 'slug', 'featured', $tax );
                if ( $term_featured && ! is_wp_error( $term_featured ) ) {
                    $term_id = intval( $term_featured->term_id );
                    $objs = get_objects_in_term( $term_id, $tax );
                    if ( is_array( $objs ) && ! empty( $objs ) ) {
                        $collected_ids = array_merge( $collected_ids, $objs );
                    }
                }

                $term_dest = get_term_by( 'slug', 'destacado', $tax );
                if ( $term_dest && ! is_wp_error( $term_dest ) ) {
                    $term_id = intval( $term_dest->term_id );
                    $objs = get_objects_in_term( $term_id, $tax );
                    if ( is_array( $objs ) && ! empty( $objs ) ) {
                        $collected_ids = array_merge( $collected_ids, $objs );
                    }
                }
            }

            $post_ids = array_map( 'intval', $collected_ids );
            $post_ids = array_values( array_unique( $post_ids ) );

            if ( empty( $post_ids ) ) {
                if ( $cache_ttl > 0 ) {
                    set_transient( $cache_key, array(), min( 60, $cache_ttl ) );
                }
                return '<p>' . esc_html__( 'No se encontraron elementos destacados.', 'featured-post-carousel-tag' ) . '</p>';
            }

            if ( $cache_ttl > 0 ) {
                set_transient( $cache_key, $post_ids, $cache_ttl );
            }
        }

        // Una sola WP_Query con post__in
        $args = array(
            'post_type'      => $post_types,
            'posts_per_page' => $per_page,
            'post_status'    => 'publish',
            'post__in'       => $post_ids,
            'orderby'        => 'date',
            'no_found_rows'  => true,
        );

        $q = new WP_Query( $args );

        if ( ! $q->have_posts() ) {
            wp_reset_postdata();
            return '<p>' . esc_html__( 'No se encontraron elementos destacados.', 'featured-post-carousel-tag' ) . '</p>';
        }

        // Encolar estilos/scripts si están registrados
        if ( wp_style_is( 'fpct_pc_owl_css', 'registered' ) ) { wp_enqueue_style( 'fpct_pc_owl_css' ); }
        if ( wp_style_is( 'fpct_pc_owl_theme', 'registered' ) ) { wp_enqueue_style( 'fpct_pc_owl_theme' ); }
        if ( wp_style_is( 'fpct_pc_styles', 'registered' ) ) { wp_enqueue_style( 'fpct_pc_styles' ); }

        if ( wp_script_is( 'fpct_pc_owl_js', 'registered' ) ) { wp_enqueue_script( 'fpct_pc_owl_js' ); }
        if ( wp_script_is( 'fpct_pc_scripts', 'registered' ) ) { wp_enqueue_script( 'fpct_pc_scripts' ); }

        $uniq = 'fpct_pc_' . wp_unique_id();
        ob_start();
        ?>
        <div id="<?php echo esc_attr( $uniq ); ?>" class="fpct-pc-carousel owl-carousel" data-fpct-items="<?php echo esc_attr( $items ); ?>" data-fpct-autoplay="<?php echo esc_attr( $autoplay ); ?>">
            <?php while ( $q->have_posts() ) : $q->the_post();
                $item_id   = get_the_ID();
                $pt        = get_post_type( $item_id );
                $title     = get_the_title( $item_id );
                $permalink = get_permalink( $item_id );
                $thumb     = get_the_post_thumbnail( $item_id, 'medium', array( 'loading' => 'lazy', 'alt' => esc_attr( $title ) ) );
                if ( ! $thumb ) {
                    $def = get_option( 'fpct_pc_default_image', '' );
                    if ( ! empty( $def ) ) {
                        $thumb = '<img src="' . esc_url( $def ) . '" alt="' . esc_attr( $title ) . '" loading="lazy" />';
                    } else {
                        $thumb = '<div class="fpct-pc-no-thumb" aria-hidden="true" style="width:100%;height:140px;background:#f4f4f4;"></div>';
                    }
                }
                $excerpt = get_the_excerpt( $item_id );
                ?>
                <article class="fpct-pc-item" role="group" aria-label="<?php echo esc_attr( $title ); ?>">
                    <a class="fpct-pc-thumb" href="<?php echo esc_url( $permalink ); ?>"><?php echo wp_kses_post( $thumb ); ?></a>
                    <h3 class="fpct-pc-title"><a href="<?php echo esc_url( $permalink ); ?>"><?php echo esc_html( $title ); ?></a></h3>
                    <?php if ( ! empty( $excerpt ) ) : ?>
                        <div class="fpct-pc-excerpt"><?php echo wp_kses_post( wp_trim_words( $excerpt, 18 ) ); ?></div>
                    <?php endif; ?>

                    <div class="fpct-pc-actions">
                        <?php
                        if ( $pt === 'product' && class_exists( 'WooCommerce' ) ) {
                            $product = wc_get_product( $item_id );
                            if ( $product ) {
                                $GLOBALS['product'] = $product;
                                echo '<div class="fpct-pc-add-to-cart">';
                                woocommerce_template_loop_add_to_cart();
                                echo '</div>';
                                unset( $GLOBALS['product'] );
                            } else {
                                echo '<a class="button fpct-pc-readmore" href="' . esc_url( $permalink ) . '">' . esc_html__( 'Leer más', 'featured-post-carousel-tag' ) . '</a>';
                            }
                        } else {
                            echo '<a class="button fpct-pc-readmore" href="' . esc_url( $permalink ) . '">' . esc_html__( 'Leer más', 'featured-post-carousel-tag' ) . '</a>';
                        }
                        ?>
                    </div>
                </article>
            <?php endwhile; ?>
        </div>
        <?php
        wp_reset_postdata();
        return ob_get_clean();
    }
}
add_shortcode( 'featured-post-carousel-tag', 'fpct_pc_shortcode_render' );
