<?php
/**
 * Página de administración: ajustes del plugin
 *
 * Reglas aplicadas:
 * - check_admin_referer() antes de procesar datos del formulario.
 * - Uso de filter_input() / filter_input_array() para evitar accesos directos a $_POST.
 * - wp_unslash() cuando procede, y sanitización con sanitize_key(), sanitize_text_field(), esc_url_raw(), intval().
 * - Escapado correcto al imprimir (esc_html, esc_attr).
 *
 * Ubicación: admin/featured-post-carousel-tag-admin.php
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

if ( ! function_exists( 'fpct_pc_admin_page_callback' ) ) {
    function fpct_pc_admin_page_callback() {
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_die( esc_html__( 'No tienes permisos suficientes', 'featured-post-carousel-tag' ) );
        }

        // 1) Construir lista filtrada de post types
        $all_post_types = get_post_types( array(), 'objects' );

        $blacklist = array(
            'attachment','revision','nav_menu_item','custom_css','customize_changeset',
            'oembed_cache','user_request','wp_block','wp_template','wp_template_part',
            'wp_global_styles','wp_navigation','wp_font_family','wp_font_face',
            'product_variation','shop_order','shop_order_refund','shop_coupon'
        );

        $filtered_post_types = array();
        foreach ( $all_post_types as $pt_obj ) {
            $pt_name = (string) $pt_obj->name;

            if ( in_array( $pt_name, $blacklist, true ) ) {
                continue;
            }

            if ( 0 === strpos( $pt_name, 'wp_' ) ) {
                continue;
            }

            $has_ui   = isset( $pt_obj->show_ui ) ? (bool) $pt_obj->show_ui : false;
            $is_public = isset( $pt_obj->public ) ? (bool) $pt_obj->public : false;

            if ( ! $has_ui && ! $is_public ) {
                continue;
            }

            $filtered_post_types[ $pt_name ] = $pt_obj;
        }

        // Orden por etiqueta humana
        uasort( $filtered_post_types, function( $a, $b ) {
            $la = ! empty( $a->labels->singular_name ) ? $a->labels->singular_name : $a->name;
            $lb = ! empty( $b->labels->singular_name ) ? $b->labels->singular_name : $b->name;
            return strcasecmp( $la, $lb );
        } );

        $allowed_names = array_keys( $filtered_post_types );

        // --------------------------
        // Procesar envío de formulario
        // --------------------------
        // Detectar envío usando filter_input para evitar uso directo de $_POST
        $form_submitted = filter_input( INPUT_POST, 'fpct_pc_save', FILTER_DEFAULT );

        if ( $form_submitted !== null ) {

            // Verificar nonce ANTES de procesar cualquier dato del formulario.
            check_admin_referer( 'fpct_pc_admin_nonce_action', 'fpct_pc_admin_nonce' );

            // --- Post types seleccionados ---
            $raw_post_types = filter_input( INPUT_POST, 'fpct_pc_post_types', FILTER_DEFAULT, FILTER_REQUIRE_ARRAY );
            if ( ! is_array( $raw_post_types ) ) {
                $raw_post_types = array();
            }

            $sanitized_post_types = array();
            foreach ( $raw_post_types as $candidate ) {
                // usar sanitize_key para slugs/nombres de post_type (más estricto)
                $candidate = (string) $candidate;
                $candidate = sanitize_key( $candidate );
                if ( in_array( $candidate, $allowed_names, true ) ) {
                    $sanitized_post_types[] = $candidate;
                }
            }
            $sanitized_post_types = array_values( array_unique( $sanitized_post_types ) );
            update_option( 'fpct_pc_post_types', $sanitized_post_types );

            // --- Imagen por defecto (URL) ---
            $raw_img = filter_input( INPUT_POST, 'fpct_pc_default_image', FILTER_SANITIZE_URL );
            $raw_img = is_string( $raw_img ) ? trim( $raw_img ) : '';
            if ( $raw_img === '' ) {
                update_option( 'fpct_pc_default_image', '' );
            } else {
                $sanitized_img = esc_url_raw( $raw_img );
                update_option( 'fpct_pc_default_image', $sanitized_img );
            }

            // --- Enabled checkbox (booleano) ---
            // Filter_input con FILTER_VALIDATE_BOOLEAN devuelve true/false/null
            $enabled_bool = filter_input( INPUT_POST, 'fpct_pc_enabled', FILTER_VALIDATE_BOOLEAN, FILTER_NULL_ON_FAILURE );
            $enabled = $enabled_bool === null ? 0 : ( $enabled_bool ? 1 : 0 );
            update_option( 'fpct_pc_enabled', $enabled );

            // Mensaje de éxito escapado
            echo '<div class="notice notice-success is-dismissible"><p>' . esc_html__( 'Ajustes guardados.', 'featured-post-carousel-tag' ) . '</p></div>';
        }

        // --------------------------
        // Cargar opciones para mostrar en el formulario (sanitizadas)
        // --------------------------
        $enabled_post_types = (array) get_option( 'fpct_pc_post_types', array( 'post', 'page', 'product' ) );
        // Asegurarnos de que sólo haya nombres permitidos (re-sanitizar por si acaso)
        $enabled_post_types = array_values( array_intersect( array_map( 'sanitize_key', $enabled_post_types ), $allowed_names ) );

        $default_image = get_option( 'fpct_pc_default_image', '' );
        $default_image = esc_url( $default_image );

        $enabled_flag = intval( get_option( 'fpct_pc_enabled', 1 ) );
        ?>
        <div class="wrap">
            <h1><?php echo esc_html__( 'Featured Post Carousel by Tag', 'featured-post-carousel-tag' ); ?></h1>
            <p><?php echo esc_html__( 'Selecciona los tipos de contenido que quieres incluir en el carrusel. Usa la etiqueta "featured" o "destacado" en los ítems que quieras mostrar.', 'featured-post-carousel-tag' ); ?></p>

            <form method="post" action="">
                <?php wp_nonce_field( 'fpct_pc_admin_nonce_action', 'fpct_pc_admin_nonce' ); ?>

                <table class="form-table">
                    <tr>
                        <th scope="row"><?php echo esc_html__( 'Activar plugin', 'featured-post-carousel-tag' ); ?></th>
                        <td>
                            <label>
                                <input type="checkbox" name="fpct_pc_enabled" value="1" <?php checked( $enabled_flag, 1 ); ?> />
                                <?php echo esc_html__( 'Habilitar salida del shortcode', 'featured-post-carousel-tag' ); ?>
                            </label>
                        </td>
                    </tr>

                    <tr>
                        <th scope="row"><?php echo esc_html__( 'Tipos de contenido', 'featured-post-carousel-tag' ); ?></th>
                        <td>
                            <?php if ( empty( $filtered_post_types ) ) : ?>
                                <p><?php echo esc_html__( 'No hay tipos de contenido disponibles.', 'featured-post-carousel-tag' ); ?></p>
                            <?php else : ?>
                                <?php foreach ( $filtered_post_types as $pt ) :
                                    $label = ! empty( $pt->labels->singular_name ) ? $pt->labels->singular_name : $pt->name;
                                    $name  = (string) $pt->name;
                                    // mostrar nombre y label escapados
                                    ?>
                                    <label style="display:block; margin:5px 0;">
                                        <input type="checkbox" name="fpct_pc_post_types[]" value="<?php echo esc_attr( sanitize_key( $name ) ); ?>" <?php checked( in_array( sanitize_key( $name ), $enabled_post_types, true ) ); ?> />
                                        <strong><?php echo esc_html( $label ); ?></strong>
                                        <small style="color:#666;">(<?php echo esc_html( $name ); ?>)</small>
                                    </label>
                                <?php endforeach; ?>
                            <?php endif; ?>
                            <p class="description"><?php echo esc_html__( 'Selecciona entradas, páginas, productos u otros CPTs que desees que busque la etiqueta "featured" o "destacado".', 'featured-post-carousel-tag' ); ?></p>
                        </td>
                    </tr>

                    <tr>
                        <th scope="row"><label for="fpct_pc_default_image"><?php echo esc_html__( 'Imagen por defecto (URL)', 'featured-post-carousel-tag' ); ?></label></th>
                        <td>
                            <input type="url" id="fpct_pc_default_image" name="fpct_pc_default_image" value="<?php echo esc_attr( $default_image ); ?>" class="regular-text" />
                            <p class="description"><?php echo esc_html__( 'URL de la imagen que se mostrará si el ítem no tiene imagen destacada (dejar vacío para no mostrar).', 'featured-post-carousel-tag' ); ?></p>
                        </td>
                    </tr>
                </table>

                <p class="submit">
                    <input type="submit" name="fpct_pc_save" id="fpct_pc_save" class="button button-primary" value="<?php echo esc_attr__( 'Guardar cambios', 'featured-post-carousel-tag' ); ?>" />
                </p>
            </form>

            <hr>
            <h2 id="wpv-about"><?php echo esc_html__( 'Acerca del plugin', 'featured-post-carousel-tag' ); ?></h2>
            <p><?php echo esc_html__( 'Este plugin muestra contenido marcado con la etiqueta "featured" o "destacado" en un carrusel responsive (OwlCarousel2). Inserta el shortcode [featured-post-carousel-tag] en cualquier página o entrada.', 'featured-post-carousel-tag' ); ?></p>

            <p>
            <strong><?php echo esc_html__( '¿Y si mi página no tiene etiquetas activadas?', 'featured-post-carousel-tag' ); ?></strong><br>
            <?php echo esc_html__( 'Por defecto, WordPress no añade etiquetas ni categorías a las páginas.', 'featured-post-carousel-tag' ); ?><br>
            <?php echo esc_html__( 'Si deseas usarlas en páginas, añade este código en el functions.php de tu tema o en un plugin personalizado:', 'featured-post-carousel-tag' ); ?>
            <pre><?php echo esc_html( "function fpct_enable_tags_for_pages() {\n    // Añadir metabox de etiquetas a páginas\n    register_taxonomy_for_object_type( 'post_tag', 'page' );\n    // Añadir metabox de categorías a páginas\n    register_taxonomy_for_object_type( 'category', 'page' );\n}\nadd_action( 'init', 'fpct_enable_tags_for_pages' );" ); ?></pre>
            </p>
            <hr>
            <h3><?php echo esc_html__( 'Desarrollador', 'featured-post-carousel-tag' ); ?></h3>
            <p><strong><?php echo esc_html__( 'Daniel Esparza | Ninja WordPress & WooCommerce', 'featured-post-carousel-tag' ); ?></strong></p>
            <ul>
                <li><a href="https://d3veloper.es/" target="_blank" rel="noopener noreferrer"><?php echo esc_html__( 'Sitio Web', 'featured-post-carousel-tag' ); ?></a></li>
                <li><a href="https://www.instagram.com/d3veloper.es/" target="_blank" rel="noopener noreferrer"><?php echo esc_html__( 'Instagram', 'featured-post-carousel-tag' ); ?></a></li>
                <li><a href="mailto:hola@d3veloper.es" target="_blank" rel="noopener noreferrer">hola@d3veloper.es</a></li>
            </ul>
            <hr>
        </div>
        <?php
    }
}
