<?php
if ( ! defined( 'ABSPATH' ) ) exit;

/**
 * Fetches data from the custom post type table.
 *
 * @param string $type The type of data to fetch (e.g., 'post' or 'taxonomy').
 * @return array The fetched data.
 */
function fahimptb_get_data( $type ) {
    global $wpdb;
    $table_name = $wpdb->prefix . 'fahimptb_post_table';
    
    $cache_key = 'fahimptb_post_data_' . md5($type);
    $cached_data = wp_cache_get($cache_key, 'fahimptb_post_data');
    if ($cached_data === false) {
        $result = $wpdb->get_results($wpdb->prepare("SELECT * FROM $table_name WHERE `type` = %s", $type));
        wp_cache_set($cache_key, $result, 'fahimptb_post_data');
    } else {
        $result = $cached_data;
    }
    return $result;
}

/**
 * Creates custom post types based on data from the database.
 */
function fahimptb_create_post_types() {
    $result = fahimptb_get_data('post');
    foreach ( $result as $value ) {
        if ( 1 == $value->status ) {
            $public = ( 1 == $value->public ) ? true : false;
            $hierarchical = ( 1 == $value->hierarchical ) ? true : false;
            $advanced_settings = json_decode( $value->advanced_settings );
            $data = $advanced_settings->supports;
            
            add_filter( 'archive_template', function ( $template ) use ( $value ) {
                global $post_type;
                if ( $post_type == $value->post_type ) {
                    $plugin_dir = PTB_PLUGIN_DIR . 'view/frontend/templates';
                    return "$plugin_dir/fahimptb_archive.php";
                }
                return $template;
            });

            register_post_type( $value->post_type, array(
                'labels' => array(
                    'name' => esc_html( $value->title ),
                    'singular_name' => esc_html( $value->singular_name ),
                ),
                'public'         => $public,
                'show_in_menu'   => true,
                'menu_icon'      => isset($value->menu_icon) ? $value->menu_icon : null,
                'has_archive'    => $hierarchical,
                'rewrite'        => array( 'slug' => $value->post_type ),
                'supports'       => $data,
            ) );
        }
    }
}
add_action( 'init', 'fahimptb_create_post_types' );

/**
 * Creates custom taxonomies based on data from the database.
 */
function fahimptb_create_taxonomies() {
    global $wpdb;
    $table_name = $wpdb->prefix . 'fahimptb_post_table';

    // Fetch taxonomy data from cache or database
    $taxonomies = wp_cache_get('fahimptb_post_taxonomies', 'fahimptb_post_data');
    if ($taxonomies === false) {
        // Data not found in cache, fetch from database
        $result = fahimptb_get_data('taxonomy');
        $taxonomies = [];
        foreach ($result as $value) {
            if (1 == $value->status) {
                $public = (1 == $value->public) ? true : false;
                $hierarchical = (1 == $value->hierarchical) ? true : false;

                $post_types = [];
                $post_type_ids = $value->post_type;
                if (strpos($post_type_ids, ',') !== false) {
                    $ids = explode(',', $post_type_ids);
                    foreach ($ids as $id) {
                        $post_type_result = $wpdb->get_results($wpdb->prepare("SELECT * FROM $table_name WHERE `id` = %s", $id));
                        if (!empty($post_type_result)) {
                            $post_types[] = $post_type_result[0]->post_type;
                        }
                    }
                } else {
                    $post_type_result = $wpdb->get_results($wpdb->prepare("SELECT * FROM $table_name WHERE `id` = %s", $post_type_ids));
                    if (!empty($post_type_result)) {
                        $post_types[] = $post_type_result[0]->post_type;
                    } else {
                        $post_types[] = '';
                    }
                }

                $taxonomy_name = '';
                if (isset($value->fahimptb_taxonomies)) {
                    $taxonomy_name = $value->fahimptb_taxonomies;
                }

                // Add taxonomy data to array
                $taxonomies[] = array(
                    'name' => $taxonomy_name,
                    'post_types' => $post_types,
                    'label' => $value->title,
                    'hierarchical' => $hierarchical,
                    'public' => $public,
                    'rewrite' => array( 'slug' => $taxonomy_name ),
                );
            }
        }

        // Cache the fetched data
        wp_cache_set('fahimptb_post_taxonomies', $taxonomies, 'fahimptb_post_data');
    }

    // Register taxonomies
    foreach ($taxonomies as $taxonomy) {
        register_taxonomy(
            $taxonomy['name'],
            $taxonomy['post_types'],
            array(
                'label' => $taxonomy['label'],
                'hierarchical' => $taxonomy['hierarchical'],
                'public' => $taxonomy['public'],
                'show_ui' => true,
                'rewrite' => $taxonomy['rewrite'],
            )
        );
    }
}
add_action( 'init', 'fahimptb_create_taxonomies' );





