<?php
/**
 * Search functionality class.
 *
 * @package ExportPatternBlockLocation
 * @since   1.0.0
 */

namespace ExportPatternBlockLocation;

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit;

/**
 * Search class.
 *
 * Handles searching for CSS classes, Gutenberg blocks, and patterns.
 *
 * @since 1.0.0
 */
class Search {

	/**
	 * Search for CSS classes in HTML content.
	 *
	 * @since 1.0.0
	 *
	 * @param string $content        Post content.
	 * @param array  $target_classes Array of classes to search for.
	 * @return array Array with found classes and occurrence count.
	 */
	public static function css_classes( $content, $target_classes ) {
		$results = array();

		foreach ( $target_classes as $class ) {
			$class = trim( $class );
			if ( empty( $class ) ) {
				continue;
			}

			// Search in class attributes with different formats.
			// class="class", class='class', class="other class other".
			$pattern = '/class\s*=\s*["\'][^"\']*\b' . preg_quote( $class, '/' ) . '\b[^"\']*["\']/i';
			$count   = preg_match_all( $pattern, $content );

			if ( $count > 0 ) {
				$results[ $class ] = $count;
			}
		}

		return $results;
	}

	/**
	 * Search for Gutenberg blocks in content (recursively).
	 *
	 * @since 1.0.0
	 *
	 * @param array $blocks        Array of parsed blocks.
	 * @param array $target_blocks Array of block names to search for.
	 * @return array Array with found blocks and occurrence count.
	 */
	public static function blocks( $blocks, $target_blocks ) {
		$results = array();

		foreach ( $blocks as $block ) {
			$block_name = isset( $block['blockName'] ) ? $block['blockName'] : '';

			if ( ! empty( $block_name ) && in_array( $block_name, $target_blocks, true ) ) {
				if ( ! isset( $results[ $block_name ] ) ) {
					$results[ $block_name ] = 0;
				}
				++$results[ $block_name ];
			}

			// Search in nested blocks.
			if ( ! empty( $block['innerBlocks'] ) ) {
				$inner_results = self::blocks( $block['innerBlocks'], $target_blocks );
				foreach ( $inner_results as $name => $count ) {
					if ( ! isset( $results[ $name ] ) ) {
						$results[ $name ] = 0;
					}
					$results[ $name ] += $count;
				}
			}
		}

		return $results;
	}

	/**
	 * Search for patterns in content.
	 *
	 * @since 1.0.0
	 *
	 * @param string $content         Post content.
	 * @param array  $target_patterns Array of pattern slugs to search for.
	 * @return array Array with found patterns and occurrence count.
	 */
	public static function patterns( $content, $target_patterns ) {
		$results = array();

		foreach ( $target_patterns as $pattern_slug ) {
			$pattern_slug = trim( $pattern_slug );
			if ( empty( $pattern_slug ) ) {
				continue;
			}

			// Search for "patternName":"pattern-slug".
			$pattern = '/"patternName"\s*:\s*"' . preg_quote( $pattern_slug, '/' ) . '"/';
			$count   = preg_match_all( $pattern, $content );

			if ( $count > 0 ) {
				$results[ $pattern_slug ] = $count;
			}
		}

		return $results;
	}

	/**
	 * Get all post IDs matching criteria.
	 *
	 * @since 1.0.0
	 *
	 * @param array  $post_types  Array of post types.
	 * @param array  $post_status Array of post statuses.
	 * @param string $lang        WPML language code.
	 * @return array Array of post IDs.
	 */
	public static function get_post_ids( $post_types, $post_status, $lang = 'all' ) {
		$all_ids = array();

		// If WPML is active and we want all languages.
		if ( 'all' === $lang && function_exists( 'icl_object_id' ) ) {
			global $sitepress;

			if ( $sitepress ) {
				// phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedHooknameFound -- WPML hook.
				$active_languages = apply_filters( 'wpml_active_languages', null );

				foreach ( $active_languages as $lang_code => $language ) {
					// Switch to language.
					// phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedHooknameFound -- WPML hook.
					do_action( 'wpml_switch_language', $lang_code );

					$args = array(
						'post_type'        => $post_types,
						'post_status'      => $post_status,
						'posts_per_page'   => -1,
						'fields'           => 'ids',
						'orderby'          => 'ID',
						'order'            => 'ASC',
						'suppress_filters' => false,
					);

					$query   = new \WP_Query( $args );
					$all_ids = array_merge( $all_ids, $query->posts );
				}

				// Remove duplicates.
				$all_ids = array_unique( $all_ids );
			}
		} elseif ( 'all' !== $lang && function_exists( 'icl_object_id' ) ) {
			// Specific language.
			// phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedHooknameFound -- WPML hook.
			do_action( 'wpml_switch_language', $lang );

			$args = array(
				'post_type'        => $post_types,
				'post_status'      => $post_status,
				'posts_per_page'   => -1,
				'fields'           => 'ids',
				'orderby'          => 'ID',
				'order'            => 'ASC',
				'suppress_filters' => false,
			);

			$query   = new \WP_Query( $args );
			$all_ids = $query->posts;
		} else {
			// Without WPML or default language.
			$args = array(
				'post_type'      => $post_types,
				'post_status'    => $post_status,
				'posts_per_page' => -1,
				'fields'         => 'ids',
				'orderby'        => 'ID',
				'order'          => 'ASC',
			);

			$query   = new \WP_Query( $args );
			$all_ids = $query->posts;
		}

		return $all_ids;
	}

	/**
	 * Get the language of a post (WPML).
	 *
	 * @since 1.0.0
	 *
	 * @param int $post_id Post ID.
	 * @return string Language code or 'n/a'.
	 */
	public static function get_post_language( $post_id ) {
		if ( function_exists( 'wpml_get_language_information' ) ) {
			$lang_info = wpml_get_language_information( null, $post_id );
			return isset( $lang_info['language_code'] ) ? $lang_info['language_code'] : 'n/a';
		}
		return 'n/a';
	}
}
