<?php
/**
 * REST API functionality class.
 *
 * @package ExportPatternBlockLocation
 * @since   1.0.0
 */

namespace ExportPatternBlockLocation;

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit;

/**
 * REST_API class.
 *
 * Handles REST API endpoints for search and export.
 *
 * @since 1.0.0
 */
class REST_API {

	/**
	 * Constructor.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {
		add_action( 'rest_api_init', array( $this, 'register_routes' ) );
	}

	/**
	 * Register REST API routes.
	 *
	 * @since 1.0.0
	 */
	public function register_routes() {
		// Main search endpoint.
		register_rest_route(
			EPBL_REST_NAMESPACE,
			'/' . EPBL_REST_ROUTE,
			array(
				'methods'             => array( 'GET', 'POST' ),
				'args'                => $this->get_search_args(),
				'permission_callback' => array( $this, 'permission_check' ),
				'callback'            => array( $this, 'search_callback' ),
			)
		);
	}

	/**
	 * Get search endpoint arguments.
	 *
	 * @since 1.0.0
	 *
	 * @return array Arguments array.
	 */
	private function get_search_args() {
		return array(
			'token'            => array(
				'type'     => 'string',
				'required' => false,
			),
			'search-class-css' => array(
				'type'     => 'string',
				'required' => false,
			),
			'search-block'     => array(
				'type'     => 'string',
				'required' => false,
			),
			'search-pattern'   => array(
				'type'     => 'string',
				'required' => false,
			),
			'post_types'       => array(
				'type'     => 'string',
				'required' => false,
				'default'  => $this->get_default_post_types(),
			),
			'post_status'      => array(
				'type'     => 'string',
				'required' => false,
				'default'  => 'publish,draft,private',
			),
			'export_format'    => array(
				'type'     => 'string',
				'required' => false,
				'default'  => 'csv',
			),
			'download'         => array(
				'type'     => 'boolean',
				'required' => false,
				'default'  => true,
			),
			'filename'         => array(
				'type'     => 'string',
				'required' => false,
			),
			'lang'             => array(
				'type'     => 'string',
				'required' => false,
				'default'  => 'all',
			),
			'_wpnonce'         => array(
				'type'     => 'string',
				'required' => false,
			),
		);
	}

	/**
	 * Get default post types that support content editor.
	 *
	 * @since 1.0.0
	 *
	 * @return string Comma-separated list of post types.
	 */
	private function get_default_post_types() {
		// Get all public post types.
		$args = array(
			'public' => true,
		);
		
		$all_post_types = get_post_types( $args, 'names' );
		
		// Filter only those that have content editor support.
		$with_content = array();
		foreach ( $all_post_types as $post_type ) {
			if ( post_type_supports( $post_type, 'editor' ) ) {
				$with_content[] = $post_type;
			}
		}
		
		// Ensure we always have at least post and page if they exist.
		$builtin_fallback = array( 'post', 'page' );
		foreach ( $builtin_fallback as $fallback ) {
			if ( post_type_exists( $fallback ) && ! in_array( $fallback, $with_content, true ) ) {
				$with_content[] = $fallback;
			}
		}
		
		return implode( ',', $with_content );
	}

	/**
	 * Permission callback for search endpoint.
	 *
	 * @since 1.0.0
	 *
	 * @param \WP_REST_Request $request REST request object.
	 * @return bool|\WP_Error Whether the request has permission.
	 */
	public function permission_check( $request ) {
		$token = (string) $request->get_param( 'token' );

		// Check for admin token first.
		if ( $token && defined( 'EPBL_ACCESS_TOKEN' ) && hash_equals( EPBL_ACCESS_TOKEN, $token ) ) {
			return true;
		}

		// Check nonce for authenticated requests.
		$nonce        = $request->get_param( '_wpnonce' );
		$nonce_header = $request->get_header( 'X-WP-Nonce' );
		$nonce_value  = $nonce ? $nonce : $nonce_header;

		if ( $nonce_value && wp_verify_nonce( $nonce_value, 'wp_rest' ) ) {
			if ( current_user_can( 'manage_options' ) ) {
				return true;
			} else {
				return new \WP_Error(
					'rest_forbidden',
					__( 'Valid nonce but insufficient privileges. You need administrator privileges.', 'export-pattern-block-location' ),
					array( 'status' => 403 )
				);
			}
		}

		// Check user capability (for apiFetch requests with automatic nonce handling).
		if ( current_user_can( 'manage_options' ) ) {
			return true;
		}

		// More specific error messages for debugging.
		$user = wp_get_current_user();
		$error_message = __( 'Authentication failed.', 'export-pattern-block-location' );
		
			if ( $user && $user->ID > 0 ) {
				$error_message = sprintf( 
					/* translators: %s is the user login of the current user attempting the request. */
					__( 'User %s does not have administrator privileges to use this endpoint.', 'export-pattern-block-location' ),
					$user->user_login
				);
			} else {
			$error_message = __( 'No user authenticated. Please log in as administrator.', 'export-pattern-block-location' );
		}

		return new \WP_Error(
			'rest_forbidden',
			$error_message,
			array( 'status' => 403 )
		);
	}

	/**
	 * Search callback.
	 *
	 * @since 1.0.0
	 *
	 * @param \WP_REST_Request $request REST request object.
	 * @return \WP_REST_Response|\WP_Error Response object or error.
	 */
	public function search_callback( \WP_REST_Request $request ) {
		// phpcs:ignore WordPress.PHP.NoSilencedErrors.Discouraged
		@ignore_user_abort( true );

		$start_time = microtime( true );

		// Get and validate parameters.
		$params = $this->get_search_params( $request );

		if ( is_wp_error( $params ) ) {
			return $params;
		}

		// Only set time limit for download operations (not for preview/check operations).
		if ( $params['download'] ) {
			// phpcs:ignore WordPress.PHP.NoSilencedErrors.Discouraged, Squiz.PHP.DiscouragedFunctions.Discouraged -- Required only for long-running download operations to prevent timeout during CSV/JSON generation.
			@set_time_limit( 0 );
		}

		// Get all post IDs.
		$all_post_ids = Search::get_post_ids(
			$params['post_types_array'],
			$params['post_status_array'],
			$params['lang']
		);

		// Process posts and collect results.
		$results = $this->process_posts( $all_post_ids, $params );

		$execution_time = round( microtime( true ) - $start_time, 2 );

		// Generate response.
		return $this->generate_response(
			$results['data'],
			$results['total_checked'],
			$params,
			$execution_time
		);
	}

	/**
	 * Get and validate search parameters.
	 *
	 * @since 1.0.0
	 *
	 * @param \WP_REST_Request $request REST request object.
	 * @return array|\WP_Error Parameters array or error.
	 */
	private function get_search_params( \WP_REST_Request $request ) {
		$search_css      = $request->get_param( 'search-class-css' );
		$search_blocks   = $request->get_param( 'search-block' );
		$search_patterns = $request->get_param( 'search-pattern' );
		$post_types      = $request->get_param( 'post_types' );
		$post_status     = $request->get_param( 'post_status' );
		$export_format   = $request->get_param( 'export_format' );
		$download        = (bool) $request->get_param( 'download' );
		$filename        = $request->get_param( 'filename' );
		$lang            = $request->get_param( 'lang' );

		// Validate at least one search criteria.
		if ( empty( $search_css ) && empty( $search_blocks ) && empty( $search_patterns ) ) {
			return new \WP_Error(
				'missing_search_criteria',
				__( 'You must provide at least one search criteria: search-class-css, search-block or search-pattern', 'export-pattern-block-location' ),
				array( 'status' => 400 )
			);
		}

		// Parse search criteria.
		$css_classes = array();
		if ( ! empty( $search_css ) ) {
			$css_classes = array_filter( array_map( 'trim', explode( ',', $search_css ) ) );
		}

		$block_types = array();
		if ( ! empty( $search_blocks ) ) {
			$block_types = array_filter( array_map( 'trim', explode( ',', $search_blocks ) ) );
		}

		$pattern_slugs = array();
		if ( ! empty( $search_patterns ) ) {
			$pattern_slugs = array_filter( array_map( 'trim', explode( ',', $search_patterns ) ) );
		}

		// Parse post types and status.
		$post_types_array  = array_map( 'trim', explode( ',', $post_types ) );
		$post_status_array = array_map( 'trim', explode( ',', $post_status ) );

		// Validate post types - must be public and support editor.
		$valid_post_types = array();
		$all_public_types = get_post_types( array( 'public' => true ), 'names' );
		
		foreach ( $all_public_types as $post_type ) {
			if ( post_type_supports( $post_type, 'editor' ) ) {
				$valid_post_types[] = $post_type;
			}
		}
		
		$post_types_array = array_intersect( $post_types_array, $valid_post_types );

		if ( empty( $post_types_array ) ) {
			return new \WP_Error(
				'invalid_post_types',
				__( 'No valid post types found. Only public post types with editor support are allowed.', 'export-pattern-block-location' ),
				array( 'status' => 400 )
			);
		}

		return array(
			'css_classes'      => $css_classes,
			'block_types'      => $block_types,
			'pattern_slugs'    => $pattern_slugs,
			'post_types_array' => $post_types_array,
			'post_status_array' => $post_status_array,
			'export_format'    => $export_format,
			'download'         => $download,
			'filename'         => $filename,
			'lang'             => $lang,
		);
	}

	/**
	 * Process posts and collect search results.
	 *
	 * @since 1.0.0
	 *
	 * @param array $post_ids Array of post IDs.
	 * @param array $params   Search parameters.
	 * @return array Results with data and total_checked.
	 */
	private function process_posts( $post_ids, $params ) {
		$results             = array();
		$total_posts_checked = 0;
		$chunk_size          = 100;
		$chunks              = array_chunk( $post_ids, $chunk_size );

		foreach ( $chunks as $chunk ) {
			$posts = get_posts(
				array(
					'post__in'       => $chunk,
					'post_type'      => $params['post_types_array'],
					'post_status'    => $params['post_status_array'],
					'posts_per_page' => $chunk_size,
				)
			);

			foreach ( $posts as $post ) {
				++$total_posts_checked;
				$content   = $post->post_content;
				$post_lang = Search::get_post_language( $post->ID );
				$author    = get_the_author_meta( 'display_name', $post->post_author );

				// Base result data.
				$base_result = array(
					'post_id'       => $post->ID,
					'post_title'    => $post->post_title,
					'post_type'     => $post->post_type,
					'post_status'   => $post->post_status,
					'post_url'      => get_permalink( $post->ID ),
					'post_date'     => $post->post_date,
					'post_modified' => $post->post_modified,
					'post_author'   => $author,
					'lang'          => $post_lang,
				);

				// Search CSS classes.
				if ( ! empty( $params['css_classes'] ) ) {
					$css_found = Search::css_classes( $content, $params['css_classes'] );
					foreach ( $css_found as $class => $count ) {
						$results[] = array_merge(
							$base_result,
							array(
								'search_type' => 'css_class',
								'search_term' => $class,
								'occurrences' => $count,
							)
						);
					}
				}

				// Search blocks.
				if ( ! empty( $params['block_types'] ) ) {
					$blocks       = parse_blocks( $content );
					$blocks_found = Search::blocks( $blocks, $params['block_types'] );
					foreach ( $blocks_found as $block_name => $count ) {
						$results[] = array_merge(
							$base_result,
							array(
								'search_type' => 'block',
								'search_term' => $block_name,
								'occurrences' => $count,
							)
						);
					}
				}

				// Search patterns.
				if ( ! empty( $params['pattern_slugs'] ) ) {
					$patterns_found = Search::patterns( $content, $params['pattern_slugs'] );
					foreach ( $patterns_found as $pattern_slug => $count ) {
						$results[] = array_merge(
							$base_result,
							array(
								'search_type' => 'pattern',
								'search_term' => $pattern_slug,
								'occurrences' => $count,
							)
						);
					}
				}
			}

			// Free memory.
			wp_cache_flush();
		}

		return array(
			'data'          => $results,
			'total_checked' => $total_posts_checked,
		);
	}

	/**
	 * Generate response based on export format.
	 *
	 * @since 1.0.0
	 *
	 * @param array $results        Search results.
	 * @param int   $total_checked  Total posts checked.
	 * @param array $params         Search parameters.
	 * @param float $execution_time Execution time in seconds.
	 * @return \WP_REST_Response|\WP_Error Response object or error.
	 */
	private function generate_response( $results, $total_checked, $params, $execution_time ) {
		$file_name = Export::generate_filename(
			$params['filename'],
			'json' === $params['export_format'] ? 'json' : 'csv'
		);

		$search_criteria = array(
			'css_classes' => $params['css_classes'],
			'blocks'      => $params['block_types'],
			'patterns'    => $params['pattern_slugs'],
			'post_types'  => $params['post_types_array'],
			'post_status' => $params['post_status_array'],
			'lang'        => $params['lang'],
		);

		if ( 'json' === $params['export_format'] ) {
			return $this->json_response( $results, $total_checked, $search_criteria, $execution_time, $file_name, $params['download'] );
		}

		return $this->csv_response( $results, $total_checked, $search_criteria, $execution_time, $file_name, $params['download'] );
	}

	/**
	 * Generate JSON response.
	 *
	 * @since 1.0.0
	 *
	 * @param array  $results         Search results.
	 * @param int    $total_checked   Total posts checked.
	 * @param array  $search_criteria Search criteria.
	 * @param float  $execution_time  Execution time.
	 * @param string $file_name       Filename.
	 * @param bool   $download        Whether to create downloadable file.
	 * @return \WP_REST_Response Response object.
	 */
	private function json_response( $results, $total_checked, $search_criteria, $execution_time, $file_name, $download ) {
		$response = array(
			'success'             => true,
			'total_posts_checked' => $total_checked,
			'total_matches'       => count( $results ),
			'search_criteria'     => $search_criteria,
			'results'             => $results,
			'execution_time'      => $execution_time . 's',
		);

		if ( $download ) {
			// Generate JSON content directly.
			$json_content = wp_json_encode( $response, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE );
			
			// Set headers for direct download.
			header( 'Content-Type: application/json; charset=utf-8' );
			header( 'Content-Disposition: attachment; filename="' . $file_name . '"' );
			header( 'Pragma: no-cache' );
			header( 'Expires: 0' );
			header( 'Content-Length: ' . strlen( $json_content ) );

			// Output JSON content directly.
			// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- output is raw JSON for direct download.
			echo $json_content;
			exit;
		}

		return rest_ensure_response( $response );
	}

	/**
	 * Generate CSV response.
	 *
	 * @since 1.0.0
	 *
	 * @param array  $results         Search results.
	 * @param int    $total_checked   Total posts checked.
	 * @param array  $search_criteria Search criteria.
	 * @param float  $execution_time  Execution time.
	 * @param string $file_name       Filename.
	 * @param bool   $download        Whether to force download.
	 * @return \WP_REST_Response|\WP_Error Response object or error.
	 */
	private function csv_response( $results, $total_checked, $search_criteria, $execution_time, $file_name, $download ) {
		if ( $download ) {
			// Generate CSV content in memory.
			$csv_content = Export::generate_csv_content( $results );
			
			if ( is_wp_error( $csv_content ) ) {
				return $csv_content;
			}

			// Set headers for direct download.
			header( 'Content-Type: text/csv; charset=utf-8' );
			header( 'Content-Disposition: attachment; filename="' . $file_name . '"' );
			header( 'Pragma: no-cache' );
			header( 'Expires: 0' );
			header( 'Content-Length: ' . strlen( $csv_content ) );

			// Output CSV content directly.
			// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- output is raw CSV for direct download.
			echo $csv_content;
			exit;
		}

		// Return JSON response if not downloading.
		return rest_ensure_response(
			array(
				'success'             => true,
				'total_posts_checked' => $total_checked,
				'total_matches'       => count( $results ),
				'search_criteria'     => $search_criteria,
				'execution_time'      => $execution_time . 's',
			)
		);
	}
}
