<?php
/**
 * Main plugin class.
 *
 * @package ExportPatternBlockLocation
 * @since   1.0.0
 */

namespace ExportPatternBlockLocation;

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit;

/**
 * Plugin main class (Singleton).
 *
 * @since 1.0.0
 */
final class Plugin {

	/**
	 * Plugin version.
	 *
	 * @var string
	 */
	const VERSION = '1.0.0';

	/**
	 * Single instance of the class.
	 *
	 * @var Plugin|null
	 */
	private static $instance = null;

	/**
	 * Admin instance.
	 *
	 * @var Admin|null
	 */
	public $admin = null;

	/**
	 * REST API instance.
	 *
	 * @var REST_API|null
	 */
	public $rest_api = null;

	/**
	 * Get the single instance of the class.
	 *
	 * @since 1.0.0
	 *
	 * @return Plugin
	 */
	public static function instance() {
		if ( is_null( self::$instance ) ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Constructor.
	 *
	 * @since 1.0.0
	 */
	private function __construct() {
		$this->define_constants();
		$this->init_hooks();
	}

	/**
	 * Define plugin constants.
	 *
	 * @since 1.0.0
	 */
	private function define_constants() {
		// REST API token for authorization.
		if ( ! defined( 'EPBL_ACCESS_TOKEN' ) ) {
			define( 'EPBL_ACCESS_TOKEN', 'change-this-to-a-UNIQUE-and-long-token' );
		}

		// REST API namespace.
		if ( ! defined( 'EPBL_REST_NAMESPACE' ) ) {
			define( 'EPBL_REST_NAMESPACE', 'epbl-search/v1' );
		}

		// REST API route.
		if ( ! defined( 'EPBL_REST_ROUTE' ) ) {
			define( 'EPBL_REST_ROUTE', 'run' );
		}

		// Export directory.
		if ( ! defined( 'EPBL_EXPORT_DIR' ) ) {
			define( 'EPBL_EXPORT_DIR', EPBL_PLUGIN_DIR . 'exports/' );
		}
	}

	/**
	 * Initialize hooks.
	 *
	 * @since 1.0.0
	 */
	private function init_hooks() {
		add_action( 'plugins_loaded', array( $this, 'init' ) );
	}

	/**
	 * Initialize plugin components.
	 *
	 * @since 1.0.0
	 */
	public function init() {
		// Initialize admin.
		if ( is_admin() ) {
			$this->admin = new Admin();
		}

		// Initialize REST API (always, routes are registered via rest_api_init hook).
		$this->rest_api = new REST_API();
	}

	/**
	 * Get the plugin URL.
	 *
	 * @since 1.0.0
	 *
	 * @return string
	 */
	public function plugin_url() {
		return EPBL_PLUGIN_URL;
	}

	/**
	 * Get the plugin path.
	 *
	 * @since 1.0.0
	 *
	 * @return string
	 */
	public function plugin_path() {
		return EPBL_PLUGIN_DIR;
	}

	/**
	 * Prevent cloning.
	 *
	 * @since 1.0.0
	 */
	private function __clone() {}

	/**
	 * Prevent unserializing.
	 *
	 * @since 1.0.0
	 */
	public function __wakeup() {
		_doing_it_wrong( __FUNCTION__, esc_html__( 'Unserializing instances of this class is forbidden.', 'export-pattern-block-location' ), '1.0.0' );
	}
}
