<?php
/**
 * Export functionality class.
 *
 * @package ExportPatternBlockLocation
 * @since   1.0.0
 */

namespace ExportPatternBlockLocation;

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit;

/**
 * Export class.
 *
 * Handles CSV/JSON generation and file management.
 *
 * @since 1.0.0
 */
class Export {

	/**
	 * Generate filename with date prefix.
	 *
	 * @since 1.0.0
	 *
	 * @param string $custom_name Custom name (optional).
	 * @param string $extension   File extension.
	 * @return string Filename.
	 */
	public static function generate_filename( $custom_name = '', $extension = 'csv' ) {
		$date = gmdate( 'Ymd' );
		$time = gmdate( 'His' );

		if ( ! empty( $custom_name ) ) {
			// Sanitize custom name.
			$custom_name = sanitize_file_name( $custom_name );
			$custom_name = preg_replace( '/\.' . preg_quote( $extension, '/' ) . '$/', '', $custom_name );
			return "{$date}-{$custom_name}-{$time}.{$extension}";
		}

		return "{$date}-search-results-{$time}.{$extension}";
	}

	/**
	 * Create export directory if it doesn't exist.
	 *
	 * @since 1.0.0
	 *
	 * @return bool True if directory exists or was created.
	 */
	public static function ensure_export_dir() {
		if ( ! file_exists( EPBL_EXPORT_DIR ) ) {
			return wp_mkdir_p( EPBL_EXPORT_DIR );
		}
		return true;
	}

	/**
	 * Generate CSV content in memory.
	 *
	 * @since 1.0.0
	 *
	 * @param array $results Array of results.
	 * @return string|\WP_Error CSV content or error.
	 */
	public static function generate_csv_content( $results ) {
		// Use output buffering to capture CSV content.
		ob_start();
		
		// Create temporary stream in memory.
		// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fopen -- using php://temp stream for in-memory CSV construction.
		$handle = fopen( 'php://temp', 'r+' );
		
		if ( ! $handle ) {
			ob_end_clean();
			return new \WP_Error(
				'csv_error',
				__( 'Could not create CSV content.', 'export-pattern-block-location' )
			);
		}

		// BOM for UTF-8 (Excel compatibility).
		// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fwrite -- writing BOM into in-memory stream.
		fwrite( $handle, chr( 0xEF ) . chr( 0xBB ) . chr( 0xBF ) );

		// Headers.
		$headers = array(
			'search_type',
			'search_term',
			'post_id',
			'post_title',
			'post_type',
			'post_status',
			'post_url',
			'post_date',
			'post_modified',
			'post_author',
			'occurrences',
			'lang',
		);

		fputcsv( $handle, $headers, ';' );

		// Data.
		foreach ( $results as $result ) {
			fputcsv(
				$handle,
				array(
					$result['search_type'],
					$result['search_term'],
					$result['post_id'],
					$result['post_title'],
					$result['post_type'],
					$result['post_status'],
					$result['post_url'],
					$result['post_date'],
					$result['post_modified'],
					$result['post_author'],
					$result['occurrences'],
					$result['lang'],
				),
				';'
			);
		}

		// Get content from stream.
		rewind( $handle );
		$csv_content = stream_get_contents( $handle );
		// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fclose -- closing in-memory stream.
		fclose( $handle );
		ob_end_clean();

		return $csv_content;
	}

	/**
	 * Generate CSV with results.
	 *
	 * @since 1.0.0
	 *
	 * @param array  $results  Array of results.
	 * @param string $filename Filename.
	 * @return string|\WP_Error File path or error.
	 */
	public static function generate_csv( $results, $filename ) {
		self::ensure_export_dir();

		$csv_content = self::generate_csv_content( $results );
		if ( is_wp_error( $csv_content ) ) {
			return $csv_content;
		}

		$filesystem = self::get_filesystem();
		if ( is_wp_error( $filesystem ) ) {
			return $filesystem;
		}

		$filepath = EPBL_EXPORT_DIR . $filename;
		$written = $filesystem->put_contents( $filepath, $csv_content, FS_CHMOD_FILE );

		if ( false === $written ) {
			return new \WP_Error(
				'csv_error',
				__( 'Could not create CSV file.', 'export-pattern-block-location' )
			);
		}

		return $filepath;
	}

	/**
	 * Generate JSON file with results.
	 *
	 * @since 1.0.0
	 *
	 * @param array  $data     Data to export.
	 * @param string $filename Filename.
	 * @return string|\WP_Error File path or error.
	 */
	public static function generate_json( $data, $filename ) {
		self::ensure_export_dir();

		$filepath = EPBL_EXPORT_DIR . $filename;
		// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_file_put_contents
		$result = file_put_contents(
			$filepath,
			wp_json_encode( $data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE )
		);

		if ( false === $result ) {
			return new \WP_Error(
				'json_error',
				__( 'Could not create JSON file.', 'export-pattern-block-location' )
			);
		}

		return $filepath;
	}

	/**
	 * Get the WP_Filesystem instance.
	 *
	 * @since 1.0.0
	 *
	 * @return \WP_Filesystem_Base|\WP_Error Filesystem instance or error.
	 */
	private static function get_filesystem() {
		if ( ! function_exists( 'WP_Filesystem' ) ) {
			require_once ABSPATH . 'wp-admin/includes/file.php';
		}

		global $wp_filesystem;

		if ( ! is_object( $wp_filesystem ) ) {
			if ( ! WP_Filesystem() ) {
				return new \WP_Error(
					'filesystem_error',
					__( 'Unable to access the filesystem.', 'export-pattern-block-location' )
				);
			}
		}

		if ( ! is_object( $wp_filesystem ) ) {
			return new \WP_Error(
				'filesystem_error',
				__( 'The filesystem could not be initialized.', 'export-pattern-block-location' )
			);
		}

		return $wp_filesystem;
	}

	/**
	 * Delete an export file.
	 *
	 * @since 1.0.0
	 *
	 * @param string $filepath Full path to file.
	 * @return bool True if deleted, false otherwise.
	 */
	public static function delete_file( $filepath ) {
		// Validate file is in our exports directory.
		if ( strpos( $filepath, EPBL_EXPORT_DIR ) !== 0 ) {
			return false;
		}

		if ( file_exists( $filepath ) ) {
			// phpcs:ignore WordPress.WP.AlternativeFunctions.unlink_unlink
			return unlink( $filepath );
		}

		return false;
	}

	/**
	 * Get the URL for an export file.
	 *
	 * @since 1.0.0
	 *
	 * @param string $filename Filename.
	 * @return string File URL.
	 */
	public static function get_file_url( $filename ) {
		return EPBL_PLUGIN_URL . 'exports/' . $filename;
	}
}
