<?php
/**
 * Admin functionality class.
 *
 * @package ExportPatternBlockLocation
 * @since   1.0.0
 */

namespace ExportPatternBlockLocation;

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit;

/**
 * Admin class.
 *
 * Handles admin page and asset enqueuing.
 *
 * @since 1.0.0
 */
class Admin {

	/**
	 * Admin page hook suffix.
	 *
	 * @var string
	 */
	private $hook_suffix = '';

	/**
	 * Constructor.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {
		add_action( 'admin_menu', array( $this, 'add_admin_menu' ) );
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_scripts' ) );
		add_filter( 'plugin_action_links_' . EPBL_PLUGIN_BASENAME, array( $this, 'add_settings_link' ) );
	}

	/**
	 * Add Settings link to the plugin row actions.
	 *
	 * @since 1.0.0
	 *
	 * @param array $links Existing action links.
	 * @return array Modified action links.
	 */
	public function add_settings_link( $links ) {
		$settings_url = admin_url( 'tools.php?page=epbl-export-locations' );

		array_unshift(
			$links,
			sprintf(
				'<a href="%1$s">%2$s</a>',
				esc_url( $settings_url ),
				esc_html__( 'Settings', 'export-pattern-block-location' )
			)
		);

		return $links;
	}

	/**
	 * Add admin menu page under Tools.
	 *
	 * @since 1.0.0
	 */
	public function add_admin_menu() {
		$this->hook_suffix = add_management_page(
			__( 'Export Pattern & Block Locations', 'export-pattern-block-location' ),
			__( 'Export Pattern & Block Locations', 'export-pattern-block-location' ),
			'manage_options',
			'epbl-export-locations',
			array( $this, 'render_page' )
		);
	}

	/**
	 * Enqueue admin scripts and styles.
	 *
	 * @since 1.0.0
	 *
	 * @param string $hook_suffix The current admin page.
	 */
	public function enqueue_scripts( $hook_suffix ) {
		// Only load on our specific page.
		if ( $this->hook_suffix !== $hook_suffix && 'tools_page_epbl-export-locations' !== $hook_suffix ) {
			return;
		}

		// Enqueue WordPress components style.
		wp_enqueue_style( 'wp-components' );

		// Check for build file (production mode).
		$build_asset_file = EPBL_PLUGIN_DIR . 'assets/build/admin.asset.php';
		$is_production    = file_exists( $build_asset_file );

		// Enqueue styles.
		if ( $is_production && file_exists( EPBL_PLUGIN_DIR . 'assets/build/style-admin.css' ) ) {
			$asset = require $build_asset_file;
			wp_enqueue_style(
				'epbl-admin',
				EPBL_PLUGIN_URL . 'assets/build/style-admin.css',
				array( 'wp-components' ),
				$asset['version']
			);
		} else {
			// Development mode - load source CSS.
			wp_enqueue_style(
				'epbl-admin',
				EPBL_PLUGIN_URL . 'assets/src/scss/admin/style.css',
				array( 'wp-components' ),
				Plugin::VERSION
			);
		}

		// Enqueue scripts.
		if ( $is_production ) {
			$asset = require $build_asset_file;
			wp_enqueue_script(
				'epbl-admin',
				EPBL_PLUGIN_URL . 'assets/build/admin.js',
				$asset['dependencies'],
				$asset['version'],
				true
			);
		} else {
			// Development mode - load source JS.
			wp_enqueue_script(
				'epbl-admin',
				EPBL_PLUGIN_URL . 'assets/src/js/admin/index.js',
				array( 'wp-element', 'wp-components', 'wp-api-fetch', 'wp-i18n' ),
				Plugin::VERSION,
				true
			);
		}

		// Pass data to script.
		wp_localize_script(
			'epbl-admin',
			'epblExportData',
			array(
				'apiUrl'  => rest_url( EPBL_REST_NAMESPACE . '/' . EPBL_REST_ROUTE ),
				'nonce'   => wp_create_nonce( 'wp_rest' ),
				'version' => Plugin::VERSION,
			)
		);
	}

	/**
	 * Render the admin page.
	 *
	 * @since 1.0.0
	 */
	public function render_page() {
		?>
		<div class="wrap">
			<h1><?php echo esc_html( get_admin_page_title() ); ?></h1>
			<div id="epbl-export-locations-app"></div>
		</div>
		<?php
	}
}
