/*!
 * Source files available in assets/src/
 */

/**
 * Export Pattern & Block Location - Admin Interface
 *
 * Uses native WordPress components (@wordpress/components)
 *
 * @package ExportPatternBlockLocation
 * @since   1.0.0
 */

import { createElement as el, useState, render } from "@wordpress/element";
import {
  TextareaControl,
  SelectControl,
  Button,
  Notice,
  Spinner,
  Card,
  CardBody,
  CardHeader,
  __experimentalHeading as Heading,
  __experimentalVStack as VStack,
  __experimentalHStack as HStack,
} from "@wordpress/components";
import apiFetch from "@wordpress/api-fetch";
import { __ } from "@wordpress/i18n";

/**
 * Main component.
 */
function EPBLExportApp() {
  const [searchTerms, setSearchTerms] = useState("");
  const [searchType, setSearchType] = useState("search-pattern");
  const [isLoading, setIsLoading] = useState(false);
  const [notice, setNotice] = useState(null);

  /**
   * Handle search and download.
   */
  const handleExport = async () => {
    // Validate terms exist.
    if (!searchTerms.trim()) {
      setNotice({
        type: "error",
        message: __(
          "Please enter at least one search term.",
          "export-pattern-block-location"
        ),
      });
      return;
    }

    setIsLoading(true);
    setNotice(null);

    try {
      const apiPathBase = "/epbl-search/v1/run";

      // Build parameters for initial check.
      const checkParams = new URLSearchParams({
        [searchType]: searchTerms,
        download: "0",
        export_format: "csv",
      });

      // First, check if there are results (no navigation, shows errors inline).
      const response = await apiFetch({
        path: `${apiPathBase}?${checkParams.toString()}`,
        method: "GET",
      });

      // Check if there are results.
      if (response.total_matches === 0) {
        setNotice({
          type: "warning",
          message:
            __(
              "No results found for the specified terms.",
              "export-pattern-block-location"
            ) +
            " " +
            response.total_posts_checked +
            " " +
            __("posts checked.", "export-pattern-block-location"),
        });
        setIsLoading(false);
        return;
      }

      // If results exist, show success message and trigger download.
      setNotice({
        type: "success",
        message:
          __("Success!", "export-pattern-block-location") +
          " " +
          response.total_matches +
          " " +
          __("matches found in", "export-pattern-block-location") +
          " " +
          response.total_posts_checked +
          " " +
          __("posts. Downloading file...", "export-pattern-block-location"),
      });

      // Generate download parameters with nonce.
      const downloadParams = new URLSearchParams({
        [searchType]: searchTerms,
        download: "1",
        export_format: "csv",
        _wpnonce: window.epblExportData?.nonce || "",
      });

      // Trigger download using apiFetch to keep the user on the page and show errors inline.
      try {
        const downloadResponse = await apiFetch({
          path: `${apiPathBase}?${downloadParams.toString()}`,
          method: "GET",
          parse: false,
          headers: {
            "X-WP-Nonce": window.epblExportData?.nonce || "",
          },
        });

        if (!downloadResponse.ok) {
          let errorMessage = `HTTP ${downloadResponse.status}: ${downloadResponse.statusText}`;
          try {
            const errorJson = await downloadResponse.json();
            if (errorJson?.message) {
              errorMessage = errorJson.message;
            }
          } catch (e) {
            // Ignore JSON parse errors and keep default message.
          }

          throw new Error(errorMessage);
        }

        // Get the filename from Content-Disposition header or generate default.
        const contentDisposition = downloadResponse.headers.get(
          "Content-Disposition"
        );
        let filename = "search-results.csv";
        if (contentDisposition) {
          const filenameMatch =
            contentDisposition.match(/filename="?([^"]+)"?/);
          if (filenameMatch) {
            filename = filenameMatch[1];
          }
        }

        // Convert response to blob and trigger download.
        const blob = await downloadResponse.blob();
        const url = window.URL.createObjectURL(blob);
        const link = document.createElement("a");
        link.href = url;
        link.download = filename;
        document.body.appendChild(link);
        link.click();
        document.body.removeChild(link);
        window.URL.revokeObjectURL(url);
      } catch (downloadError) {
        throw new Error(`Download failed: ${downloadError.message}`);
      }
    } catch (error) {
      setNotice({
        type: "error",
        message:
          __("Error:", "export-pattern-block-location") +
          " " +
          (error.message ||
            __(
              "Could not complete the search.",
              "export-pattern-block-location"
            )),
      });
    } finally {
      setIsLoading(false);
    }
  };

  return el(
    "div",
    {
      className: "epbl-export-locations-container",
      style: { maxWidth: "800px", marginTop: "20px" },
    },
    notice &&
      el(
        Notice,
        {
          status: notice.type,
          isDismissible: true,
          onRemove: () => setNotice(null),
        },
        notice.message
      ),
    el(
      Card,
      null,
      el(
        CardHeader,
        null,
        el(
          Heading,
          { level: 2 },
          __("Search and Export Locations", "export-pattern-block-location")
        )
      ),
      el(
        CardBody,
        null,
        el(
          VStack,
          { spacing: 4 },
          el(SelectControl, {
            label: __("Search type", "export-pattern-block-location"),
            value: searchType,
            options: [
              {
                label: __("Patterns", "export-pattern-block-location"),
                value: "search-pattern",
              },
              {
                label: __("Gutenberg Blocks", "export-pattern-block-location"),
                value: "search-block",
              },
              {
                label: __("CSS Classes", "export-pattern-block-location"),
                value: "search-class-css",
              },
            ],
            onChange: (value) => setSearchType(value),
            help:
              searchType === "search-pattern"
                ? __(
                    "Search patterns like: blog-featured, product-cat, etc.",
                    "export-pattern-block-location"
                  )
                : searchType === "search-block"
                ? __(
                    "Search blocks like: core/button, acf/hero, etc.",
                    "export-pattern-block-location"
                  )
                : __(
                    "Search CSS classes like: btn-primary, container, etc.",
                    "export-pattern-block-location"
                  ),
          }),
          el(TextareaControl, {
            label: __(
              "Search terms (separated by commas)",
              "export-pattern-block-location"
            ),
            value: searchTerms,
            onChange: (value) => setSearchTerms(value),
            placeholder:
              searchType === "search-pattern"
                ? "blog-featured, product-cat, global-featured"
                : searchType === "search-block"
                ? "core/button, acf/hero, core/gallery"
                : "btn-primary, container, hero-section",
            rows: 5,
            help: __(
              "Enter one or more terms separated by commas. They will be searched in all posts, pages and CPTs.",
              "export-pattern-block-location"
            ),
          }),
          el(
            Notice,
            {
              status: "info",
              isDismissible: false,
              className: "epbl-info-notice",
            },
            el(
              "div",
              null,
              el(
                "strong",
                null,
                __("Information:", "export-pattern-block-location")
              ),
              el(
                "ul",
                { style: { marginTop: "8px", marginBottom: "0" } },
                el(
                  "li",
                  null,
                  __(
                    "Search will be performed on all post types with content editor support",
                    "export-pattern-block-location"
                  )
                ),
                el(
                  "li",
                  null,
                  __(
                    "Only published, draft and private content will be included",
                    "export-pattern-block-location"
                  )
                ),
                el(
                  "li",
                  null,
                  __(
                    "CSV file will be downloaded directly if there are results",
                    "export-pattern-block-location"
                  )
                ),
                el(
                  "li",
                  null,
                  __(
                    "If no results (0), no file will be downloaded",
                    "export-pattern-block-location"
                  )
                )
              )
            )
          ),
          el(
            HStack,
            { justify: "flex-start" },
            el(
              Button,
              {
                variant: "primary",
                onClick: handleExport,
                disabled: isLoading || !searchTerms.trim(),
                icon: isLoading ? el(Spinner) : null,
              },
              isLoading
                ? __("Searching...", "export-pattern-block-location")
                : __("Search and Download CSV", "export-pattern-block-location")
            )
          )
        )
      )
    ),
    el(
      Card,
      { style: { marginTop: "20px" } },
      el(
        CardHeader,
        null,
        el(
          Heading,
          { level: 3 },
          __("Usage examples", "export-pattern-block-location")
        )
      ),
      el(
        CardBody,
        null,
        el(
          "div",
          null,
          el(
            "p",
            null,
            el("strong", null, __("Patterns:", "export-pattern-block-location"))
          ),
          el("code", null, "blog-featured, product-cat, global-featured"),
          el(
            "p",
            { style: { marginTop: "16px" } },
            el(
              "strong",
              null,
              __("Gutenberg Blocks:", "export-pattern-block-location")
            )
          ),
          el(
            "code",
            null,
            "core/button, core/gallery, acf/hero, woocommerce/product-grid"
          ),
          el(
            "p",
            { style: { marginTop: "16px" } },
            el(
              "strong",
              null,
              __("CSS Classes:", "export-pattern-block-location")
            )
          ),
          el("code", null, "btn-primary, container, hero-section, cta-button")
        )
      )
    )
  );
}

// Render app when DOM is ready.
document.addEventListener("DOMContentLoaded", function () {
  const container = document.getElementById("epbl-export-locations-app");
  if (container) {
    render(el(EPBLExportApp), container);
  }
});
