<?php declare(strict_types=1);

namespace Expedico;

use WC_Shipping_Zones;

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Plugin Name:         Expedico
 * Description:         Single point for your ecommerce parcel delivery, returns handling and fulfillment service mainly in Eastern Europe.
 * Version:             2.0.6
 * Author:              Expedico
 * Author URI:          https://expedico.eu
 * License:             GPL-2.0
 * License URI:         https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain:         expedico
 * WordPress Available: yes
 * Requires License:    no
 * Text Domain:         expedico
 * Domain Path:         /languages
 * Requires PHP:        7.4
 */
final class Expedico
{
    public const DOMAIN = 'expedico';
	public const SETTINGS_PAGE_SLUG = self::DOMAIN . '_settings';
	public const TRACKING_URL = 'https://expedico.eu/tracking?code=';
	public const SHIPPING_CARRIER_SYS_NAME_META = 'expedico_carrier_id';
	public const PICKUP_POINT_ID_META = 'expedico_pickup_id';
	public const EXPEDICO_PARCEL_ID = 'expedico_parcel_id';

	private string $action = 'expedico_download_labels';

	private ExpedicoSettings $settings;
	private ExpedicoAPI $expedicoAPI;
	private ExpedicoOrderStatus $expedicoOrderStatus;

    public function __construct()
    {
	    require_once __DIR__ . '/ExpedicoSettings.php';
	    require_once __DIR__ . '/DeliveryShippingMethod.php';
	    require_once __DIR__ . '/PickupPointsShippingMethod.php';
	    require_once __DIR__ . '/ExpedicoAPI.php';
		require_once __DIR__ . '/ExpedicoOrderStatus.php';

	    $this->expedicoAPI = new ExpedicoAPI();
	    $this->settings = new ExpedicoSettings($this->expedicoAPI);
		$this->expedicoOrderStatus = new ExpedicoOrderStatus($this->expedicoAPI);

		// load translations
        add_action('init', [$this, 'loadTranslation']);

	    $this->cron();

	    // add settings link to plugins page
	    add_filter( 'plugin_action_links_' . plugin_basename(__FILE__), function (array $links) {
		    $links[] = $this->getSettingsUrl();
		    return $links;
	    });

		if (!$this->settings->are_credentials_set()) {
			add_action('admin_notices', function (): void {
				printf('<div class="notice notice-error"><p>' . __('Please enter your Expedico credentials in order to use the plugin. You can set it up in', 'expedico') . ' ' .  $this->getSettingsUrl() .  '</p></div>');
			});

			return;
		}

		// init woocommerce
        add_action('woocommerce_init', function (): void {
			$addActionFn = function (array $actions): array {
				$actions[$this->action] = __('Expedico - Download labels', 'expedico');

				return $actions;
			};
			$addOrderActions = function ($actions) {
				global $theorder;

				// Get the order object from global $theorder if not set
				if ( ! is_object( $theorder ) ) {
					$order_id = absint( $_GET['post'] ?? 0 );
					$theorder = wc_get_order( $order_id );
				}

				if (!$theorder) {
					return $actions;
				}

				if ($theorder->get_meta(self::EXPEDICO_PARCEL_ID)) {
					$actions['expedico_download_label'] = __('Expedico - Download label', 'expedico');
				} else {
					$actions['expedico_resend_order'] = __('Expedico - Resend order', 'expedico');
				}

				return $actions;
			};
			$downloadLabelsFn = function (string $redirectUrl, string $action, array $orderIds): string {
				if ($action !== $this->action) {
					return $redirectUrl;
				}

				$parcelIds = [];
				foreach ($orderIds as $order_id) {
					// Get the order object
					$order = wc_get_order($order_id);

					// Continue if no order is found
					if (!$order) {
						continue;
					}

					// Check if the order has the 'expedico_parcel_id' meta key
					$expedico_parcel_id = $order->get_meta(self::EXPEDICO_PARCEL_ID);

					// Add the order to the result if it has the 'expedico_parcel_id' meta key
					if (!empty($expedico_parcel_id)) {
						$parcelIds[] = $expedico_parcel_id;
					}
				}

				if (!$parcelIds) {
					return $redirectUrl;
				}

				$labels = $this->expedicoAPI->getLabelsPdfString($parcelIds);

				if (!$labels) {
					return add_query_arg([$this->action => TRUE], $redirectUrl);
				}

				header('Content-type: application/pdf');
				header('Content-Disposition: attachment; filename=labels.pdf');
				echo $labels;

				exit(0);
			};
			$downloadOrderLabel = function (\WC_Order $order){
				$label = $this->expedicoAPI->getLabelsPdfString([$order->get_meta(self::EXPEDICO_PARCEL_ID)]);

				if (!$label) {
					wp_die(__('Something gone wrong when calling Expedico.eu API!', 'expedico'));
				}

				header('Content-type: application/pdf');
				header('Content-Disposition: attachment; filename=labels.pdf');
				echo $label;

				exit(0);
			};
	        $resendOrderAction = function (\WC_Order $order){
		        $this->expedicoAPI->createParcel($order);
	        };

            add_filter(
                'bulk_actions-woocommerce_page_wc-orders',
                $addActionFn,
                100,
                1,
            );
            add_filter(
	            'bulk_actions-edit-shop_order',
	            $addActionFn,
	            100,
	            1,
            );

	        add_filter(
		        'handle_bulk_actions-woocommerce_page_wc-orders',
		        $downloadLabelsFn,
		        100,
		        3
	        );
            add_filter(
                'handle_bulk_actions-edit-shop_order',
                $downloadLabelsFn,
                100,
                3
            );

	        add_filter(
		        'woocommerce_order_actions',
		        $addOrderActions,
		        100,
		        1,
	        );

	        add_action('woocommerce_order_action_expedico_download_label', $downloadOrderLabel);
	        add_action('woocommerce_order_action_expedico_resend_order', $resendOrderAction);
        });

        add_filter('woocommerce_shipping_methods', [$this, 'add_shipping_method']);

		add_filter('woocommerce_cart_shipping_method_full_label', function ($label, $method) {
			// $method is a WC_Shipping_Rate object
			if (!is_object($method) || !method_exists($method, 'get_id')) {
				return $label;
			}
		
			if (strpos($method->get_id(), DeliveryShippingMethod::EXPEDICO_METHOD_ID) !== false) {
				$availableCarriers = $this->settings->getAvailableDeliveryCarriersWithPrices(); // or however you get it
		
				if (count($availableCarriers) === 1) {
					return 'Expedico delivery'; // return empty label to suppress WooCommerce default output
				}
			}
		
			return $label;
		}, 10, 2);
		

        add_action('admin_notices', function (): void {
            if (!isset($_REQUEST[$this->action])) {
                return;
            }

            printf('<div class="notice notice-error"><p>' . __('Something gone wrong when calling Expedico.eu API!', 'expedico') . '</p></div>');
        });

        add_action('woocommerce_after_shipping_rate', [$this, 'renderWidgetButtonTableRow'], 10, 2);
        add_action('woocommerce_checkout_process', [$this, 'validateCheckoutData']);
        add_action('woocommerce_checkout_order_created', [$this, 'orderCreated']);

		// shipping recalculation
	    add_action('wp_enqueue_scripts', function() {
		    if (is_checkout()) {
			    wp_enqueue_script('recalculateShippingJs', plugins_url('public/js/recalculateShipping.js', __FILE__ ), [ 'jquery' ], '', true);
			    wp_localize_script('recalculateShippingJs', 'custom_shipping_params', [
				    'ajax_url' => admin_url('admin-ajax.php'),
				    'nonce' => wp_create_nonce('custom-shipping-nonce'),
			    ] );
		    }
	    });

	    $handleCustomShippingOption = function () {
		    check_ajax_referer('custom-shipping-nonce', 'nonce');
		    if (isset($_POST['deliveryId'])) {
			    WC()->session->set('expedicoDeliveryId', sanitize_text_field($_POST['deliveryId']));
		    }

		    if (isset($_POST['pickupPointId'])) {
			    WC()->session->set('expedicoPickupPointId', sanitize_text_field($_POST['pickupPointId']));

		    }
		    wp_send_json_success();
	    };
	    add_action('wp_ajax_save_custom_shipping_option', $handleCustomShippingOption);
	    add_action('wp_ajax_nopriv_save_custom_shipping_option', $handleCustomShippingOption);

	    $adjust_shipping_rates_based_on_option = function ($rates, $package) {
		    $deliveryId = WC()->session->get('expedicoDeliveryId');
		    $pickupPointId = WC()->session->get('expedicoPickupPointId');

		    if (!empty($deliveryId)) {
			    foreach ($rates as &$rate) {
				    if ($rate->method_id === DeliveryShippingMethod::EXPEDICO_METHOD_ID) {
					    $is_taxable = false;

					    $shipping_zone = WC_Shipping_Zones::get_shipping_method($rate->instance_id);
					    if ($shipping_zone) {
						    $is_taxable = $shipping_zone->get_option('tax_status') === 'taxable';
					    }

					    ['price_without_tax' => $price, 'tax_object' => $tax_object] = ExpedicoSettings::calculatePriceWithTax($this->settings->get_carrierPrice($deliveryId), $is_taxable && wc_tax_enabled());

					    $rate->cost = $price;
					    $rate->label .= ': ' . $this->expedicoAPI->getCarriers()[$deliveryId];

					    if ($is_taxable) {
						    $rate->taxes = $tax_object;
					    } else {
						    $rate->taxes = false;
					    }
				    }
			    }
		    }

		    if (!empty($pickupPointId)) {
			    $pickupPoint = $this->expedicoAPI->getPickupPointDetails((int) $pickupPointId);
				//$pickupPoint = [
				//	'carrierId' => 166,
				//	'nameLocal' => 'test'
				//];

				if ($pickupPoint) {
					foreach ($rates as &$rate) {
						if ($rate->method_id === PickupPointsShippingMethod::EXPEDICO_METHOD_ID) {
							$is_taxable = false;

							$shipping_zone = WC_Shipping_Zones::get_shipping_method($rate->instance_id);
							if ($shipping_zone) {
								$is_taxable = $shipping_zone->get_option('tax_status') === 'taxable';
							}


							['price_without_tax' => $price, 'tax_object' => $tax_object] = ExpedicoSettings::calculatePriceWithTax($this->settings->get_carrierPrice((int) $pickupPoint['carrierId']), $is_taxable && wc_tax_enabled());

							$rate->cost = $price;
							$rate->label .= ': ' . $pickupPoint['nameLocal'];

							if ($is_taxable) {
								$rate->taxes = $tax_object;
							} else {
								$rate->taxes = false;
							}
						}
					}
				}
		    }
		    return $rates;
	    };
	    add_filter('woocommerce_package_rates', $adjust_shipping_rates_based_on_option, 20, 2);

	    add_filter('woocommerce_cart_shipping_packages', function ($packages) {
		    foreach ($packages as &$package) {
			    $package['rate_cache'] = wp_rand();
		    }
		    return $packages;
	    }, 100);

		// uninstall hook
        register_uninstall_hook(__FILE__, [__CLASS__, 'uninstall']);
    }

	public function getSettingsUrl(): string
	{
		$url = get_admin_url() . "options-general.php?page=" . self::SETTINGS_PAGE_SLUG;
		return '<a href="' . $url . '">' . __('Settings', 'expedico') . '</a>';
	}

    public function validateCheckoutData(): void
    {
		$shippingMethods = implode('', wc_get_chosen_shipping_method_ids());

	    if (str_contains($shippingMethods, PickupPointsShippingMethod::EXPEDICO_METHOD_ID)) {
		    if (!array_key_exists('pickupId', $_COOKIE)) {
			    wc_add_notice(__('Pickup point is not selected.', 'expedico'), 'error');
		    }
		}

	    if (str_contains($shippingMethods, DeliveryShippingMethod::EXPEDICO_METHOD_ID)) {
		    if (empty($_POST['expedico_shipping_carrier'])) {
			    wc_add_notice(__('Shipping carrier is not selected.', 'expedico'), 'error');
		    }
	    }
    }

    public function orderCreated(\WC_Order $order): void
    {
		$expedicoShippingMethod = false;

	    foreach ($order->get_shipping_methods() as $shipping_method ) {
		    if ($shipping_method->get_method_id() === DeliveryShippingMethod::EXPEDICO_METHOD_ID) {
				if (!empty($_POST['expedico_shipping_carrier'])) {
					$carrierSystemName = $this->expedicoAPI->getCarriers('system')[$_POST['expedico_shipping_carrier']];
					$order->add_meta_data(self::SHIPPING_CARRIER_SYS_NAME_META, $carrierSystemName);
				}

				$expedicoShippingMethod = DeliveryShippingMethod::EXPEDICO_METHOD_ID;
				break;
		    } elseif ($shipping_method->get_method_id() === PickupPointsShippingMethod::EXPEDICO_METHOD_ID) {
			    $order->add_meta_data(self::PICKUP_POINT_ID_META, (int) sanitize_text_field($_COOKIE['pickupId']));
			    setcookie("pickupId", "", time() - 3600);

				$expedicoShippingMethod = PickupPointsShippingMethod::EXPEDICO_METHOD_ID;
				break;
		    }
		}

		if ($expedicoShippingMethod === false) {
			return;
		}

		$this->expedicoAPI->createParcel($order);

		$order->save_meta_data();

	    WC()->session->set('expedicoDeliveryId', null);
    }

    public function renderWidgetButtonTableRow(\WC_Shipping_Rate $shippingRate, int $index): void
    {

		if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['expedico_shipping_carrier'])) {
			WC()->session->set('expedico_shipping_carrier', sanitize_text_field($_POST['expedico_shipping_carrier']));
			WC()->session->set('expedicoDeliveryId', sanitize_text_field($_POST['expedico_shipping_carrier']));
		}
		
        $method = WC()->session->get('chosen_shipping_methods')[$index];

	    if (!is_checkout()) {
		    return;
	    }

        if (str_contains($method, PickupPointsShippingMethod::EXPEDICO_METHOD_ID) && $method === $shippingRate->get_id()) {
	        //setcookie("pickupId", "", time() - 3600);

	        echo wp_kses(
		        str_replace(
			        ['optionsPlaceholder', 'choosePickupPlaceholder', 'widgetLibraryPlaceholder'],
			        [
				        json_encode(
					        [
						        "carrier_ids" => $this->settings->get_carriersIds(),
					        ]
				        ),
				        __('Choose pickup point', 'expedico'),
				        plugins_url('public/js/widget.js', __FILE__ ),
			        ],
			        file_get_contents(__DIR__ . "/choose-pickup.html")
		        ),
		        [
			        'button' => ['type' => [], 'onclick' => []],
			        'p'      => ['id' => []],
			        'script' => [],
		        ]
	        );
        }

	    if (str_contains($method, DeliveryShippingMethod::EXPEDICO_METHOD_ID) && $method === $shippingRate->get_id()) {
			$availableCarriers = $this->settings->getAvailableDeliveryCarriersWithPrices();
	
			if (count($availableCarriers) === 1) {
				$defaultCarrierId = array_key_first($availableCarriers);
				WC()->session->set('expedico_shipping_carrier', $defaultCarrierId);
				WC()->session->set('expedicoDeliveryId', $defaultCarrierId);
		
			
				echo '<input type="hidden" name="expedico_shipping_carrier" value="' . esc_attr($defaultCarrierId) . '">';

				echo '<p>' . esc_html($availableCarriers[$defaultCarrierId]) . '</p>';


			} else {
			 $selectedCarrier = $_POST['expedico_shipping_carrier'] ?? WC()->session->get('expedico_shipping_carrier') ?? WC()->session->get('expedicoDeliveryId');

		    woocommerce_form_field('expedico_shipping_carrier', [
			    'type'          => 'select',
			    'class'         => ['form-row-wide'],
			    'label'         => __('Carrier', 'expedico'),
			    'required'      => true,
			    'options'       => ['' => __('Select a carrier', 'expedico')] + $availableCarriers,
		//    ], WC()->session->get('expedico_shipping_carrier') ?? WC()->session->get('expedicoDeliveryId'));
				], (string) $selectedCarrier);
	    }
		}
    }

    public function add_shipping_method(array $methods): array
    {
        $methods[PickupPointsShippingMethod::EXPEDICO_METHOD_ID] = PickupPointsShippingMethod::class;

		if (!empty($this->settings->isDeliveryCarriersSet())) {
			$methods[DeliveryShippingMethod::EXPEDICO_METHOD_ID] = DeliveryShippingMethod::class;
		}

        return $methods;
    }

    public static function getLocale(): string
    {
        return (string) apply_filters(
            'plugin_locale',
            (is_admin() ? get_user_locale() : get_locale()),
            self::DOMAIN
        );
    }

    public function loadTranslation(): void
    {
        $domain = self::DOMAIN;
        $locale = self::getLocale();

        $moFile = __DIR__ . "/languages/$domain-$locale.mo";
        $result = load_textdomain($domain, $moFile);

        if (FALSE === $result) {
            $moFile = __DIR__ . "/languages/$domain-en_US.mo";
            load_textdomain($domain, $moFile);
        }
    }

	public function cron() {
		$cron = function () {
			if (!wp_next_scheduled('expedico_sync_order_statuses')) {
				wp_schedule_event(time(), 'hourly', 'expedico_sync_order_statuses');
			}
		};
		add_action('init', $cron);

		add_action('expedico_sync_order_statuses', function () {
			$this->expedicoOrderStatus->syncOrderStatuses();
		});
	}
}

add_action('plugins_loaded', fn(): Expedico => new Expedico());
