<?php

declare( strict_types=1 );

namespace Expedico;


use Exception;
use WC_Shipping_Zones;
use WC_Tax;

class ExpedicoSettings {
	const SETTINGS_GROUP = 'expedico_settings_group';
	const SETTINGS_OPTIONS = 'expedico_settings_options';

	private ExpedicoAPI $api;

	private bool $validated = false;

	public function __construct(ExpedicoAPI $api) {
		add_action('admin_init', [$this, 'register_settings']);
		add_action('admin_init', [$this, 'register_expedico_carrier_prices_settings']);
		add_action('admin_menu', [$this, 'add_settings_page']);

		$this->api = $api;
	}

	public function are_credentials_set(): bool {
		return $this->get_username() && $this->get_password() && $this->get_senderId();
	}

	// Getter functions
	public function get_username() {
		$options = get_option(self::SETTINGS_OPTIONS);
		return is_array($options) && isset($options['username']) ? $options['username'] : '';
	}

	public function get_password() {
		$options = get_option(self::SETTINGS_OPTIONS);
		return is_array($options) && isset($options['password']) ? $options['password'] : '';
	}

	public function get_senderId() {
		$options = get_option(self::SETTINGS_OPTIONS);
		return is_array($options) && isset($options['senderId']) ? $options['senderId'] : '';
	}

	public function get_label_format() {
		$options = get_option(self::SETTINGS_OPTIONS);
		// Set default value if unset
		$default = 'a4_4'; // Default value
		return is_array($options) && isset($options['label_format']) ? $options['label_format'] : $default;
	}

	public function get_carriersIds() {
		$options = get_option(self::SETTINGS_OPTIONS);
		return is_array($options) && isset($options['carriersIds']) ? $options['carriersIds'] : [];
	}

	public function get_carriersPrices() {
		return get_option('expedico_carrier_prices', []);
	}

	public function get_carrierPrice($carrierId) {
		$carriersPrices = $this->get_carriersPrices();
		return isset($carriersPrices[$carrierId]) ? $carriersPrices[$carrierId] : 0;
	}

	public function register_settings() {
		register_setting(self::SETTINGS_GROUP, self::SETTINGS_OPTIONS, [$this, 'validate_settings']);
		add_settings_section('expedico_main', __('Main Settings', 'expedico'), [$this, 'section_text'], Expedico::SETTINGS_PAGE_SLUG);
		add_settings_field('expedico_username', __('API Key', 'expedico'), [$this, 'input_username'], Expedico::SETTINGS_PAGE_SLUG, 'expedico_main');
		add_settings_field('expedico_password', __('API Password', 'expedico'), [$this, 'input_password'], Expedico::SETTINGS_PAGE_SLUG, 'expedico_main');
		add_settings_field('expedico_senderId', __('Sender ID', 'expedico'), [$this, 'input_senderId'], Expedico::SETTINGS_PAGE_SLUG, 'expedico_main');
		add_settings_field('expedico_label_format', __('Label Format', 'expedico'), [$this, 'input_label_format'], Expedico::SETTINGS_PAGE_SLUG, 'expedico_main');
		add_settings_field('expedico_carriersIds', __('Carriers', 'expedico'), [$this, 'input_carriersIds'], Expedico::SETTINGS_PAGE_SLUG, 'expedico_main');
	}

	public function register_expedico_carrier_prices_settings() {
		register_setting('expedico_settings_group', 'expedico_carrier_prices');
	}

	public function section_text() {
		echo '';
	}

	public function input_username() {
		$options = get_option(self::SETTINGS_OPTIONS);
		$username = is_array($options) ? $options['username'] : '';
		echo "<input id='expedico_username' name='" . self::SETTINGS_OPTIONS . "[username]' autocomplete='new-password' type='text' value='{$username}' required/>";
	}

	public function input_password() {
		$options = get_option(self::SETTINGS_OPTIONS);
		$password = is_array($options) ? $options['password'] : '';
		echo "<input id='expedico_password' name='" . self::SETTINGS_OPTIONS . "[password]' autocomplete='new-password' type='password' value='{$password}' required/>";
	}

	public function input_senderId() {
		$options = get_option(self::SETTINGS_OPTIONS);
		$senderId = is_array($options) ? $options['senderId'] : '';
		echo "<input id='expedico_senderId' name='" . self::SETTINGS_OPTIONS . "[senderId]' autocomplete='new-password' type='number' min='0' step='1' value='{$senderId}' required/>";
	}

	public function input_label_format() {
		$options = get_option(self::SETTINGS_OPTIONS);
		$label_format = is_array($options) && isset($options['label_format']) ? $options['label_format'] : '';
		echo "<select id='expedico_label_format' name='" . self::SETTINGS_OPTIONS . "[label_format]' required='required'>";
		echo "<option value=''>" . __('--- Select Option ---', 'expedico') . "</option>";
		echo "<option value='a6_1'" . selected($label_format, 'a6_1', false) . ">A6 (1 per page)</option>";
		echo "<option value='a4_2'" . selected($label_format, 'a4_2', false) . ">A4 (2 per page)</option>";
		echo "<option value='a4_4'" . selected($label_format, 'a4_4', false) . ">A4 (4 per page)</option>";
		echo "<option value='a4_8'" . selected($label_format, 'a4_8', false) . ">A4 (8 per page)</option>";
		echo "</select>";
	}

	public function input_carriersIds() {
		if ($this->api->areCredentialsValid()) {
			$carriers = $this->api->getCarriers('label', false);
		} else {
			$carriers = [];
		}

		$options = get_option(self::SETTINGS_OPTIONS);
		$carriersIds = is_array($options) && isset($options['carriersIds']) ? $options['carriersIds'] : [];
		echo "<select id='expedico_carriersIds' name='" . self::SETTINGS_OPTIONS . "[carriersIds][]' multiple='multiple' size='" . count($carriers) . "' style='width: 100%;'>";

		foreach ($carriers as $carrierId => $carrierName) {
			$selected = in_array($carrierId, $carriersIds) ? 'selected' : '';

			echo "<option value='$carrierId' $selected>$carrierName</option>";
		}

		echo "</select>";
	}

	public function validate_settings($input) {
		if (!$input['username'] || !$input['password'] || $this->validated || $input['carriersIds']) {
			$this->validated = true;
			return $input;
		}

		if (!$this->api->areCredentialsValid($input['username'], $input['password'])) {
			add_settings_error('expedico_username', 'credentials_invalid', __('Invalid Expedico credentials.', 'expedico'));

			$input['username'] = '';
		} elseif (empty($this->get_carriersIds())) {
			add_settings_error('expedico_username', 'credentials_valid', __('Expedico credentials are valid. You can now set carriers.', 'expedico'), 'updated');
		}

		$this->validated = true;
		return $input;
	}

	public function add_settings_page() {
		add_options_page(__('Expedico - Settings', 'expedico'), 'Expedico', 'manage_options', Expedico::SETTINGS_PAGE_SLUG, [$this, 'create_settings_page']);
	}

	public function create_settings_page() {
		$selected_carrier_ids = $this->getAllCarriersNames();
		$carrier_prices = $this->get_carriersPrices();

		?>
		<div class="wrap">
			<h2><?php echo __('Expedico - Settings', 'expedico') ?></h2>
			<?php settings_errors(); ?>
			<form method="post" action="options.php" autocomplete="off">
				<?php settings_fields(self::SETTINGS_GROUP); ?>
				<?php do_settings_sections(Expedico::SETTINGS_PAGE_SLUG); ?>

				<?php if ($this->are_credentials_set() && !empty($this->get_carriersIds())): ?>
					<hr>
					<h2><?php echo __('Expedico - Carrier prices settings', 'expedico') ?></h2>
					<?php settings_fields('expedico_settings_group'); ?>
					<table class="form-table">
					<?php foreach ($selected_carrier_ids as $carrier_id => $carrier_name): ?>
						<tr>
							<th>
								<label for="carrier_price_<?php echo esc_attr($carrier_id); ?>">
									Price for Carrier <?php echo esc_html($carrier_name); ?>:
								</label>
							</th>
							<td>
								<input type="number" step="0.01" required
								       id="carrier_price_<?php echo esc_attr($carrier_id); ?>"
								       name="expedico_carrier_prices[<?php echo esc_attr($carrier_id); ?>]"
								       value="<?php echo isset($carrier_prices[$carrier_id]) ? esc_attr($carrier_prices[$carrier_id]) : ''; ?>"
								/>
							</td>
						</tr>
					<?php endforeach; ?>
					</table>
				<?php endif ?>

				<?php submit_button(); ?>
			</form>
		</div>
		<?php
	}

	public function getAllCarriersNames(): array
	{
		$selectedCarriers = $this->get_carriersIds();
		$allCarriers = $this->api->getCarriers('label', false);
		$deliveryCarriers = [];

		foreach ($selectedCarriers as $selectedCarrier) {
			if (!array_key_exists($selectedCarrier, $allCarriers)) {
				continue;
			}

			$deliveryCarriers[$selectedCarrier] = $allCarriers[$selectedCarrier];
		}

		return $deliveryCarriers;
	}

	public function isDeliveryCarriersSet(): bool
	{
		$selectedCarriers = $this->get_carriersIds();
		$allCarriers = $this->api->getCarriers();

		foreach ($selectedCarriers as $selectedCarrier) {
			if (!array_key_exists($selectedCarrier, $allCarriers)) {
				continue;
			}

			if (!str_contains($allCarriers[$selectedCarrier], 'Pickup Points')) {
				return true;
			}
		}

		return false;
	}

	public static function calculatePriceWithTax($base_price, bool $is_taxable): array
	{
		$tax_display_mode = get_option('woocommerce_tax_display_cart');
		$prices_include_tax = wc_prices_include_tax();

		if ($is_taxable) {
			if ($prices_include_tax) {
				$tax_rates = WC_Tax::get_shipping_tax_rates();
				$base_excl_tax = WC_Tax::calc_tax($base_price, $tax_rates, true);
				$base_price = $base_price - array_sum($base_excl_tax);
				$tax_object = $base_excl_tax;
				$tax = array_sum($base_excl_tax);

				if ($tax_display_mode === 'incl') {
					$price_without_tax = $base_price;
					$price = $base_price + $tax;
				} else {
					$price = $price_without_tax = $base_price;
				}
			} else {
				$tax_rates = WC_Tax::get_shipping_tax_rates();
				$tax_object = WC_Tax::calc_tax($base_price, $tax_rates, false);
				$tax = array_sum($tax_object);

				if ($tax_display_mode === 'incl') {
					$price_without_tax = $base_price;
					$price = $base_price + $tax;
				} else {
					$price = $price_without_tax = $base_price;
				}
			}
		} else {
			$price = $price_without_tax = $base_price;
			$tax = 0;
			$tax_object = false;
		}

		return [
			'price_without_tax' => $price_without_tax,
			'price' => $price,
			'tax' => $tax,
			'tax_object' => $tax_object,
		];
	}

	public function getAvailableDeliveryCarriersWithPrices(): array
	{
		$selectedCarriers = $this->get_carriersIds();
		$allCarriers = $this->api->getCarriers();
		$carrierPrices = $this->get_carriersPrices();
		$deliveryCarriers = [];

		$decimals = wc_get_price_decimals();
		$decimal_separator = wc_get_price_decimal_separator();
		$thousand_separator = wc_get_price_thousand_separator();
		$tax_display_mode = get_option('woocommerce_tax_display_cart');
		$prices_include_tax = wc_prices_include_tax();

		$shipping_method = null;
		$all_zones = array_merge(
			WC_Shipping_Zones::get_zones(),
			[WC_Shipping_Zones::get_zone(0)->get_data()]
		);

		foreach ($all_zones as $zone) {
			$methods = $zone['shipping_methods'] ?? [];
			if (empty($methods)) {
				try {
					$zone_obj = WC_Shipping_Zones::get_zone($zone['zone_id'] ?? 0);
					$methods = $zone_obj->get_shipping_methods();
				} catch (Exception $e) {
					continue;
				}
			}

			foreach ($methods as $method) {
				if ($method->id === DeliveryShippingMethod::EXPEDICO_METHOD_ID) {
					$shipping_method = $method;
					break 2;
				}
			}
		}

		$is_taxable = (!$shipping_method || $shipping_method->get_option( 'tax_status' ) === 'taxable') && wc_tax_enabled();

		foreach ($selectedCarriers as $selectedCarrier) {
			if (!array_key_exists($selectedCarrier, $allCarriers)) {
				continue;
			}

			$carrierName = $allCarriers[$selectedCarrier];
			if (str_contains($carrierName, 'Pickup Points')) {
				continue;
			}

			$base_price = isset($carrierPrices[$selectedCarrier])
				? (float)$carrierPrices[$selectedCarrier]
				: 0.0;

			['price' => $price] = $this->calculatePriceWithTax($base_price, $is_taxable);

			$formatted_price = number_format(
				$price,
				$decimals,
				$decimal_separator,
				$thousand_separator
			);

			$price_suffix = '';
			if ($is_taxable && !$prices_include_tax && $tax_display_mode !== 'excl') {
				$price_suffix = __('(incl. VAT)', 'woocommerce');
			} elseif ($is_taxable && $prices_include_tax && $tax_display_mode === 'excl') {
				$price_suffix = __('(ex. VAT)', 'woocommerce');
			}

			$deliveryCarriers[$selectedCarrier] = $carrierName . ' - ' . $formatted_price . ' ' . get_woocommerce_currency_symbol() . ' ' . $price_suffix;
		}

		return $deliveryCarriers;
	}
}