<?php

declare(strict_types=1);

namespace Expedico;


class ExpedicoAPI
{
	private const BASE_URL = 'https://expedico.eu/api/v2';
	private const API_MAX_RETRY = 3;
	private const CARRIERS_CACHE_KEY = 'expedico_api_carriers';
	private int $carriersRetry = 0;
	private int $parcelRetry = 0;

	public function createParcel(\WC_Order $order): bool
	{
		//$streetAndNumber = $this->parseStreetAndNumber($order->get_shipping_address_1() ?: $order->get_billing_address_1());
		$rawStreet = $order->get_shipping_address_1() ?: $order->get_billing_address_1();
		$streetAndNumber = $this->parseStreetAndNumber($rawStreet);

		// Fallbacks
		$street = $streetAndNumber['street'] ?? $rawStreet;
		$streetNumber = $streetAndNumber['number'] ?? '';
		$apartment = $order->get_shipping_address_2() ?: $order->get_billing_address_2();


		$data = [
//			"carrier" => "{carrier_sys}",
			'orderNumber' => $order->get_order_number(),
//			'currency' => '{currency code}',
//			'codAmount' => '{cod amount}',
			'senderId' => (int) $this->get_senderId(),
			'recipient' => [
				'firstname'     => $order->get_shipping_first_name() ?: $order->get_billing_first_name(),
				'lastname'      => $order->get_shipping_last_name() ?: $order->get_billing_last_name(),
				'street' 		=> $street,
				'streetNumber'  => trim($streetNumber . ' ' . $apartment),
				'city'          => $order->get_shipping_city() ?: $order->get_billing_city(),
				'zipCode'       => $order->get_shipping_postcode() ?: $order->get_billing_postcode(),
				'country'       => $order->get_shipping_country() ?: $order->get_billing_country(),
				'email'         => $order->get_billing_email(),
				'phone'         => $order->get_billing_phone(),
			],
//			'description' => sprintf(__('Order from %1$s no. %2$s', 'expedico'), get_bloginfo('name'), $order->get_order_number()),
			'description' => $this->generate_product_description($order),
//			'weightInKg' => '{weight}',
			'carrierPickupDate' => date('Y-m-d')
		];

		// cash on delivery
		if ($order->get_payment_method() === 'cod') {
			$data['codAmount'] = (float) $order->get_total();
			$data['currency'] = $order->get_currency();
		}

		// check delivery type
		if ($carrier = $order->get_meta(Expedico::SHIPPING_CARRIER_SYS_NAME_META)) {
			$data['carrier'] = $carrier;
		} elseif ($pickupPoint = $order->get_meta(Expedico::PICKUP_POINT_ID_META)) {
			$data['pickupPoint'] = (int) $pickupPoint;
		}

		$total_weight_kg = 0.1;

		// Loop through each item in the order
		foreach ($order->get_items() as $item) {
			// Get the product object
			$product = $item->get_product();

			// Check if the product exists and has weight
			if ($product && $product->has_weight()) {
				// Get the product weight
				$product_weight = $product->get_weight();

				// Convert the product weight to kg based on the store's weight unit
				$weight_unit = get_option('woocommerce_weight_unit');
				switch ($weight_unit) {
					case 'g':
						// Convert grams to kg
						$weight_in_kg = floatval($product_weight) / 1000;
						break;
					case 'lbs':
						// Convert pounds to kg
						$weight_in_kg = floatval($product_weight) * 0.453592;
						break;
					case 'oz':
						// Convert ounces to kg
						$weight_in_kg = floatval($product_weight) * 0.0283495;
						break;
					case 'kg':
					default:
						// Default to direct conversion (assuming the weight is in kg)
						$weight_in_kg = floatval($product_weight);
						break;
				}

				// Add the product weight in kg multiplied by the quantity to the total weight
				$total_weight_kg += $weight_in_kg * $item->get_quantity();
			}
		}

		$data['weightInKg'] = $total_weight_kg;

		$data = ['data' => $data];

		$curl = curl_init();
		curl_setopt_array($curl, array(
			CURLOPT_URL => self::BASE_URL . '/parcels',
			CURLOPT_RETURNTRANSFER => true,
			CURLOPT_ENCODING => '',
			CURLOPT_MAXREDIRS => 10,
			CURLOPT_TIMEOUT => 0,
			CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
			CURLOPT_POST => true,
			CURLOPT_HTTPHEADER => array(
				'Authorization: Basic ' . base64_encode($this->get_username() . ':' . $this->get_password()),
				'Content-Type: application/json'
			),
			CURLOPT_POSTFIELDS => json_encode($data, JSON_UNESCAPED_UNICODE)
		));

		$headers = [];
		curl_setopt($curl, CURLOPT_HEADERFUNCTION,
			function($curl, $header) use (&$headers)
			{
				$len = strlen($header);
				$header = explode(':', $header, 2);
				if (count($header) < 2) // ignore invalid headers
					return $len;

				$headers[strtolower(trim($header[0]))] = trim($header[1]);

				return $len;
			}
		);
		$response = curl_exec($curl);
		$httpcode = curl_getinfo($curl, CURLINFO_HTTP_CODE);
		curl_close($curl);

		// validation error
		if ($httpcode === 422) {
			$order->add_order_note(__('The following error occurred when submitting an order to the Expedico API', 'expedico') . ': ' . json_encode($response));
			return false;
		}

		// unknown error
		if ($httpcode !== 201 && $this->parcelRetry <= self::API_MAX_RETRY) {
			$this->parcelRetry++;
			return $this->createParcel($order);
		} elseif ($httpcode !== 201 && $this->parcelRetry >= self::API_MAX_RETRY) {
			$order->add_order_note(__('There was a problem communicating with the Expedico API. The order was not sent.', 'expedico'));
			return false;
		}

		$order->add_meta_data(Expedico::EXPEDICO_PARCEL_ID, $headers['id']);
		$order->save_meta_data();
		$order->add_order_note(__('Tracking link', 'expedico') . ': ' . Expedico::TRACKING_URL . $headers['carrier-tracking-code'], true);

		return true;
	}

	private function generate_product_description(\WC_Order $order): string
{
    $lines = [];

    foreach ($order->get_items() as $item) {
        $product = $item->get_product();
        if (!$product) {
            continue;
        }

        $sku = $product->get_sku() ?: $product->get_name();
        $qty = $item->get_quantity();

        $lines[] = "{$qty} - {$sku}";
    }

    return implode(', ', $lines);
}


	public function getLabelsPdfString(array $parcelIds): ?string
	{
		$data = json_encode([
			'data' => [
				'parcel_ids' => implode(',', $parcelIds),
				'format' => $this->get_label_format()
			]
		], JSON_UNESCAPED_UNICODE);

		$curl = curl_init();
		curl_setopt_array($curl, array(
			CURLOPT_URL => self::BASE_URL . '/label_collections',
			CURLOPT_RETURNTRANSFER => true,
			CURLOPT_ENCODING => '',
			CURLOPT_MAXREDIRS => 10,
			CURLOPT_TIMEOUT => 0,
			CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
			CURLOPT_POST => true,
			CURLOPT_HTTPHEADER => array(
				'Authorization: Basic ' . base64_encode($this->get_username() . ':' . $this->get_password()),
				'Content-Type: application/json'
			),
			CURLOPT_POSTFIELDS => $data
		));

		$response = curl_exec($curl);
		$httpcode = curl_getinfo($curl, CURLINFO_HTTP_CODE);
		curl_close($curl);

		if ($httpcode !== 200) {
			return null;
		}

		$labelsUrl = json_decode($response, true)['data']['attributes']['labels_url'];

		if (!$labelsUrl) {
			return null;
		}

		return wp_remote_retrieve_body(wp_remote_get($labelsUrl));
	}

	/**
	 * @param string $value label/system
	 * @return array
	 */
	public function getCarriers(string $value = 'label', bool $cache = true): array
	{
		if (!$this->areCredentialsValid()) {
			return [];
		}

		if ((($data = get_transient(self::CARRIERS_CACHE_KEY)) === false) || $cache === false) {
			['status' => $status, 'response' => $response] = $this->callApiGet('/carriers?page[number]=1&page[size]=100000');

			if (!$status) {
				error_log('Something gone wrong when calling Expedico.eu API! Error in API call /carriers, response: ' . $response);
				return [];
			}

			$data = json_decode($response, true)['data'];

			if (!empty($data)) {
				set_transient(self::CARRIERS_CACHE_KEY, $data, 60 * 60 * 24);
			}
		}

		$carriers = [];

		foreach ($data as $carrier) {
			if ($value === 'label') {
				$carriers[$carrier['id']] = $carrier['attributes']['labelEn'];
			} elseif ($value === 'system') {
				$carriers[$carrier['id']] = $carrier['attributes']['sys'];
			}
		}

		asort($carriers);
		return $carriers;
	}

	public function getDeliveredParcelIds(array $parcelIds) {
		if (!$this->areCredentialsValid()) {
			return [];
		}

		$parcelIdsImploded = implode(',', $parcelIds);
		['status' => $status, 'response' => $response] = $this->callApiGet('/parcels_tracking?filter[ids]=' . $parcelIdsImploded);

		if (!$status) {
			error_log('Error occurred while calling Expedico API! Error in API call /parcels_tracking?filter[ids]=, response: ' . $response);
			return [];
		}

		$response = json_decode($response, true)['data'];
		$deliveredParcelIds = [];

		foreach ( $parcelIds as $orderId => $parcel_id ) {
			if (!array_key_exists($parcel_id, $response)) {
				continue;
			}

			if ($response[$parcel_id]['parcelStatusId'] == ExpedicoOrderStatus::STATUS_DELIVERED) {
				$deliveredParcelIds[$orderId] = $parcel_id;
			}
		}

		return $deliveredParcelIds;
	}

	public function getPickupPointDetails(int $pickupPointId) {
		if (!$this->areCredentialsValid()) {
			return [];
		}

		['status' => $status, 'response' => $response] = $this->callApiGet('/pickup_points/' . $pickupPointId);

		if (!$status) {
			error_log('Error occurred while calling Expedico API! Error in API call /pickup_points/{id}, response: ' . $response);
			return [];
		}

		$response = json_decode($response, true)['data'];

		return $response['attributes'] ?? null;
	}

	public function areCredentialsValid(string $username = null, string $password = null): bool
	{
		if (!$username) {
			$username = $this->get_username();
		}

		if (!$password) {
			$password = $this->get_password();
		}

		if (empty($username) || empty($password)) {
			return false;
		}

		$curl = curl_init();

		curl_setopt_array($curl, array(
			CURLOPT_URL => self::BASE_URL . '/carriers',
			CURLOPT_RETURNTRANSFER => true,
			CURLOPT_ENCODING => '',
			CURLOPT_MAXREDIRS => 10,
			CURLOPT_TIMEOUT => 0,
			CURLOPT_FOLLOWLOCATION => true,
			CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
			CURLOPT_NOBODY => true,
			CURLOPT_HTTPHEADER => array(
				'Authorization: Basic ' . base64_encode("$username:$password"),
			),
		));

		curl_exec($curl);
		$httpcode = curl_getinfo($curl, CURLINFO_HTTP_CODE);
		curl_close($curl);
		return $httpcode === 200;
	}

	private function callApiGet(string $url)
	{
		$curl = curl_init();

		curl_setopt_array($curl, array(
			CURLOPT_URL => self::BASE_URL . $url,
			CURLOPT_RETURNTRANSFER => true,
			CURLOPT_ENCODING => '',
			CURLOPT_MAXREDIRS => 10,
			CURLOPT_TIMEOUT => 0,
			CURLOPT_FOLLOWLOCATION => true,
			CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
			CURLOPT_HTTPHEADER => array(
				'Authorization: Basic ' . base64_encode($this->get_username() . ':' . $this->get_password()),
			),
		));

		$response = curl_exec($curl);
		$httpcode = curl_getinfo($curl, CURLINFO_HTTP_CODE);
		curl_close($curl);

		if ($httpcode !== 200 && $this->carriersRetry <= self::API_MAX_RETRY) {
			$this->carriersRetry++;
			return $this->callApiGet($url);
		} elseif ($httpcode !== 200 && $this->carriersRetry >= self::API_MAX_RETRY) {
			return ['status' => false, 'response' => $response];
		}

		return ['status' => true, 'response' => $response];
	}

	private function get_username() {
		$options = get_option(ExpedicoSettings::SETTINGS_OPTIONS);
		return is_array($options) && isset($options['username']) ? $options['username'] : '';
	}

	private function get_password() {
		$options = get_option(ExpedicoSettings::SETTINGS_OPTIONS);
		return is_array($options) && isset($options['password']) ? $options['password'] : '';
	}

	private function get_senderId() {
		$options = get_option(ExpedicoSettings::SETTINGS_OPTIONS);
		return is_array($options) && isset($options['senderId']) ? $options['senderId'] : '';
	}

	public function get_label_format() {
		$options = get_option(ExpedicoSettings::SETTINGS_OPTIONS);
		// Set default value if unset
		$default = 'a4_8'; // Default value
		return is_array($options) && isset($options['label_format']) ? $options['label_format'] : $default;
	}

	private function parseStreetAndNumber(string $streetAndNumber): array
	{
		preg_match('~^(.*[^0-9]+) (([1-9][0-9]*)\/)?([1-9][0-9]*[a-zA-Z]?)$~', $streetAndNumber, $matches);

		return [
			'street' => $matches[1],
			'number' => $matches[3] ?: $matches[4]
		];
	}
}