<?php
/**
 * Voice Navigation Module Main Class
 *
 * Core module class that extends EVAS_Module_Base.
 *
 * @package     Everyone_Accessibility_Suite
 * @subpackage  Modules/Voice_Navigation
 * @version     1.0.0
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Class EVAS_Voice_Navigation_Module
 *
 * Main module class for Voice Navigation.
 */
class EVAS_Voice_Navigation_Module extends EVAS_Module_Base {

    /**
     * Constructor
     *
     * @param EVAS_Settings $settings_manager Settings manager instance.
     * @param array         $settings Plugin settings.
     */
    public function __construct( $settings_manager = null, $settings = array() ) {
        $this->module_id      = 'voice_navigation';
        $this->module_name    = __( 'Voice Navigation', 'everyone-accessibility-suite' );
        $this->module_version = defined( 'EVAS_VN_VERSION' ) ? EVAS_VN_VERSION : '1.0.0';
        
        parent::__construct( $settings_manager, $settings );
    }

    /**
     * Get module settings
     *
     * @return array Module settings.
     */
    public function get_module_settings() {
        return [
            'enabled'         => (bool) get_option( 'evas_vn_enabled', true ),
            'language'        => get_option( 'evas_vn_language', 'en-US' ),
            'continuous'      => (bool) get_option( 'evas_vn_continuous', true ),
            'show_indicator'  => (bool) get_option( 'evas_vn_show_indicator', true ),
            'feedback_sound'  => (bool) get_option( 'evas_vn_feedback_sound', true ),
            'visual_feedback' => (bool) get_option( 'evas_vn_visual_feedback', true ),
            'sensitivity'     => (float) get_option( 'evas_vn_sensitivity', 0.8 ),
            'auto_start'      => (bool) get_option( 'evas_vn_auto_start', false ),
        ];
    }

    /**
     * Initialize the module
     */
    public function init() {
        // Register REST routes
        add_action( 'rest_api_init', [ $this, 'register_rest_routes' ] );

        // Enqueue frontend assets
        add_action( 'wp_enqueue_scripts', [ $this, 'enqueue_frontend_assets' ] );

        // Add voice control to panel
        add_action( 'evas_panel_controls', [ $this, 'render_voice_control' ], 30 );
    }

    /**
     * Register module settings
     */
    public function register_settings() {
        $settings = [
            'enabled'             => [ 'type' => 'boolean', 'default' => true ],
            'language'            => [ 'type' => 'string', 'default' => 'en-US' ],
            'continuous'          => [ 'type' => 'boolean', 'default' => true ],
            'show_indicator'      => [ 'type' => 'boolean', 'default' => true ],
            'feedback_sound'      => [ 'type' => 'boolean', 'default' => true ],
            'visual_feedback'     => [ 'type' => 'boolean', 'default' => true ],
            'sensitivity'         => [ 'type' => 'number', 'default' => 0.8 ],
            'auto_start'          => [ 'type' => 'boolean', 'default' => false ],
        ];

        foreach ( $settings as $key => $args ) {
            register_setting( 'evas_vn_settings', 'evas_vn_' . $key, $args );
        }
    }

    /**
     * Register REST API routes
     *
     * @return void
     */
    public function register_rest_routes(): void {
        $controller = new EVAS_VN_REST_Controller();
        $controller->register_routes();
    }

    /**
     * Register admin pages/routes
     *
     * @return array Admin pages configuration.
     */
    public function register_admin_pages(): array {
        return [
            [
                'path'  => '/voice-navigation',
                'title' => __( 'Voice Navigation', 'everyone-accessibility-suite' ),
                'icon'  => 'mic',
            ],
        ];
    }

    /**
     * Enqueue frontend assets
     *
     * @return void
     */
    public function enqueue_frontend_assets(): void {
        if ( ! $this->is_enabled() ) {
            return;
        }

        // Main JavaScript
        wp_enqueue_script(
            'evas-voice-nav',
            EVAS_VN_URL . 'assets/js/voice-navigation.js',
            [ 'jquery' ],
            EVAS_VERSION, // Use main plugin version for cache busting
            true
        );

        // Main styles
        wp_enqueue_style(
            'evas-voice-nav',
            EVAS_VN_URL . 'assets/css/voice-navigation.css',
            [],
            EVAS_VN_VERSION
        );

        // Localize script
        wp_localize_script( 'evas-voice-nav', 'evasVoice', [
            'restUrl'  => rest_url( 'evas-voice/v1/' ),
            'settings' => $this->get_frontend_settings(),
            'commands' => EVAS_Voice_Commands::get_all_commands(),
            'i18n'     => $this->get_i18n_strings(),
        ] );
    }

    /**
     * Render voice control in accessibility panel
     *
     * @return void
     */
    public function render_voice_control(): void {
        if ( ! $this->is_enabled() ) {
            return;
        }
        ?>
        <div id="evas-action-voice-navigation"
             class="evas-action-box evas-toggle-box"
             data-feature="voice_navigation"
             tabindex="0"
             role="button"
             aria-pressed="false"
             aria-label="<?php esc_attr_e( 'Voice Navigation', 'everyone-accessibility-suite' ); ?>">
            <span class="evas-action-icon"></span>
            <span class="evas-action-label"><?php esc_html_e( 'Voice Navigation', 'everyone-accessibility-suite' ); ?></span>
        </div>
        
        <div id="evas-voice-indicator" class="evas-voice-indicator" aria-live="polite" style="display: none; position: fixed; bottom: 20px; left: 20px; z-index: 99999; background: rgba(0,0,0,0.8); color: white; padding: 10px 20px; border-radius: 8px;">
            <span class="evas-voice-pulse" style="display: inline-block; width: 10px; height: 10px; background: red; border-radius: 50%; margin-right: 10px; animation: pulse 1.5s infinite;"></span>
            <span class="evas-voice-text"><?php esc_html_e( 'Listening...', 'everyone-accessibility-suite' ); ?></span>
        </div>
        
        <div id="evas-voice-transcript" class="evas-voice-transcript" aria-live="polite" style="display: none; position: fixed; bottom: 60px; left: 20px; z-index: 99999; background: rgba(255,255,255,0.95); padding: 10px 20px; border-radius: 8px; max-width: 300px;"></div>
        <?php
    }

    /**
     * Get settings for frontend use
     *
     * @return array
     */
    private function get_frontend_settings(): array {
        return [
            'enabled'         => (bool) get_option( 'evas_vn_enabled', true ),
            'language'        => get_option( 'evas_vn_language', 'en-US' ),
            'continuous'      => (bool) get_option( 'evas_vn_continuous', true ),
            'showIndicator'   => (bool) get_option( 'evas_vn_show_indicator', true ),
            'feedbackSound'   => (bool) get_option( 'evas_vn_feedback_sound', true ),
            'visualFeedback'  => (bool) get_option( 'evas_vn_visual_feedback', true ),
            'sensitivity'     => (float) get_option( 'evas_vn_sensitivity', 0.8 ),
            'autoStart'       => (bool) get_option( 'evas_vn_auto_start', false ),
        ];
    }

    /**
     * Get internationalization strings
     *
     * @return array
     */
    private function get_i18n_strings(): array {
        return [
            'listening'          => __( 'Listening...', 'everyone-accessibility-suite' ),
            'startListening'     => __( 'Start listening', 'everyone-accessibility-suite' ),
            'stopListening'      => __( 'Stop listening', 'everyone-accessibility-suite' ),
            'commandRecognized'  => __( 'Command recognized', 'everyone-accessibility-suite' ),
            'commandNotFound'    => __( 'Command not found', 'everyone-accessibility-suite' ),
            'noMicrophone'       => __( 'No microphone found', 'everyone-accessibility-suite' ),
            'microphoneBlocked'  => __( 'Microphone access blocked', 'everyone-accessibility-suite' ),
            'notSupported'       => __( 'Voice recognition is not supported in this browser', 'everyone-accessibility-suite' ),
            'error'              => __( 'Voice recognition error', 'everyone-accessibility-suite' ),
            'helpTitle'          => __( 'Available Voice Commands', 'everyone-accessibility-suite' ),
        ];
    }

    /**
     * Get module metadata for REST API
     *
     * @return array
     */
    public function get_metadata(): array {
        return [
            'id'           => $this->get_id(),
            'name'         => $this->get_name(),
            'description'  => $this->get_description(),
            'version'      => $this->get_version(),
            'enabled'      => $this->is_enabled(),
            'commandCount' => count( EVAS_Voice_Commands::get_all_commands() ),
        ];
    }
}

