<?php
/**
 * Voice Commands Configuration
 *
 * Contains all voice commands for navigation.
 *
 * @package     Everyone_Accessibility_Suite
 * @subpackage  Modules/Voice_Navigation
 * @version     1.0.0
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Class EVAS_Voice_Commands
 *
 * Provides voice command configurations.
 */
class EVAS_Voice_Commands {

    /**
     * Get default voice commands
     *
     * @return array
     */
    public static function get_default_commands(): array {
        return [
            // Navigation commands
            'scroll_up' => [
                'id'          => 'scroll_up',
                'category'    => 'navigation',
                'phrases'     => [ 'scroll up', 'go up', 'move up', 'page up' ],
                'action'      => 'scrollUp',
                'description' => __( 'Scroll page up', 'everyone-accessibility-suite' ),
            ],
            'scroll_down' => [
                'id'          => 'scroll_down',
                'category'    => 'navigation',
                'phrases'     => [ 'scroll down', 'go down', 'move down', 'page down' ],
                'action'      => 'scrollDown',
                'description' => __( 'Scroll page down', 'everyone-accessibility-suite' ),
            ],
            'go_top' => [
                'id'          => 'go_top',
                'category'    => 'navigation',
                'phrases'     => [ 'go to top', 'scroll to top', 'top of page', 'go top' ],
                'action'      => 'scrollToTop',
                'description' => __( 'Scroll to top of page', 'everyone-accessibility-suite' ),
            ],
            'go_bottom' => [
                'id'          => 'go_bottom',
                'category'    => 'navigation',
                'phrases'     => [ 'go to bottom', 'scroll to bottom', 'bottom of page', 'go bottom' ],
                'action'      => 'scrollToBottom',
                'description' => __( 'Scroll to bottom of page', 'everyone-accessibility-suite' ),
            ],
            'go_back' => [
                'id'          => 'go_back',
                'category'    => 'navigation',
                'phrases'     => [ 'go back', 'back', 'previous page', 'go to previous' ],
                'action'      => 'goBack',
                'description' => __( 'Go to previous page', 'everyone-accessibility-suite' ),
            ],
            'go_forward' => [
                'id'          => 'go_forward',
                'category'    => 'navigation',
                'phrases'     => [ 'go forward', 'forward', 'next page' ],
                'action'      => 'goForward',
                'description' => __( 'Go forward in history', 'everyone-accessibility-suite' ),
            ],
            'go_home' => [
                'id'          => 'go_home',
                'category'    => 'navigation',
                'phrases'     => [ 'go home', 'go to home', 'home page', 'main page' ],
                'action'      => 'goHome',
                'description' => __( 'Go to homepage', 'everyone-accessibility-suite' ),
            ],

            // Click commands
            'click' => [
                'id'          => 'click',
                'category'    => 'interaction',
                'phrases'     => [ 'click *', 'press *', 'tap *' ],
                'action'      => 'clickElement',
                'hasParam'    => true,
                'description' => __( 'Click element by text', 'everyone-accessibility-suite' ),
            ],
            'click_button' => [
                'id'          => 'click_button',
                'category'    => 'interaction',
                'phrases'     => [ 'click button *', 'press button *' ],
                'action'      => 'clickButton',
                'hasParam'    => true,
                'description' => __( 'Click button by text', 'everyone-accessibility-suite' ),
            ],
            'click_link' => [
                'id'          => 'click_link',
                'category'    => 'interaction',
                'phrases'     => [ 'click link *', 'open link *', 'follow link *' ],
                'action'      => 'clickLink',
                'hasParam'    => true,
                'description' => __( 'Click link by text', 'everyone-accessibility-suite' ),
            ],

            // Menu commands
            'open_menu' => [
                'id'          => 'open_menu',
                'category'    => 'menu',
                'phrases'     => [ 'open menu', 'show menu', 'menu' ],
                'action'      => 'openMenu',
                'description' => __( 'Open navigation menu', 'everyone-accessibility-suite' ),
            ],
            'close_menu' => [
                'id'          => 'close_menu',
                'category'    => 'menu',
                'phrases'     => [ 'close menu', 'hide menu' ],
                'action'      => 'closeMenu',
                'description' => __( 'Close navigation menu', 'everyone-accessibility-suite' ),
            ],

            // Accessibility commands
            'open_accessibility' => [
                'id'          => 'open_accessibility',
                'category'    => 'accessibility',
                'phrases'     => [ 'open accessibility', 'accessibility menu', 'accessibility panel', 'show accessibility' ],
                'action'      => 'openAccessibilityPanel',
                'description' => __( 'Open accessibility panel', 'everyone-accessibility-suite' ),
            ],
            'close_accessibility' => [
                'id'          => 'close_accessibility',
                'category'    => 'accessibility',
                'phrases'     => [ 'close accessibility', 'hide accessibility' ],
                'action'      => 'closeAccessibilityPanel',
                'description' => __( 'Close accessibility panel', 'everyone-accessibility-suite' ),
            ],
            'read_page' => [
                'id'          => 'read_page',
                'category'    => 'accessibility',
                'phrases'     => [ 'read page', 'read aloud', 'read this page', 'start reading' ],
                'action'      => 'readPage',
                'description' => __( 'Read page content aloud', 'everyone-accessibility-suite' ),
            ],
            'stop_reading' => [
                'id'          => 'stop_reading',
                'category'    => 'accessibility',
                'phrases'     => [ 'stop reading', 'stop', 'pause', 'silence' ],
                'action'      => 'stopReading',
                'description' => __( 'Stop reading', 'everyone-accessibility-suite' ),
            ],

            // Zoom commands
            'zoom_in' => [
                'id'          => 'zoom_in',
                'category'    => 'zoom',
                'phrases'     => [ 'zoom in', 'make bigger', 'increase size', 'larger' ],
                'action'      => 'zoomIn',
                'description' => __( 'Increase text size', 'everyone-accessibility-suite' ),
            ],
            'zoom_out' => [
                'id'          => 'zoom_out',
                'category'    => 'zoom',
                'phrases'     => [ 'zoom out', 'make smaller', 'decrease size', 'smaller' ],
                'action'      => 'zoomOut',
                'description' => __( 'Decrease text size', 'everyone-accessibility-suite' ),
            ],
            'reset_zoom' => [
                'id'          => 'reset_zoom',
                'category'    => 'zoom',
                'phrases'     => [ 'reset zoom', 'normal size', 'default size' ],
                'action'      => 'resetZoom',
                'description' => __( 'Reset text size to default', 'everyone-accessibility-suite' ),
            ],

            // Search commands
            'search' => [
                'id'          => 'search',
                'category'    => 'search',
                'phrases'     => [ 'search *', 'search for *', 'find *', 'look for *' ],
                'action'      => 'search',
                'hasParam'    => true,
                'description' => __( 'Search the website', 'everyone-accessibility-suite' ),
            ],

            // Help
            'help' => [
                'id'          => 'help',
                'category'    => 'help',
                'phrases'     => [ 'help', 'what can I say', 'commands', 'voice help' ],
                'action'      => 'showHelp',
                'description' => __( 'Show available commands', 'everyone-accessibility-suite' ),
            ],
        ];
    }

    /**
     * Get command categories
     *
     * @return array
     */
    public static function get_categories(): array {
        return [
            'navigation'    => __( 'Navigation', 'everyone-accessibility-suite' ),
            'interaction'   => __( 'Click & Interact', 'everyone-accessibility-suite' ),
            'menu'          => __( 'Menu Control', 'everyone-accessibility-suite' ),
            'accessibility' => __( 'Accessibility', 'everyone-accessibility-suite' ),
            'zoom'          => __( 'Zoom & Size', 'everyone-accessibility-suite' ),
            'search'        => __( 'Search', 'everyone-accessibility-suite' ),
            'help'          => __( 'Help', 'everyone-accessibility-suite' ),
        ];
    }

    /**
     * Get commands by category
     *
     * @param string $category Category ID.
     * @return array
     */
    public static function get_by_category( string $category ): array {
        $commands = self::get_default_commands();
        
        return array_filter( $commands, function( $cmd ) use ( $category ) {
            return ( $cmd['category'] ?? '' ) === $category;
        });
    }

    /**
     * Get supported languages for speech recognition
     *
     * @return array
     */
    public static function get_recognition_languages(): array {
        return [
            'en-US' => __( 'English (US)', 'everyone-accessibility-suite' ),
            'en-GB' => __( 'English (UK)', 'everyone-accessibility-suite' ),
            'es-ES' => __( 'Spanish', 'everyone-accessibility-suite' ),
            'fr-FR' => __( 'French', 'everyone-accessibility-suite' ),
            'de-DE' => __( 'German', 'everyone-accessibility-suite' ),
            'it-IT' => __( 'Italian', 'everyone-accessibility-suite' ),
            'pt-BR' => __( 'Portuguese (Brazil)', 'everyone-accessibility-suite' ),
            'ru-RU' => __( 'Russian', 'everyone-accessibility-suite' ),
            'ja-JP' => __( 'Japanese', 'everyone-accessibility-suite' ),
            'ko-KR' => __( 'Korean', 'everyone-accessibility-suite' ),
            'zh-CN' => __( 'Chinese (Mandarin)', 'everyone-accessibility-suite' ),
        ];
    }

    /**
     * Get custom user commands
     *
     * @return array
     */
    public static function get_custom_commands(): array {
        return get_option( 'evas_vn_custom_commands', [] );
    }

    /**
     * Save custom command
     *
     * @param array $command Command data.
     * @return bool
     */
    public static function save_custom_command( array $command ): bool {
        $commands = self::get_custom_commands();
        $commands[ $command['id'] ] = $command;
        
        return update_option( 'evas_vn_custom_commands', $commands );
    }

    /**
     * Delete custom command
     *
     * @param string $command_id Command ID.
     * @return bool
     */
    public static function delete_custom_command( string $command_id ): bool {
        $commands = self::get_custom_commands();
        
        if ( isset( $commands[ $command_id ] ) ) {
            unset( $commands[ $command_id ] );
            return update_option( 'evas_vn_custom_commands', $commands );
        }

        return false;
    }

    /**
     * Get all commands (default + custom)
     *
     * @return array
     */
    public static function get_all_commands(): array {
        return array_merge(
            self::get_default_commands(),
            self::get_custom_commands()
        );
    }
}

