<?php
/**
 * Voice Navigation REST Controller
 *
 * REST API endpoints for Voice Navigation module.
 *
 * @package     Everyone_Accessibility_Suite
 * @subpackage  Modules/Voice_Navigation
 * @version     1.0.0
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Class EVAS_VN_REST_Controller
 *
 * REST API controller for Voice Navigation module.
 */
class EVAS_VN_REST_Controller extends WP_REST_Controller {

    /**
     * REST API namespace
     *
     * @var string
     */
    protected $namespace = 'evas-voice/v1';

    /**
     * Register REST routes
     *
     * @return void
     */
    public function register_routes(): void {
        // GET /wp-json/evas-voice/v1/commands
        register_rest_route( $this->namespace, '/commands', [
            'methods'             => WP_REST_Server::READABLE,
            'callback'            => [ $this, 'get_commands' ],
            'permission_callback' => '__return_true',
        ] );

        // GET /wp-json/evas-voice/v1/languages
        register_rest_route( $this->namespace, '/languages', [
            'methods'             => WP_REST_Server::READABLE,
            'callback'            => [ $this, 'get_languages' ],
            'permission_callback' => '__return_true',
        ] );

        // GET/POST /wp-json/evas-voice/v1/settings
        register_rest_route( $this->namespace, '/settings', [
            [
                'methods'             => WP_REST_Server::READABLE,
                'callback'            => [ $this, 'get_settings' ],
                'permission_callback' => [ $this, 'get_permissions_check' ],
            ],
            [
                'methods'             => WP_REST_Server::EDITABLE,
                'callback'            => [ $this, 'update_settings' ],
                'permission_callback' => [ $this, 'update_permissions_check' ],
            ],
        ] );

        // POST /wp-json/evas-voice/v1/commands/custom
        register_rest_route( $this->namespace, '/commands/custom', [
            [
                'methods'             => WP_REST_Server::READABLE,
                'callback'            => [ $this, 'get_custom_commands' ],
                'permission_callback' => [ $this, 'get_permissions_check' ],
            ],
            [
                'methods'             => WP_REST_Server::CREATABLE,
                'callback'            => [ $this, 'save_custom_command' ],
                'permission_callback' => [ $this, 'update_permissions_check' ],
            ],
        ] );

        // DELETE /wp-json/evas-voice/v1/commands/custom/(?P<id>[\w-]+)
        register_rest_route( $this->namespace, '/commands/custom/(?P<id>[\w-]+)', [
            'methods'             => WP_REST_Server::DELETABLE,
            'callback'            => [ $this, 'delete_custom_command' ],
            'permission_callback' => [ $this, 'update_permissions_check' ],
            'args'                => [
                'id' => [
                    'required'          => true,
                    'type'              => 'string',
                    'sanitize_callback' => 'sanitize_key',
                ],
            ],
        ] );
    }

    /**
     * Get all commands
     *
     * @param WP_REST_Request $request Request object.
     * @return WP_REST_Response
     */
    public function get_commands( WP_REST_Request $request ): WP_REST_Response {
        $commands   = EVAS_Voice_Commands::get_all_commands();
        $categories = EVAS_Voice_Commands::get_categories();

        return rest_ensure_response( [
            'success'    => true,
            'commands'   => $commands,
            'categories' => $categories,
            'count'      => count( $commands ),
        ] );
    }

    /**
     * Get supported languages
     *
     * @param WP_REST_Request $request Request object.
     * @return WP_REST_Response
     */
    public function get_languages( WP_REST_Request $request ): WP_REST_Response {
        return rest_ensure_response( [
            'success'   => true,
            'languages' => EVAS_Voice_Commands::get_recognition_languages(),
        ] );
    }

    /**
     * Get settings
     *
     * @param WP_REST_Request $request Request object.
     * @return WP_REST_Response
     */
    public function get_settings( WP_REST_Request $request ): WP_REST_Response {
        return rest_ensure_response( [
            'success'  => true,
            'settings' => [
                'enabled'             => (bool) get_option( 'evas_vn_enabled', true ),
                'language'            => get_option( 'evas_vn_language', 'en-US' ),
                'continuous'          => (bool) get_option( 'evas_vn_continuous', true ),
                'show_indicator'      => (bool) get_option( 'evas_vn_show_indicator', true ),
                'feedback_sound'      => (bool) get_option( 'evas_vn_feedback_sound', true ),
                'visual_feedback'     => (bool) get_option( 'evas_vn_visual_feedback', true ),
                'sensitivity'         => (float) get_option( 'evas_vn_sensitivity', 0.8 ),
                'auto_start'          => (bool) get_option( 'evas_vn_auto_start', false ),
            ],
        ] );
    }

    /**
     * Update settings
     *
     * @param WP_REST_Request $request Request object.
     * @return WP_REST_Response
     */
    public function update_settings( WP_REST_Request $request ): WP_REST_Response {
        $params = $request->get_json_params();

        $allowed_settings = [
            'enabled'             => 'rest_sanitize_boolean',
            'language'            => 'sanitize_text_field',
            'continuous'          => 'rest_sanitize_boolean',
            'show_indicator'      => 'rest_sanitize_boolean',
            'feedback_sound'      => 'rest_sanitize_boolean',
            'visual_feedback'     => 'rest_sanitize_boolean',
            'sensitivity'         => 'floatval',
            'auto_start'          => 'rest_sanitize_boolean',
        ];

        foreach ( $params as $key => $value ) {
            if ( isset( $allowed_settings[ $key ] ) ) {
                $sanitize = $allowed_settings[ $key ];
                $value    = $sanitize( $value );
                update_option( 'evas_vn_' . $key, $value );
            }
        }

        return rest_ensure_response( [
            'success' => true,
            'message' => __( 'Settings saved successfully', 'everyone-accessibility-suite' ),
        ] );
    }

    /**
     * Get custom commands
     *
     * @param WP_REST_Request $request Request object.
     * @return WP_REST_Response
     */
    public function get_custom_commands( WP_REST_Request $request ): WP_REST_Response {
        return rest_ensure_response( [
            'success'  => true,
            'commands' => EVAS_Voice_Commands::get_custom_commands(),
        ] );
    }

    /**
     * Save custom command
     *
     * @param WP_REST_Request $request Request object.
     * @return WP_REST_Response
     */
    public function save_custom_command( WP_REST_Request $request ): WP_REST_Response {
        $params = $request->get_json_params();

        $command = [
            'id'          => sanitize_key( $params['id'] ?? uniqid( 'custom_' ) ),
            'category'    => 'custom',
            'phrases'     => array_map( 'sanitize_text_field', (array) ( $params['phrases'] ?? [] ) ),
            'action'      => sanitize_text_field( $params['action'] ?? 'custom' ),
            'actionData'  => $params['actionData'] ?? null,
            'description' => sanitize_text_field( $params['description'] ?? '' ),
            'isCustom'    => true,
        ];

        $result = EVAS_Voice_Commands::save_custom_command( $command );

        if ( $result ) {
            return rest_ensure_response( [
                'success' => true,
                'message' => __( 'Custom command saved', 'everyone-accessibility-suite' ),
                'command' => $command,
            ] );
        }

        return new WP_Error(
            'save_failed',
            __( 'Failed to save command', 'everyone-accessibility-suite' ),
            [ 'status' => 500 ]
        );
    }

    /**
     * Delete custom command
     *
     * @param WP_REST_Request $request Request object.
     * @return WP_REST_Response|WP_Error
     */
    public function delete_custom_command( WP_REST_Request $request ) {
        $id     = $request->get_param( 'id' );
        $result = EVAS_Voice_Commands::delete_custom_command( $id );

        if ( $result ) {
            return rest_ensure_response( [
                'success' => true,
                'message' => __( 'Command deleted', 'everyone-accessibility-suite' ),
            ] );
        }

        return new WP_Error(
            'delete_failed',
            __( 'Command not found or could not be deleted', 'everyone-accessibility-suite' ),
            [ 'status' => 404 ]
        );
    }

    /**
     * Check read permissions
     *
     * @param WP_REST_Request $request Request object.
     * @return bool
     */
    public function get_permissions_check( WP_REST_Request $request ): bool {
        return current_user_can( 'manage_options' );
    }

    /**
     * Check update permissions
     *
     * @param WP_REST_Request $request Request object.
     * @return bool
     */
    public function update_permissions_check( WP_REST_Request $request ): bool {
        return current_user_can( 'manage_options' );
    }
}

