/**
 * EVAS Voice Navigation
 *
 * JavaScript for voice navigation using Web Speech API.
 *
 * @package Everyone_Accessibility_Suite
 * @version 1.0.0
 */

(function($) {
    'use strict';

    /**
     * EVAS Voice Navigation Controller
     */
    const EVAS_Voice = {
        settings: window.evasVoice?.settings || {},
        commands: window.evasVoice?.commands || {},
        i18n: window.evasVoice?.i18n || {},

        recognition: null,
        isListening: false,
        isSupported: false,

        /**
         * Initialize Voice Navigation
         */
        init() {
            this.checkSupport();
            
            if (!this.isSupported) {
                console.log('EVAS Voice: Speech recognition not supported');
                this.disableUI();
                return;
            }

            this.setupRecognition();
            this.bindEvents();

            // Update button state when panel opens
            $(document).on('evas-panel-opened', () => {
                this.updateUI();
            });

            // Auto-start if enabled
            if (this.settings.autoStart) {
                this.start();
            }
        },

        /**
         * Check browser support
         */
        checkSupport() {
            const SpeechRecognition = window.SpeechRecognition || window.webkitSpeechRecognition;
            this.isSupported = !!SpeechRecognition;
        },

        /**
         * Setup speech recognition
         */
        setupRecognition() {
            const SpeechRecognition = window.SpeechRecognition || window.webkitSpeechRecognition;
            
            this.recognition = new SpeechRecognition();
            this.recognition.continuous = this.settings.continuous;
            this.recognition.interimResults = true;
            this.recognition.lang = this.settings.language || 'en-US';
            this.recognition.maxAlternatives = 3;

            // Event handlers
            this.recognition.onstart = () => this.onStart();
            this.recognition.onend = () => this.onEnd();
            this.recognition.onresult = (e) => this.onResult(e);
            this.recognition.onerror = (e) => this.onError(e);
        },

        /**
         * Bind event handlers
         */
        bindEvents() {
            // Toggle button
            $(document).on('click', '#evas-action-voice-navigation', () => {
                this.toggle();
            });

            // Help button
            $(document).on('click', '#evas-voice-help', () => {
                this.showHelp();
            });

            // Close help modal
            $(document).on('click', '#evas-voice-help-close, .evas-voice-help-overlay', () => {
                this.hideHelp();
            });

            // Keyboard shortcut (Alt + V)
            $(document).on('keydown', (e) => {
                if (e.altKey && e.key === 'v') {
                    e.preventDefault();
                    this.toggle();
                }
            });
        },

        /**
         * Start listening
         */
        start() {
            if (!this.isSupported || this.isListening) return;

            try {
                this.recognition.start();
            } catch (e) {
                console.error('EVAS Voice: Error starting recognition', e);
            }
        },

        /**
         * Stop listening
         */
        stop() {
            if (!this.isListening) return;

            try {
                this.recognition.stop();
            } catch (e) {
                console.error('EVAS Voice: Error stopping recognition', e);
            }
        },

        /**
         * Toggle listening
         */
        toggle() {
            if (this.isListening) {
                this.stop();
            } else {
                this.start();
            }
        },

        /**
         * Handle recognition start
         */
        onStart() {
            this.isListening = true;
            this.updateUI();
            this.showIndicator();
            
            if (this.settings.feedbackSound) {
                this.playSound('start');
            }

            $(document).trigger('evas:voice:start');
        },

        /**
         * Handle recognition end
         */
        onEnd() {
            this.isListening = false;
            this.updateUI();
            this.hideIndicator();

            // Restart if continuous mode
            if (this.settings.continuous && this.isListening === false) {
                // Check if we should restart
                const shouldRestart = $('#evas-action-voice-navigation').attr('aria-pressed') === 'true';
                if (shouldRestart) {
                    setTimeout(() => this.start(), 100);
                }
            }

            $(document).trigger('evas:voice:end');
        },

        /**
         * Handle recognition result
         * @param {SpeechRecognitionEvent} event
         */
        onResult(event) {
            const results = event.results;
            const lastResult = results[results.length - 1];

            if (lastResult.isFinal) {
                const transcript = lastResult[0].transcript.trim().toLowerCase();
                this.showTranscript(transcript);
                this.processCommand(transcript);
            } else {
                // Show interim result
                const interimTranscript = lastResult[0].transcript;
                this.showTranscript(interimTranscript, true);
            }
        },

        /**
         * Handle recognition error
         * @param {SpeechRecognitionError} event
         */
        onError(event) {
            console.error('EVAS Voice: Recognition error', event.error);
            
            this.isListening = false;
            this.updateUI();
            this.hideIndicator();

            let message = this.i18n.error;

            switch (event.error) {
                case 'no-speech':
                    // Silent, just restart
                    if (this.settings.continuous) {
                        setTimeout(() => this.start(), 500);
                    }
                    return;
                case 'audio-capture':
                    message = this.i18n.noMicrophone;
                    break;
                case 'not-allowed':
                    message = this.i18n.microphoneBlocked;
                    break;
            }

            this.showFeedback(message, 'error');
            $(document).trigger('evas:voice:error', { error: event.error });
        },

        /**
         * Process recognized command
         * @param {string} transcript Recognized text
         */
        processCommand(transcript) {
            let matched = false;
            let matchedCommand = null;
            let param = null;

            for (const [id, command] of Object.entries(this.commands)) {
                for (const phrase of command.phrases) {
                    // Handle wildcard phrases
                    if (phrase.includes('*')) {
                        const regex = new RegExp('^' + phrase.replace('*', '(.+)') + '$', 'i');
                        const match = transcript.match(regex);
                        
                        if (match) {
                            matched = true;
                            matchedCommand = command;
                            param = match[1];
                            break;
                        }
                    } else if (transcript.includes(phrase.toLowerCase())) {
                        matched = true;
                        matchedCommand = command;
                        break;
                    }
                }
                
                if (matched) break;
            }

            if (matched && matchedCommand) {
                this.executeCommand(matchedCommand, param);
                
                if (this.settings.visualFeedback) {
                    this.showFeedback(this.i18n.commandRecognized + ': ' + matchedCommand.description, 'success');
                }
                
                if (this.settings.feedbackSound) {
                    this.playSound('success');
                }
            } else {
                if (this.settings.visualFeedback) {
                    this.showFeedback(this.i18n.commandNotFound + ': ' + transcript, 'warning');
                }
            }
        },

        /**
         * Execute command action
         * @param {Object} command Command configuration
         * @param {string|null} param Command parameter
         */
        executeCommand(command, param = null) {
            const action = command.action;
            
            switch (action) {
                // Navigation
                case 'scrollUp':
                    window.scrollBy({ top: -300, behavior: 'smooth' });
                    break;
                case 'scrollDown':
                    window.scrollBy({ top: 300, behavior: 'smooth' });
                    break;
                case 'scrollToTop':
                    window.scrollTo({ top: 0, behavior: 'smooth' });
                    break;
                case 'scrollToBottom':
                    window.scrollTo({ top: document.body.scrollHeight, behavior: 'smooth' });
                    break;
                case 'goBack':
                    history.back();
                    break;
                case 'goForward':
                    history.forward();
                    break;
                case 'goHome':
                    window.location.href = '/';
                    break;

                // Click actions
                case 'clickElement':
                    this.clickByText(param);
                    break;
                case 'clickButton':
                    this.clickByText(param, 'button, [role="button"], input[type="submit"], input[type="button"]');
                    break;
                case 'clickLink':
                    this.clickByText(param, 'a');
                    break;

                // Menu
                case 'openMenu':
                    this.toggleMenu(true);
                    break;
                case 'closeMenu':
                    this.toggleMenu(false);
                    break;

                // Accessibility
                case 'openAccessibilityPanel':
                    $('#evas-panel-toggle, .evas-panel__trigger').click();
                    break;
                case 'closeAccessibilityPanel':
                    $('.evas-panel__close').click();
                    break;
                case 'readPage':
                    if (window.EVAS_TTS) {
                        window.EVAS_TTS.readPage();
                    }
                    break;
                case 'stopReading':
                    if (window.EVAS_TTS) {
                        window.EVAS_TTS.stop();
                    }
                    break;

                // Zoom
                case 'zoomIn':
                    this.adjustZoom(10);
                    break;
                case 'zoomOut':
                    this.adjustZoom(-10);
                    break;
                case 'resetZoom':
                    $('body').css('font-size', '');
                    break;

                // Search
                case 'search':
                    this.performSearch(param);
                    break;

                // Help
                case 'showHelp':
                    this.showHelp();
                    break;

                // Custom action
                case 'custom':
                    if (command.actionData) {
                        this.executeCustomAction(command.actionData, param);
                    }
                    break;

                default:
                    console.log('EVAS Voice: Unknown action', action);
            }

            $(document).trigger('evas:voice:command', { command, param });
        },

        /**
         * Click element by text content
         * @param {string} text Text to search
         * @param {string} selector Optional selector
         */
        clickByText(text, selector = '*') {
            if (!text) return;

            const textLower = text.toLowerCase();
            const elements = $(selector).filter(function() {
                return $(this).text().toLowerCase().includes(textLower);
            });

            if (elements.length > 0) {
                elements.first()[0].click();
            }
        },

        /**
         * Toggle navigation menu
         * @param {boolean} open Open or close
         */
        toggleMenu(open) {
            const menuSelectors = [
                '.menu-toggle',
                '.nav-toggle',
                '.hamburger',
                '[aria-label*="menu"]',
                '[aria-label*="Menu"]'
            ];

            const $button = $(menuSelectors.join(', ')).first();
            
            if ($button.length) {
                $button.click();
            }
        },

        /**
         * Adjust zoom level
         * @param {number} delta Change amount in percent
         */
        adjustZoom(delta) {
            const $body = $('body');
            let currentSize = parseInt($body.css('font-size')) || 16;
            const newSize = currentSize + (currentSize * delta / 100);
            
            $body.css('font-size', newSize + 'px');
        },

        /**
         * Perform search
         * @param {string} query Search query
         */
        performSearch(query) {
            if (!query) return;

            // Try to find search input
            const $searchInput = $('input[type="search"], input[name="s"], input[name="q"], .search-field').first();
            
            if ($searchInput.length) {
                $searchInput.val(query).closest('form').submit();
            } else {
                // Fallback to URL search
                window.location.href = '/?s=' + encodeURIComponent(query);
            }
        },

        /**
         * Execute custom action
         * @param {Object} actionData Action data
         * @param {string} param Parameter
         */
        executeCustomAction(actionData, param) {
            if (actionData.type === 'url') {
                let url = actionData.url;
                if (param) {
                    url = url.replace('{param}', encodeURIComponent(param));
                }
                window.location.href = url;
            } else if (actionData.type === 'js' && actionData.code) {
                try {
                    eval(actionData.code);
                } catch (e) {
                    console.error('EVAS Voice: Custom action error', e);
                }
            }
        },

        // =====================================
        // UI Methods
        // =====================================

        /**
         * Update UI based on listening state
         */
        updateUI() {
            const $button = $('#evas-action-voice-navigation');
            
            if (this.isListening) {
                $button
                    .addClass('evas-active evas-listening')
                    .attr('aria-pressed', 'true');
            } else {
                $button
                    .removeClass('evas-active evas-listening')
                    .attr('aria-pressed', 'false');
            }
        },

        /**
         * Disable UI when not supported
         */
        disableUI() {
            $('#evas-action-voice-navigation')
                .prop('disabled', true)
                .addClass('evas-disabled')
                .attr('title', this.i18n.notSupported);
        },

        /**
         * Show listening indicator
         */
        showIndicator() {
            if (this.settings.showIndicator) {
                $('#evas-voice-indicator').fadeIn(200);
            }
        },

        /**
         * Hide listening indicator
         */
        hideIndicator() {
            $('#evas-voice-indicator').fadeOut(200);
        },

        /**
         * Show transcript
         * @param {string} text Transcript text
         * @param {boolean} interim Is interim result
         */
        showTranscript(text, interim = false) {
            const $transcript = $('#evas-voice-transcript');
            
            $transcript
                .text(text)
                .toggleClass('evas-interim', interim)
                .show();

            // Auto-hide after delay
            if (!interim) {
                setTimeout(() => $transcript.fadeOut(), 3000);
            }
        },

        /**
         * Show feedback message
         * @param {string} message Message text
         * @param {string} type Message type
         */
        showFeedback(message, type = 'info') {
            // Remove existing feedback
            $('.evas-voice-feedback').remove();

            const $feedback = $('<div>')
                .addClass('evas-voice-feedback evas-voice-feedback--' + type)
                .text(message)
                .appendTo('body');

            setTimeout(() => $feedback.addClass('evas-visible'), 10);
            setTimeout(() => {
                $feedback.removeClass('evas-visible');
                setTimeout(() => $feedback.remove(), 300);
            }, 3000);
        },

        /**
         * Show help modal
         */
        showHelp() {
            if ($('#evas-voice-help-modal').length) {
                $('#evas-voice-help-modal').fadeIn(200);
                return;
            }

            const categories = {};
            
            for (const [id, cmd] of Object.entries(this.commands)) {
                const cat = cmd.category || 'other';
                if (!categories[cat]) {
                    categories[cat] = [];
                }
                categories[cat].push(cmd);
            }

            let html = `
                <div id="evas-voice-help-modal" class="evas-voice-help-modal">
                    <div class="evas-voice-help-overlay"></div>
                    <div class="evas-voice-help-content">
                        <div class="evas-voice-help-header">
                            <h3>${this.i18n.helpTitle}</h3>
                            <button type="button" id="evas-voice-help-close" aria-label="Close">&times;</button>
                        </div>
                        <div class="evas-voice-help-body">
            `;

            for (const [cat, cmds] of Object.entries(categories)) {
                html += `<div class="evas-voice-help-category">
                    <h4>${cat.charAt(0).toUpperCase() + cat.slice(1)}</h4>
                    <ul>`;
                
                for (const cmd of cmds) {
                    const phrase = cmd.phrases[0];
                    html += `<li>
                        <strong>"${phrase}"</strong>
                        <span>${cmd.description}</span>
                    </li>`;
                }
                
                html += '</ul></div>';
            }

            html += '</div></div></div>';

            $('body').append(html);
            setTimeout(() => $('#evas-voice-help-modal').fadeIn(200), 10);
        },

        /**
         * Hide help modal
         */
        hideHelp() {
            $('#evas-voice-help-modal').fadeOut(200);
        },

        /**
         * Play sound effect
         * @param {string} type Sound type
         */
        playSound(type) {
            // Simple beep using Web Audio API
            try {
                const ctx = new (window.AudioContext || window.webkitAudioContext)();
                const oscillator = ctx.createOscillator();
                const gain = ctx.createGain();

                oscillator.connect(gain);
                gain.connect(ctx.destination);

                oscillator.type = 'sine';
                oscillator.frequency.value = type === 'start' ? 440 : type === 'success' ? 880 : 330;
                
                gain.gain.setValueAtTime(0.1, ctx.currentTime);
                gain.gain.exponentialRampToValueAtTime(0.01, ctx.currentTime + 0.2);

                oscillator.start(ctx.currentTime);
                oscillator.stop(ctx.currentTime + 0.2);
            } catch (e) {
                // Audio not available
            }
        }
    };

    // Initialize when DOM is ready
    $(document).ready(() => EVAS_Voice.init());

    // Expose for external access
    window.EVAS_Voice = EVAS_Voice;

})(jQuery);

