<?php
/**
 * Virtual Keyboard Module Main Class
 *
 * @package     Everyone_Accessibility_Suite
 * @subpackage  Modules/Virtual_Keyboard
 * @version     1.0.0
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Class EVAS_Virtual_Keyboard_Module
 */
class EVAS_Virtual_Keyboard_Module extends EVAS_Module_Base {

    public function __construct( $settings_manager = null, $settings = array() ) {
        $this->module_id      = 'virtual_keyboard';
        $this->module_name    = __( 'Virtual Keyboard', 'everyone-accessibility-suite' );
        $this->module_version = defined( 'EVAS_VK_VERSION' ) ? EVAS_VK_VERSION : '1.0.0';
        
        parent::__construct( $settings_manager, $settings );
    }

    public function get_module_settings() {
        return [
            'enabled'        => (bool) get_option( 'evas_vk_enabled', true ),
            'default_layout' => get_option( 'evas_vk_default_layout', 'qwerty' ),
            'theme'          => get_option( 'evas_vk_theme', 'default' ),
            'key_size'       => get_option( 'evas_vk_key_size', 'medium' ),
            'show_on_focus'  => (bool) get_option( 'evas_vk_show_on_focus', true ),
            'sound_enabled'  => (bool) get_option( 'evas_vk_sound_enabled', true ),
        ];
    }

    public function register_settings() {}

    public function init() {
        add_action( 'rest_api_init', [ $this, 'register_rest_routes' ] );
        add_action( 'wp_enqueue_scripts', [ $this, 'enqueue_frontend_assets' ] );
        add_action( 'evas_panel_controls', [ $this, 'render_keyboard_control' ], 40 );
        add_action( 'wp_footer', [ $this, 'render_keyboard_container' ] );
    }

    public function register_rest_routes(): void {
        $controller = new EVAS_VK_REST_Controller();
        $controller->register_routes();
    }

    public function enqueue_frontend_assets(): void {
        if ( ! $this->is_enabled() ) {
            return;
        }

        wp_enqueue_script(
            'evas-virtual-keyboard',
            EVAS_VK_URL . 'assets/js/virtual-keyboard.js',
            [ 'jquery' ],
            EVAS_VERSION, // Use main plugin version for cache busting
            true
        );

        wp_enqueue_style(
            'evas-virtual-keyboard',
            EVAS_VK_URL . 'assets/css/virtual-keyboard.css',
            [],
            EVAS_VK_VERSION
        );

        wp_localize_script( 'evas-virtual-keyboard', 'evasKeyboard', [
            'layouts'     => EVAS_Keyboard_Layouts::get_all_layouts(),
            'specialKeys' => EVAS_Keyboard_Layouts::get_special_keys(),
            'settings'    => $this->get_frontend_settings(),
            'i18n'        => [
                'virtualKeyboard' => __( 'Virtual Keyboard', 'everyone-accessibility-suite' ),
                'close'           => __( 'Close', 'everyone-accessibility-suite' ),
            ],
        ] );
    }

    public function render_keyboard_control(): void {
        if ( ! $this->is_enabled() ) {
            return;
        }
        ?>
        <div id="evas-action-virtual-keyboard"
             class="evas-action-box evas-toggle-box"
             data-feature="virtual_keyboard"
             tabindex="0"
             role="button"
             aria-pressed="false"
             aria-label="<?php esc_attr_e( 'Virtual Keyboard', 'everyone-accessibility-suite' ); ?>">
            <span class="evas-action-icon"></span>
            <span class="evas-action-label"><?php esc_html_e( 'Virtual Keyboard', 'everyone-accessibility-suite' ); ?></span>
        </div>
        <?php
    }

    public function render_keyboard_container(): void {
        if ( ! $this->is_enabled() ) {
            return;
        }
        ?>
        <div id="evas-keyboard-container" class="evas-keyboard-container" style="display: none;" role="application" aria-label="<?php esc_attr_e( 'Virtual Keyboard', 'everyone-accessibility-suite' ); ?>">
            <div class="evas-keyboard-header">
                <span class="evas-keyboard-title"><?php esc_html_e( 'Virtual Keyboard', 'everyone-accessibility-suite' ); ?></span>
                <select id="evas-keyboard-layout" class="evas-keyboard-layout-select">
                    <?php foreach ( EVAS_Keyboard_Layouts::get_available_layouts() as $id => $name ) : ?>
                        <option value="<?php echo esc_attr( $id ); ?>"><?php echo esc_html( $name ); ?></option>
                    <?php endforeach; ?>
                </select>
                <button type="button" id="evas-keyboard-close" class="evas-keyboard-close" aria-label="<?php esc_attr_e( 'Close', 'everyone-accessibility-suite' ); ?>">&times;</button>
            </div>
            <div id="evas-keyboard-keys" class="evas-keyboard-keys"></div>
        </div>
        <?php
    }

    private function get_frontend_settings(): array {
        return [
            'defaultLayout' => get_option( 'evas_vk_default_layout', 'qwerty' ),
            'theme'         => get_option( 'evas_vk_theme', 'default' ),
            'keySize'       => get_option( 'evas_vk_key_size', 'medium' ),
            'showOnFocus'   => (bool) get_option( 'evas_vk_show_on_focus', true ),
            'soundEnabled'  => (bool) get_option( 'evas_vk_sound_enabled', true ),
        ];
    }
}

