<?php
/**
 * Virtual Keyboard REST Controller
 *
 * @package     Everyone_Accessibility_Suite
 * @subpackage  Modules/Virtual_Keyboard
 * @version     1.0.0
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Class EVAS_VK_REST_Controller
 */
class EVAS_VK_REST_Controller extends WP_REST_Controller {

    protected $namespace = 'evas-keyboard/v1';

    public function register_routes(): void {
        register_rest_route( $this->namespace, '/layouts', [
            'methods'             => WP_REST_Server::READABLE,
            'callback'            => [ $this, 'get_layouts' ],
            'permission_callback' => '__return_true',
        ] );

        register_rest_route( $this->namespace, '/settings', [
            [
                'methods'             => WP_REST_Server::READABLE,
                'callback'            => [ $this, 'get_settings' ],
                'permission_callback' => [ $this, 'permissions_check' ],
            ],
            [
                'methods'             => WP_REST_Server::EDITABLE,
                'callback'            => [ $this, 'update_settings' ],
                'permission_callback' => [ $this, 'permissions_check' ],
            ],
        ] );
    }

    public function get_layouts( WP_REST_Request $request ): WP_REST_Response {
        return rest_ensure_response( [
            'success'     => true,
            'layouts'     => EVAS_Keyboard_Layouts::get_all_layouts(),
            'specialKeys' => EVAS_Keyboard_Layouts::get_special_keys(),
            'themes'      => EVAS_Keyboard_Layouts::get_themes(),
        ] );
    }

    public function get_settings( WP_REST_Request $request ): WP_REST_Response {
        return rest_ensure_response( [
            'success'  => true,
            'settings' => [
                'enabled'        => (bool) get_option( 'evas_vk_enabled', true ),
                'default_layout' => get_option( 'evas_vk_default_layout', 'qwerty' ),
                'theme'          => get_option( 'evas_vk_theme', 'default' ),
                'key_size'       => get_option( 'evas_vk_key_size', 'medium' ),
                'show_on_focus'  => (bool) get_option( 'evas_vk_show_on_focus', true ),
                'sound_enabled'  => (bool) get_option( 'evas_vk_sound_enabled', true ),
            ],
        ] );
    }

    public function update_settings( WP_REST_Request $request ): WP_REST_Response {
        $params = $request->get_json_params();

        // Whitelist & sanitize to prevent storing unsafe/unexpected values.
        $allowed = [
            'enabled'        => 'rest_sanitize_boolean',
            'show_on_focus'  => 'rest_sanitize_boolean',
            'sound_enabled'  => 'rest_sanitize_boolean',
            'default_layout' => 'sanitize_key',
            'theme'          => 'sanitize_key',
            'key_size'       => 'sanitize_key',
        ];

        // Optional strict whitelists for enumerations.
        $allowed_key_sizes = [ 'small', 'medium', 'large' ];

        foreach ( (array) $params as $key => $value ) {
            if ( ! isset( $allowed[ $key ] ) ) {
                continue;
            }

            $sanitize = $allowed[ $key ];
            $value    = $sanitize( $value );

            if ( 'key_size' === $key && ! in_array( $value, $allowed_key_sizes, true ) ) {
                $value = 'medium';
            }

            update_option( 'evas_vk_' . $key, $value );
        }

        return rest_ensure_response( [
            'success' => true,
            'message' => __( 'Settings saved', 'everyone-accessibility-suite' ),
        ] );
    }

    public function permissions_check(): bool {
        return current_user_can( 'manage_options' );
    }
}

