<?php
/**
 * Keyboard Layouts
 *
 * Contains keyboard layouts for different languages.
 *
 * @package     Everyone_Accessibility_Suite
 * @subpackage  Modules/Virtual_Keyboard
 * @version     1.0.0
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Class EVAS_Keyboard_Layouts
 *
 * Provides keyboard layout configurations.
 */
class EVAS_Keyboard_Layouts {

    /**
     * Get available layouts
     *
     * @return array
     */
    public static function get_available_layouts(): array {
        return [
            'qwerty'   => __( 'QWERTY (English)', 'everyone-accessibility-suite' ),
            'qwertz'   => __( 'QWERTZ (German)', 'everyone-accessibility-suite' ),
            'azerty'   => __( 'AZERTY (French)', 'everyone-accessibility-suite' ),
            'russian'  => __( 'Russian', 'everyone-accessibility-suite' ),
            'spanish'  => __( 'Spanish', 'everyone-accessibility-suite' ),
            'arabic'   => __( 'Arabic', 'everyone-accessibility-suite' ),
            'hebrew'   => __( 'Hebrew', 'everyone-accessibility-suite' ),
            'numeric'  => __( 'Numeric', 'everyone-accessibility-suite' ),
        ];
    }

    /**
     * Get layout configuration
     *
     * @param string $layout_id Layout identifier.
     * @return array|null
     */
    public static function get_layout( string $layout_id ): ?array {
        $layouts = [
            'qwerty' => [
                'id'   => 'qwerty',
                'name' => 'QWERTY',
                'rtl'  => false,
                'rows' => [
                    [ '1', '2', '3', '4', '5', '6', '7', '8', '9', '0', '-', '=' ],
                    [ 'q', 'w', 'e', 'r', 't', 'y', 'u', 'i', 'o', 'p', '[', ']' ],
                    [ 'a', 's', 'd', 'f', 'g', 'h', 'j', 'k', 'l', ';', "'", '{enter}' ],
                    [ '{shift}', 'z', 'x', 'c', 'v', 'b', 'n', 'm', ',', '.', '/', '{shift}' ],
                    [ '{space}' ],
                ],
                'shift' => [
                    [ '!', '@', '#', '$', '%', '^', '&', '*', '(', ')', '_', '+' ],
                    [ 'Q', 'W', 'E', 'R', 'T', 'Y', 'U', 'I', 'O', 'P', '{', '}' ],
                    [ 'A', 'S', 'D', 'F', 'G', 'H', 'J', 'K', 'L', ':', '"', '{enter}' ],
                    [ '{shift}', 'Z', 'X', 'C', 'V', 'B', 'N', 'M', '<', '>', '?', '{shift}' ],
                    [ '{space}' ],
                ],
            ],

            'qwertz' => [
                'id'   => 'qwertz',
                'name' => 'QWERTZ',
                'rtl'  => false,
                'rows' => [
                    [ '1', '2', '3', '4', '5', '6', '7', '8', '9', '0', 'ß', '´' ],
                    [ 'q', 'w', 'e', 'r', 't', 'z', 'u', 'i', 'o', 'p', 'ü', '+' ],
                    [ 'a', 's', 'd', 'f', 'g', 'h', 'j', 'k', 'l', 'ö', 'ä', '{enter}' ],
                    [ '{shift}', 'y', 'x', 'c', 'v', 'b', 'n', 'm', ',', '.', '-', '{shift}' ],
                    [ '{space}' ],
                ],
                'shift' => [
                    [ '!', '"', '§', '$', '%', '&', '/', '(', ')', '=', '?', '`' ],
                    [ 'Q', 'W', 'E', 'R', 'T', 'Z', 'U', 'I', 'O', 'P', 'Ü', '*' ],
                    [ 'A', 'S', 'D', 'F', 'G', 'H', 'J', 'K', 'L', 'Ö', 'Ä', '{enter}' ],
                    [ '{shift}', 'Y', 'X', 'C', 'V', 'B', 'N', 'M', ';', ':', '_', '{shift}' ],
                    [ '{space}' ],
                ],
            ],

            'azerty' => [
                'id'   => 'azerty',
                'name' => 'AZERTY',
                'rtl'  => false,
                'rows' => [
                    [ '&', 'é', '"', "'", '(', '-', 'è', '_', 'ç', 'à', ')', '=' ],
                    [ 'a', 'z', 'e', 'r', 't', 'y', 'u', 'i', 'o', 'p', '^', '$' ],
                    [ 'q', 's', 'd', 'f', 'g', 'h', 'j', 'k', 'l', 'm', 'ù', '{enter}' ],
                    [ '{shift}', 'w', 'x', 'c', 'v', 'b', 'n', ',', ';', ':', '!', '{shift}' ],
                    [ '{space}' ],
                ],
                'shift' => [
                    [ '1', '2', '3', '4', '5', '6', '7', '8', '9', '0', '°', '+' ],
                    [ 'A', 'Z', 'E', 'R', 'T', 'Y', 'U', 'I', 'O', 'P', '¨', '£' ],
                    [ 'Q', 'S', 'D', 'F', 'G', 'H', 'J', 'K', 'L', 'M', '%', '{enter}' ],
                    [ '{shift}', 'W', 'X', 'C', 'V', 'B', 'N', '?', '.', '/', '§', '{shift}' ],
                    [ '{space}' ],
                ],
            ],

            'russian' => [
                'id'   => 'russian',
                'name' => 'Russian',
                'rtl'  => false,
                'rows' => [
                    [ '1', '2', '3', '4', '5', '6', '7', '8', '9', '0', '-', '=' ],
                    [ 'й', 'ц', 'у', 'к', 'е', 'н', 'г', 'ш', 'щ', 'з', 'х', 'ъ' ],
                    [ 'ф', 'ы', 'в', 'а', 'п', 'р', 'о', 'л', 'д', 'ж', 'э', '{enter}' ],
                    [ '{shift}', 'я', 'ч', 'с', 'м', 'и', 'т', 'ь', 'б', 'ю', '.', '{shift}' ],
                    [ '{space}' ],
                ],
                'shift' => [
                    [ '!', '"', '№', ';', '%', ':', '?', '*', '(', ')', '_', '+' ],
                    [ 'Й', 'Ц', 'У', 'К', 'Е', 'Н', 'Г', 'Ш', 'Щ', 'З', 'Х', 'Ъ' ],
                    [ 'Ф', 'Ы', 'В', 'А', 'П', 'Р', 'О', 'Л', 'Д', 'Ж', 'Э', '{enter}' ],
                    [ '{shift}', 'Я', 'Ч', 'С', 'М', 'И', 'Т', 'Ь', 'Б', 'Ю', ',', '{shift}' ],
                    [ '{space}' ],
                ],
            ],

            'spanish' => [
                'id'   => 'spanish',
                'name' => 'Spanish',
                'rtl'  => false,
                'rows' => [
                    [ '1', '2', '3', '4', '5', '6', '7', '8', '9', '0', "'", '¡' ],
                    [ 'q', 'w', 'e', 'r', 't', 'y', 'u', 'i', 'o', 'p', '`', '+' ],
                    [ 'a', 's', 'd', 'f', 'g', 'h', 'j', 'k', 'l', 'ñ', '´', '{enter}' ],
                    [ '{shift}', 'z', 'x', 'c', 'v', 'b', 'n', 'm', ',', '.', '-', '{shift}' ],
                    [ '{space}' ],
                ],
                'shift' => [
                    [ '!', '"', '·', '$', '%', '&', '/', '(', ')', '=', '?', '¿' ],
                    [ 'Q', 'W', 'E', 'R', 'T', 'Y', 'U', 'I', 'O', 'P', '^', '*' ],
                    [ 'A', 'S', 'D', 'F', 'G', 'H', 'J', 'K', 'L', 'Ñ', '¨', '{enter}' ],
                    [ '{shift}', 'Z', 'X', 'C', 'V', 'B', 'N', 'M', ';', ':', '_', '{shift}' ],
                    [ '{space}' ],
                ],
            ],

            'numeric' => [
                'id'   => 'numeric',
                'name' => 'Numeric',
                'rtl'  => false,
                'rows' => [
                    [ '7', '8', '9' ],
                    [ '4', '5', '6' ],
                    [ '1', '2', '3' ],
                    [ '0', '.', '{backspace}' ],
                ],
            ],
        ];

        return $layouts[ $layout_id ] ?? null;
    }

    /**
     * Get all layouts
     *
     * @return array
     */
    public static function get_all_layouts(): array {
        $layouts = [];
        
        foreach ( array_keys( self::get_available_layouts() ) as $layout_id ) {
            $layouts[ $layout_id ] = self::get_layout( $layout_id );
        }

        return $layouts;
    }

    /**
     * Get special key definitions
     *
     * @return array
     */
    public static function get_special_keys(): array {
        return [
            '{shift}'     => [ 'label' => '⇧', 'action' => 'shift' ],
            '{enter}'     => [ 'label' => '↵', 'action' => 'enter' ],
            '{backspace}' => [ 'label' => '⌫', 'action' => 'backspace' ],
            '{tab}'       => [ 'label' => '⇥', 'action' => 'tab' ],
            '{space}'     => [ 'label' => ' ', 'action' => 'space', 'class' => 'evas-key-space' ],
            '{caps}'      => [ 'label' => '⇪', 'action' => 'capslock' ],
            '{close}'     => [ 'label' => '✕', 'action' => 'close' ],
        ];
    }

    /**
     * Get themes
     *
     * @return array
     */
    public static function get_themes(): array {
        return [
            'default' => __( 'Default', 'everyone-accessibility-suite' ),
            'dark'    => __( 'Dark', 'everyone-accessibility-suite' ),
            'light'   => __( 'Light', 'everyone-accessibility-suite' ),
            'minimal' => __( 'Minimal', 'everyone-accessibility-suite' ),
            'modern'  => __( 'Modern', 'everyone-accessibility-suite' ),
        ];
    }
}

